/* otfview.c -- View glyphs of OpenType fonts.

Copyright (C) 2003, 2004
  National Institute of Advanced Industrial Science and Technology (AIST)
  Registration Number H15PRO167

This file is part of libotf.

Libotf is free software; you can redistribute it and/or modify it
under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation; either version 2.1 of the License, or
(at your option) any later version.

Libotf is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library, in a file named COPYING; if not,
write to the Free Software Foundation, Inc., 59 Temple Place, Suite
330, Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <libgen.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Viewport.h>

#include <ft2build.h>
#include FT_FREETYPE_H

#include <otf.h>

#define DEFAULT_PIXEL_SIZE 30
#define DEFAULT_FONT_NAME "6x13"
XFontStruct *font;
#define FONT_HEIGHT (font->ascent + font->descent)
#define FONT_ASCENT (font->ascent)
#define FONT_DESCENT (font->descent)
#define FONT_WIDTH (font->max_bounds.width)

XtAppContext context;
/* Widget structure.
   +--- frame (form) -------------------------+
   | +--- command_area (box) ---------------+ |
   | | quit dump charmap ...                | |
   | +--------------------------------------+ |
   | +---- navi_area (box) -----------------+ |
   | | FIRST PREV prev label next NEXT LAST | |
   | +--------------------------------------+ |
   | +--- glyph_area (form) ----------------+ |
   | | glyph[0]        ...        glyph[15] | |
   | |   ...                        ...     | |
   | | glyph[112]      ...        glyph[127]| |
   | +--------------------------------------+ |
   | +--- render_area (form) ---------------+ |
   | | clear                                | |
   | | +--- raw (box) --------------------+ | |
   | | | raw_label raw_image              | | |
   | | +--- seq (box) --------------------+ | |
   | | | seq_label seq_image              | | |
   | | +--- gsub (box) -------------------+ | |
   | | | gsub_label gsub_image            | | |
   | | +--- gpos (box) -------------------+ | |
   | | | gpos_label gpos_image            | | |
   | | +----------------------------------+ | |
   | +--------------------------------------+ |
   +------------------------------------------+ */
Widget shell, frame;
Widget command_area, quit, dump, *charmap;
Widget navi_area, FIRST, PREV, prev, label, next, NEXT, LAST;
Widget glyph_area, glyph[128];
Widget render_area, clear, raw, seq, gsub, gpos;
Widget raw_label, raw_image, seq_label, seq_image;
Widget gsub_label, gsub_image, gpos_label, gpos_image;

int glyph_char[128];

Display *display;
GC gc, gc_set, gc_or, gc_inv;

typedef struct {
  Pixmap pixmap;
  unsigned width, height;
  int x, y;
  int advance;
} BitmapRec;

BitmapRec bitmap[0x10000];

int render_width, render_height;
Pixmap raw_pixmap, seq_pixmap, gsub_pixmap, gpos_pixmap;
Pixmap none_pixmap;

FT_Face face;

struct {
  int platform_id;
  int encoding_id;
  char name[20];
} charmap_rec[10];

int charmap_index;

unsigned glyph_width, glyph_height;
int glyph_x, glyph_y;
int glyph_index;

struct {
  int n_glyphs;
  int glyphs[64];
  int codes[64];
} glyph_rec;

OTF *otf;
char *filename;

void
create_pixmap (int pixel_size, int index)
{
  int err = FT_Load_Glyph (face, index, FT_LOAD_RENDER | FT_LOAD_MONOCHROME);
  XImage ximage;
  Pixmap pixmap;
  int height = glyph_height + 1 + FONT_HEIGHT;
  char index_buf[5];
  
  if (err)
    {
      bitmap[index].pixmap = (Pixmap) 0;
      return;
    }
  ximage.height = face->glyph->bitmap.rows;
  ximage.width = face->glyph->bitmap.width;
  ximage.depth = 1;
  ximage.bits_per_pixel = 1;
  ximage.xoffset = 0;
  ximage.format = XYPixmap;
  ximage.data = (char *) face->glyph->bitmap.buffer;
  ximage.byte_order = MSBFirst;
  ximage.bitmap_unit = 8;
  ximage.bitmap_bit_order = MSBFirst;
  ximage.bitmap_pad = 8;
  ximage.bytes_per_line = face->glyph->bitmap.pitch;
  XInitImage (&ximage);
  pixmap = XCreatePixmap (display, DefaultRootWindow (display),
			  glyph_width, height, 1);
  XFillRectangle (display, pixmap, gc, 0, 0, glyph_width, height);
  XPutImage (display, pixmap, gc, &ximage, 0, 0,
	     glyph_x + face->glyph->bitmap_left,
	     glyph_y - face->glyph->bitmap_top,
	     ximage.width, ximage.height);
  sprintf (index_buf, "%04X", index);
  XDrawLine (display, pixmap, gc_inv,
	     0, glyph_height + 1, glyph_width, glyph_height + 1);
  XDrawString (display, pixmap, gc_inv,
	       (glyph_width - XTextWidth (font, index_buf, 4)) / 2,
	       height - FONT_DESCENT, index_buf, 4);
  bitmap[index].pixmap = pixmap;
  bitmap[index].width = ximage.width;
  bitmap[index].height = ximage.height;
  bitmap[index].x = face->glyph->bitmap_left;
  bitmap[index].y = - face->glyph->bitmap_top;
  bitmap[index].advance = face->glyph->metrics.horiAdvance >> 6;
}

void
update_glyph_area ()
{
  int i;
  Arg arg[2];
  char buf[16];

  for (i = 0; i < 128; i++)
    {
      int index = glyph_index + i;

      if (charmap_index >= 0)
	index = FT_Get_Char_Index (face, (FT_ULong) index);
      if (bitmap[index].pixmap)
	XtSetArg (arg[0], XtNbitmap, bitmap[index].pixmap);
      else
	XtSetArg (arg[0], XtNbitmap, none_pixmap);
      XtSetValues (glyph[i], arg, 1);
    }

  sprintf (buf, " %04X-%04X ", glyph_index, glyph_index + 0x7F);
  XtSetArg (arg[0], XtNlabel, buf);
  XtSetValues (label, arg, 1);
}

void
update_render_area ()
{
  int i;
  int x;
  Arg arg[1];

  XFillRectangle (display, raw_pixmap, gc, 0, 0, render_width, render_height);
  XFillRectangle (display, seq_pixmap, gc, 0, 0, render_width, render_height);
  for (i = 0, x = glyph_x; i < glyph_rec.n_glyphs; i++)
    {
      BitmapRec *bmp = bitmap + glyph_rec.glyphs[i];
      char buf[5];

      XCopyArea (display, bmp->pixmap, raw_pixmap, gc,
		 0, 0, glyph_width, glyph_height,
		 (glyph_width + 1) * i + 1, 1);
      XDrawRectangle (display, raw_pixmap, gc_set,
		      (glyph_width + 1) * i, 0,
		      glyph_width + 1, glyph_height + 1);
      XDrawLine (display, raw_pixmap, gc_set,
		 (glyph_width + 1) * i + 1 + glyph_x, 1,
		 (glyph_width + 1) * i + 1 + glyph_x, glyph_height + 1);
      XDrawLine (display, raw_pixmap, gc_set,
		 (glyph_width + 1) * i + 1 + glyph_x + bmp->advance, 1,
		 (glyph_width + 1) * i + 1 + glyph_x + bmp->advance,
		 glyph_height + 1);

      sprintf (buf, "%04X", glyph_rec.codes[i]);
      XDrawString (display, raw_pixmap, gc_inv, 
		   (glyph_width + 1) * i + 1
		   + (glyph_width - XTextWidth (font, buf, 4)) / 2,
		   glyph_height + 2 + FONT_HEIGHT, buf, 4);
      XCopyArea (display, bmp->pixmap, seq_pixmap, gc_or,
		 glyph_x + bmp->x, glyph_y + bmp->y, bmp->width, bmp->height,
		 x + bmp->x, glyph_y + bmp->y);
      x += bmp->advance;
    }
  XtSetArg (arg[0], XtNbitmap, raw_pixmap);
  XtSetValues (raw_image, arg, 1);
  XtSetArg (arg[0], XtNbitmap, seq_pixmap);
  XtSetValues (seq_image, arg, 1);
  if (! otf)
    return;
  XFillRectangle (display, gsub_pixmap, gc, 0, 0, render_width, render_height);
  XFillRectangle (display, gpos_pixmap, gc, 0, 0, render_width, render_height);
  XtSetArg (arg[0], XtNbitmap, gsub_pixmap);
  XtSetValues (gsub_image, arg, 1);
  XtSetArg (arg[0], XtNbitmap, gpos_pixmap);
  XtSetValues (gpos_image, arg, 1);
}

void
QuitProc (Widget w, XtPointer client_data, XtPointer call_data)
{
  XtAppSetExitFlag (XtWidgetToApplicationContext (w));
}

void
DumpProc (Widget w, XtPointer client_data, XtPointer pixel_size)
{
  int g_width, g_height, g_x, g_y, pix_width, pix_height;
  int margin = 20 * 300 / 25.4;
  int a4_width = 210 * 300 / 25.4 - margin * 2;
  int a4_height = 297 * 300 / 25.4 - margin * 2;
  int size = 100;
  Pixmap pixmap;
  XImage ximage, *image;
  int i, x, y;
  char *data;
  int bytes_per_line;

  FT_Set_Pixel_Sizes (face, 0, size);
  g_width = ((face->bbox.xMax - face->bbox.xMin) * size / face->units_per_EM);
  g_height = ((face->bbox.yMax - face->bbox.yMin) * size / face->units_per_EM);
  pix_width = (g_width + 1) * 16 + margin + 1;
  pix_height = (g_height + 1 + FONT_HEIGHT) * 16 + margin + 1;
  while (pix_width > a4_width || pix_height > a4_height)
    {
      size--;
      FT_Set_Pixel_Sizes (face, 0, size);
      g_width = ((face->bbox.xMax - face->bbox.xMin)
		 * size / face->units_per_EM);
      g_height = ((face->bbox.yMax - face->bbox.yMin)
		  * size / face->units_per_EM);
      pix_width = (g_width + 1) * 16 + margin + 1;
      pix_height = (g_height + 1 + FONT_HEIGHT) * 16 + margin + 1;
    }

  g_x = - (face->bbox.xMin * size / face->units_per_EM);
  g_y = face->bbox.yMax * size / face->units_per_EM;
  for (i = 0; i < 0xFF; i++)
    {
      int idx;

      if (charmap_index >= 0)
	idx = FT_Get_Char_Index (face, (FT_ULong) i);
      else
	idx = i;
      if (FT_Load_Glyph (face, idx, FT_LOAD_RENDER | FT_LOAD_MONOCHROME) == 0)
	{
	  if (g_x < - face->glyph->bitmap_left)
	    g_x = - face->glyph->bitmap_left;
	  if (g_y < face->glyph->bitmap_top)
	    g_y = face->glyph->bitmap_top;
	  if (g_width
	      < g_x + face->glyph->bitmap_left + face->glyph->bitmap.width)
	    g_width
	      = g_x + face->glyph->bitmap_left + face->glyph->bitmap.width;
	  if (g_height
	      < g_y - face->glyph->bitmap_top + face->glyph->bitmap.rows)
	    g_height
	      = g_y - face->glyph->bitmap_top + face->glyph->bitmap.rows;
	}
    }
  pix_width = (g_width + 1) * 16 + margin + 1;
  pix_height = (g_height + FONT_HEIGHT + 1) * 16 + margin + 1;
  pixmap = XCreatePixmap (display,
			  RootWindow (display, DefaultScreen (display)),
			  pix_width, pix_height, 1);
  XFillRectangle (display, pixmap, gc, 0, 0, pix_width, pix_height);

  for (i = 0, x = margin; i <= 16; i++, x += g_width + 1)
    XDrawLine (display, pixmap, gc_set, x, margin,
	       x, margin + (g_height + FONT_HEIGHT + 1) * 16);
  for (i = 0, y = margin; i <= 16; i++, y += g_height + FONT_HEIGHT + 1)
    XDrawLine (display, pixmap, gc_set, margin, y,
	       margin + (g_width + 1) * 16, y);
  for (i = 0; i < 256; i++)
    {
      char str[5];
      int idx;

      if (charmap_index >= 0)
	idx = FT_Get_Char_Index (face, (FT_ULong) i);
      else
	idx = i;
      x = margin + (g_width + 1) * (i % 16);
      y = margin + (g_height + FONT_HEIGHT + 1) * (i / 16);
      if (FT_Load_Glyph (face, idx, FT_LOAD_RENDER | FT_LOAD_MONOCHROME) == 0)
	{
	  ximage.height = face->glyph->bitmap.rows;
	  ximage.width = face->glyph->bitmap.width;
	  ximage.depth = 1;
	  ximage.bits_per_pixel = 1;
	  ximage.xoffset = 0;
	  ximage.format = XYPixmap;
	  ximage.data = (char *) face->glyph->bitmap.buffer;
	  ximage.byte_order = MSBFirst;
	  ximage.bitmap_unit = 8;
	  ximage.bitmap_bit_order = MSBFirst;
	  ximage.bitmap_pad = 8;
	  ximage.bytes_per_line = face->glyph->bitmap.pitch;
	  XInitImage (&ximage);
	  XPutImage (display, pixmap, gc, &ximage, 0, 0,
		     x + g_x + face->glyph->bitmap_left,
		     y + g_y - face->glyph->bitmap_top, 
		     ximage.width, ximage.height);
	}
      sprintf (str, "0x%02X", i);
      XDrawString (display, pixmap, gc_inv,
		   x + (g_width - XTextWidth (font, str, 4))/ 2,
		   y + g_height + FONT_ASCENT, str, 4);
    }

  image = XGetImage (display, pixmap, 0, 0, pix_width, pix_height,
		     AllPlanes, XYPixmap);
  XInitImage (image);
  {
    char *name = alloca (strlen (filename) + 5);
    FILE *fp;

    sprintf (name, "%s.pbm", filename);
    printf ("Writing %s ...", name);
    fp = fopen (name, "w");
    fprintf (fp, "P4\n%d %d\n", image->width, image->height);
    bytes_per_line = (image->width + 7) / 8;
    data = image->data;
    for (y = 0; y < image->height; y++, data += image->bytes_per_line)
      fwrite (data, 1, bytes_per_line, fp);
    fclose (fp);
    printf ("done\n");
  }
  FT_Set_Pixel_Sizes (face, 0, (int) pixel_size);
}


void
GlyphProc (Widget w, XtPointer client_data, XtPointer call_data)
{
  int old_glyph_index = glyph_index;

  if ((int) client_data == -3 && glyph_index > 0)
    glyph_index = 0;
  else if ((int) client_data == -2 && glyph_index > 0)
    glyph_index = (glyph_index - 1) & 0xF000;
  else if ((int) client_data == -1 && glyph_index > 0)
    glyph_index -= 0x80;
  else if ((int) client_data == 1 && glyph_index < 0xFF80)
    glyph_index += 0x80;
  else if ((int) client_data == 2 && glyph_index < 0xF000)
    glyph_index = (glyph_index + 0x1000) & 0xF000;
  else if ((int) client_data == 3 && glyph_index < 0xF000)
    glyph_index = 0xFF80;
  if (glyph_index != old_glyph_index)
    update_glyph_area ();
}

void
CharmapProc (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (charmap_index == (int) client_data)
    return;
  charmap_index = (int) client_data;
  if (charmap_index >= 0)
    FT_Set_Charmap (face, face->charmaps[charmap_index]);
  update_glyph_area ();
}

void
RenderProc (Widget w, XtPointer client_data, XtPointer call_data)
{
  if ((int) client_data < 0)
    {
      glyph_rec.n_glyphs = 0;
      update_render_area ();
    }
  else if (glyph_rec.n_glyphs < 64)
    {
      int index = glyph_index + (int) client_data;

      if (charmap_index >= 0)
	index = FT_Get_Char_Index (face, (FT_ULong) index);
      if (bitmap[index].pixmap)
	{
	  glyph_rec.codes[glyph_rec.n_glyphs] = glyph_index + (int) client_data;
	  glyph_rec.glyphs[glyph_rec.n_glyphs++] = index;
	  update_render_area ();
	}
    }
}

void
create_widgets (int pixel_size)
{
  String quit_action = "<KeyPress>q: set() notify() unset()";
  String FIRST_action = "~Shift<KeyPress>f: set() notify() unset()";
  String PREV_action = "Shift<KeyPress>p: set() notify() unset()";
  String prev_action = "~Shift<KeyPress>p: set() notify() unset()";
  String next_action = "~Shift<KeyPress>n: set() notify() unset()";
  String NEXT_action = "Shift<KeyPress>n: set() notify() unset()";
  String LAST_action = "~Shift<KeyPress>l: set() notify() unset()";
  Arg arg[10];
  int i, j;
  int glyph_widget_height;

  frame = XtCreateManagedWidget ("frame", formWidgetClass, shell, NULL, 0);
  XtSetArg (arg[0], XtNleft, XawChainLeft);
  XtSetArg (arg[1], XtNright, XawChainLeft);
  XtSetArg (arg[2], XtNtop, XawChainTop);
  XtSetArg (arg[3], XtNbottom, XawChainTop);
  XtSetArg (arg[4], XtNborderWidth, 0);
  XtSetArg (arg[5], XtNorientation, XtorientHorizontal);
  command_area = XtCreateManagedWidget ("command-area", boxWidgetClass,
					frame, arg, 6);
  XtSetArg (arg[6], XtNfromVert, command_area);
  navi_area = XtCreateManagedWidget ("navi-area", boxWidgetClass,
				     frame, arg, 7);
  XtSetArg (arg[4], XtNborderWidth, 1);
  XtSetArg (arg[5], XtNfromVert, navi_area);
  XtSetArg (arg[6], XtNdefaultDistance, 0);
  glyph_area = XtCreateManagedWidget ("glyph-area", formWidgetClass,
				      frame, arg, 7);
  XtSetArg (arg[4], XtNborderWidth, 0);
  XtSetArg (arg[5], XtNfromVert, glyph_area);
  render_area = XtCreateManagedWidget ("render-area", formWidgetClass,
				       frame, arg, 6);

  XtSetArg (arg[0], XtNaccelerators, XtParseAcceleratorTable (quit_action));
  quit = XtCreateManagedWidget ("Quit", commandWidgetClass,
				command_area, arg, 1);
  XtAddCallback (quit, XtNcallback, QuitProc, NULL);

  dump = XtCreateManagedWidget ("Dump Image", commandWidgetClass,
				command_area, arg, 1);
  XtAddCallback (dump, XtNcallback, DumpProc, (XtPointer) pixel_size);

  charmap = alloca (sizeof (Widget) * (face->num_charmaps + 1));
  XtSetArg (arg[0], XtNstate, True);
  charmap[0] = XtCreateManagedWidget (charmap_rec[0].name, toggleWidgetClass,
				      command_area, arg, 1);
  XtAddCallback (charmap[0], XtNcallback, CharmapProc, (XtPointer) -1);
  XtSetArg (arg[0], XtNradioGroup, charmap[0]);
  for (i = 0; i < face->num_charmaps; i++)
    {
      charmap[i + 1] = XtCreateManagedWidget (charmap_rec[i + 1].name,
					      toggleWidgetClass,
					      command_area, arg, 1);
      XtAddCallback (charmap[i + 1], XtNcallback, CharmapProc, (XtPointer) i);
    }

  XtSetArg (arg[0], XtNlabel, " |< (f)");
  XtSetArg (arg[1], XtNaccelerators, XtParseAcceleratorTable (FIRST_action));
  FIRST = XtCreateManagedWidget ("FIRST", commandWidgetClass,
				 navi_area, arg, 2);
  XtAddCallback (FIRST, XtNcallback, GlyphProc, (XtPointer) -3);
  XtSetArg (arg[0], XtNlabel, "<< (P)");
  XtSetArg (arg[1], XtNaccelerators, XtParseAcceleratorTable (PREV_action));
  PREV = XtCreateManagedWidget ("PREV", commandWidgetClass,
				navi_area, arg, 2);
  XtAddCallback (PREV, XtNcallback, GlyphProc, (XtPointer) -2);
  XtSetArg (arg[0], XtNlabel, "< (p)");
  XtSetArg (arg[1], XtNaccelerators, XtParseAcceleratorTable (prev_action));
  prev = XtCreateManagedWidget ("prev", commandWidgetClass,
				navi_area, arg, 2);
  XtAddCallback (prev, XtNcallback, GlyphProc, (XtPointer) -1);
  XtSetArg (arg[0], XtNlabel, " 0000 ");
  label = XtCreateManagedWidget ("label", labelWidgetClass,
				 navi_area, arg, 1);
  XtSetArg (arg[0], XtNlabel, "> (n)");
  XtSetArg (arg[1], XtNaccelerators, XtParseAcceleratorTable (next_action));
  next = XtCreateManagedWidget ("next", commandWidgetClass,
				navi_area, arg, 2);
  XtAddCallback (next, XtNcallback, GlyphProc, (XtPointer) 1);
  XtSetArg (arg[0], XtNlabel, ">> (N)");
  XtSetArg (arg[1], XtNaccelerators, XtParseAcceleratorTable (NEXT_action));
  NEXT = XtCreateManagedWidget ("NEXT", commandWidgetClass,
				navi_area, arg, 2);
  XtAddCallback (NEXT, XtNcallback, GlyphProc, (XtPointer) 2);
  XtSetArg (arg[0], XtNlabel, ">| (l)");
  XtSetArg (arg[1], XtNaccelerators, XtParseAcceleratorTable (LAST_action));
  LAST = XtCreateManagedWidget ("LAST", commandWidgetClass,
				navi_area, arg, 2);
  XtAddCallback (LAST, XtNcallback, GlyphProc, (XtPointer) 3);

  glyph_widget_height = glyph_height + 1 + FONT_HEIGHT;
  XtSetArg (arg[0], XtNleft, XawChainLeft);
  XtSetArg (arg[1], XtNright, XawChainLeft);
  XtSetArg (arg[2], XtNtop, XawChainTop);
  XtSetArg (arg[3], XtNbottom, XawChainTop);
  for (i = 0; i < 8; i++)
    for (j = 0; j < 16; j++)
      {
	int k = i * 16 + j;
	int num_args = 4;

	XtSetArg (arg[num_args], XtNwidth, glyph_width), num_args++;
	XtSetArg (arg[num_args], XtNheight, glyph_widget_height), num_args++;
	if (j > 0)
	  XtSetArg (arg[num_args], XtNfromHoriz, glyph[k - 1]), num_args++;
	if (i > 0)
	  XtSetArg (arg[num_args], XtNfromVert, glyph[k - 16]), num_args++;
	XtSetArg (arg[num_args], XtNinternalWidth, 0), num_args++;
	glyph[k] = XtCreateManagedWidget ("glyph", commandWidgetClass,
					  glyph_area, arg, num_args);
	XtAddCallback (glyph[k], XtNcallback, RenderProc, (XtPointer) k);
      }

  XtSetArg (arg[0], XtNleft, XawChainLeft);
  XtSetArg (arg[1], XtNright, XawChainLeft);
  XtSetArg (arg[2], XtNtop, XawChainTop);
  XtSetArg (arg[3], XtNbottom, XawChainTop);
  clear = XtCreateManagedWidget ("clear", commandWidgetClass,
				 render_area, arg, 4);
  XtAddCallback (clear, XtNcallback, RenderProc, (XtPointer) -1);
  XtSetArg (arg[4], XtNorientation, XtorientHorizontal);
  XtSetArg (arg[5], XtNborderWidth, 0);
  XtSetArg (arg[6], XtNfromVert, clear);
  raw = XtCreateManagedWidget ("raw", boxWidgetClass,
				render_area, arg, 7);
  XtSetArg (arg[0], XtNborderWidth, 0);
  XtSetArg (arg[1], XtNlabel, "raw: ");
  raw_label = XtCreateManagedWidget ("raw-label", labelWidgetClass,
				      raw, arg, 2);
  XtSetArg (arg[1], XtNbitmap, raw_pixmap);
  raw_image = XtCreateManagedWidget ("raw-image", labelWidgetClass,
				      raw, arg, 2);
  XtSetArg (arg[6], XtNfromVert, raw);
  seq = XtCreateManagedWidget ("seq", boxWidgetClass,
				render_area, arg, 7);
  XtSetArg (arg[0], XtNborderWidth, 0);
  XtSetArg (arg[1], XtNlabel, "seq: ");
  seq_label = XtCreateManagedWidget ("seq-label", labelWidgetClass,
				      seq, arg, 2);
  XtSetArg (arg[1], XtNbitmap, seq_pixmap);
  seq_image = XtCreateManagedWidget ("seq-image", labelWidgetClass,
				      seq, arg, 2);
  if (otf)
    {
      XtSetArg (arg[6], XtNfromVert, seq);
      gsub = XtCreateManagedWidget ("gsub", boxWidgetClass,
				    render_area, arg, 7);
      XtSetArg (arg[0], XtNborderWidth, 0);
      XtSetArg (arg[1], XtNlabel, "gsub: ");
      gsub_label = XtCreateManagedWidget ("gsub-label", labelWidgetClass,
					  gsub, arg, 2);
      gsub_image = XtCreateManagedWidget ("gsub-image", labelWidgetClass,
					  gsub, arg, 1);
      XtSetArg (arg[6], XtNfromVert, gsub);
      gpos = XtCreateManagedWidget ("gpos", boxWidgetClass,
				    render_area, arg, 7);
      XtSetArg (arg[0], XtNborderWidth, 0);
      XtSetArg (arg[1], XtNlabel, "gpos: ");
      gpos_label = XtCreateManagedWidget ("gpos-label", labelWidgetClass,
					  gpos, arg, 2);
      gpos_image = XtCreateManagedWidget ("gpos-image", labelWidgetClass,
					  gpos, arg, 1);
    }

  XtInstallAllAccelerators (shell, shell);
}


/* Format MSG by FMT and print the result to the stderr, and exit.  */

#define FATAL_ERROR(fmt, arg)	\
  do {				\
    fprintf (stderr, fmt, arg);	\
    exit (1);			\
  } while (0)

int
main (int argc, char **argv)
{
  FT_Library library;

  OTF_GlyphString gstring;
  OTF_Glyph *g;

  int err;
  int i;
  int pixel_size = DEFAULT_PIXEL_SIZE;
  int fixed_pixel_size = 0;
  int display_width;

  {
    char *str = getenv ("PIXEL_SIZE");

    if (str && (i = atoi (str)) > 0)
      {
	pixel_size = i;
	fixed_pixel_size = 1;
      }
  }

  gstring.size = gstring.used = 256;
  g = calloc (256, sizeof (OTF_Glyph));
  gstring.glyphs = g;

  shell = XtOpenApplication (&context, "OTFView", NULL, 0, &argc, argv, NULL,
			     shellWidgetClass, NULL, 0);
  display = XtDisplay (shell);
  display_width = DisplayWidth (display,
				XScreenNumberOfScreen (XtScreen (shell)));
  font = XLoadQueryFont (display, DEFAULT_FONT_NAME);
  if (! font)
    font = XLoadQueryFont (display, "fixed");

  if (argc != 2 || !strcmp (argv[1], "-h") || !strcmp (argv[1], "--help"))
    {
      fprintf (stderr, "Usage: %s [ X-OPTION ... ]  OTF-FILE\n",
	       basename (argv[0]));
      exit (argc != 2);
    }
  filename = argv[1];
  if (strstr (filename, ".ttf")
      || strstr (filename, ".TTF")
      || strstr (filename, ".otf")
      || strstr (filename, ".OTF"))
    {
      otf = OTF_open (filename);
      if (! otf
	  || OTF_get_table (otf, "head") < 0
	  || OTF_get_table (otf, "cmap") < 0
	  || (OTF_get_table (otf, "gsub") < 0
	      && OTF_get_table (otf, "gpos") < 0))
	otf = NULL;
    }

  if ((err = FT_Init_FreeType (&library)))
    FATAL_ERROR ("%s\n", "FT_Init_FreeType: error");
  err = FT_New_Face (library, filename, 0, &face);
  if (err == FT_Err_Unknown_File_Format)
    FATAL_ERROR ("%s\n", "FT_New_Face: unknown file format");
  else if (err)
    FATAL_ERROR ("%s\n", "FT_New_Face: unknown error");
  if ((err = FT_Set_Pixel_Sizes (face, 0, pixel_size)))
    FATAL_ERROR ("%s\n", "FT_Set_Pixel_Sizes: error");

  {
    char title[256];
    Arg arg[1];

    filename = basename (filename);
    sprintf (title, "%s family:%s style:%s",
	     filename, face->family_name, face->style_name);
    XtSetArg (arg[0], XtNtitle, title);
    XtSetValues (shell, arg, 1);
  }

  glyph_width = ((face->bbox.xMax - face->bbox.xMin)
		 * pixel_size / face->units_per_EM);
  if (! fixed_pixel_size && glyph_width * 16 > display_width * 0.8)
    {
      pixel_size = (pixel_size * display_width * 0.8 / 16 / glyph_width);
      FT_Set_Pixel_Sizes (face, 0, pixel_size);
      glyph_width = ((face->bbox.xMax - face->bbox.xMin)
		     * pixel_size / face->units_per_EM);
    }
  if (glyph_width < FONT_WIDTH * 4)
    glyph_width = FONT_WIDTH * 4;

  glyph_height = ((face->bbox.yMax - face->bbox.yMin)
		  *  pixel_size / face->units_per_EM);

  glyph_x = - (face->bbox.xMin * pixel_size / face->units_per_EM);
  glyph_y = face->bbox.yMax * pixel_size / face->units_per_EM;

  for (i = 0; i < 0x10000; i++)
    if (FT_Load_Glyph (face, i, FT_LOAD_RENDER | FT_LOAD_MONOCHROME) == 0)
      {
	if (glyph_x < - face->glyph->bitmap_left)
	  glyph_x = - face->glyph->bitmap_left;
	if (glyph_y < face->glyph->bitmap_top)
	  glyph_y = face->glyph->bitmap_top;
	if (glyph_width
	    < glyph_x + face->glyph->bitmap_left + face->glyph->bitmap.width)
	  glyph_width
	    = glyph_x + face->glyph->bitmap_left + face->glyph->bitmap.width;
	if (glyph_height
	    < glyph_y - face->glyph->bitmap_top + face->glyph->bitmap.rows)
	  glyph_height
	    = glyph_y - face->glyph->bitmap_top + face->glyph->bitmap.rows;
      }

  none_pixmap = XCreatePixmap (display, DefaultRootWindow (display),
			       glyph_width, glyph_height + 1 + FONT_HEIGHT, 1);

  {
    unsigned long valuemask =  GCFunction | GCLineWidth;
    XGCValues values;

    gc = XCreateGC (display, none_pixmap, (unsigned long) 0, NULL);
    values.function = GXset;
    values.line_width = 1;
    gc_set = XCreateGC (display, none_pixmap, valuemask, &values);
    values.function = GXor;
    gc_or = XCreateGC (display, none_pixmap, valuemask, &values);
    values.function = GXcopyInverted;
    gc_inv = XCreateGC (display, none_pixmap, valuemask, &values);
  }

  XFillRectangle (display, none_pixmap, gc, 0, 0,
		  glyph_width, glyph_height + 1 + FONT_HEIGHT);
  XDrawString (display, none_pixmap, gc_inv,
	       (glyph_width - XTextWidth (font, "none", 4)) / 2,
	       glyph_height / 2, "none", 4);

  render_width = (glyph_width + 1) * 15 + 1;
  render_height = glyph_height + FONT_HEIGHT + 2;

  charmap_rec[0].platform_id = -1;
  charmap_rec[0].encoding_id = -1;
  strcpy (charmap_rec[0].name, "no charmap");

  for (i = 0; i < face->num_charmaps; i++)
    {
      charmap_rec[i + 1].platform_id = face->charmaps[i]->platform_id;
      charmap_rec[i + 1].encoding_id = face->charmaps[i]->encoding_id;
      sprintf (charmap_rec[i + 1].name, "%d-%d",
	       charmap_rec[i + 1].platform_id, charmap_rec[i + 1].encoding_id);
      if (face->charmaps[i]->platform_id == 0
	  || (face->charmaps[i]->platform_id == 3
	      && face->charmaps[i]->encoding_id == 1))
	strcat (charmap_rec[i + 1].name, " (unicode)");
      else if (face->charmaps[i]->platform_id == 1
	       && face->charmaps[i]->encoding_id == 0)
	strcat (charmap_rec[i + 1].name, " (apple-roman)");
    }

  raw_pixmap = XCreatePixmap (display, DefaultRootWindow (display),
			      render_width, render_height, 1);
  seq_pixmap = XCreatePixmap (display, DefaultRootWindow (display),
			      render_width, render_height, 1);
  gsub_pixmap = XCreatePixmap (display, DefaultRootWindow (display),
			       render_width, render_height, 1);
  gpos_pixmap = XCreatePixmap (display, DefaultRootWindow (display),
			       render_width, render_height, 1);

  for (i = 0; i < 0x10000; i++)
    create_pixmap (pixel_size, i);
  create_widgets (pixel_size);
  glyph_index = 0;
  charmap_index = -1;
  update_glyph_area ();
  update_render_area ();

  XtRealizeWidget (shell);
  XtAppMainLoop (context);

  exit (0);
}
