/*
 * Copyright (c) 2001-2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#ifndef __PFMLIB_H__
#define __PFMLIB_H__

#ifdef __cplusplus
extern "C" {
#endif
#include <stdio.h>
#include <inttypes.h>

#include <perfmon/pfmlib_os.h>
#include <perfmon/pfmlib_comp.h>

#define PFMLIB_VERSION		(3 << 16 | 2)
#define PFMLIB_MAJ_VERSION(v)	((v)>>16)
#define PFMLIB_MIN_VERSION(v)	((v) & 0xffff)

/*
 * Maximum number of PMCs/PMDs supported by the library (especially bitmasks)
 */
#define PFMLIB_MAX_PMCS		256		/* maximum number of PMCS supported by the library */
#define PFMLIB_MAX_PMDS		256		/* maximum number of PMDS supported by the library */

/*
 * privilege level mask (mask can be combined)
 */
#define PFM_PLM0	0x1			/* kernel level (most privileged) */
#define PFM_PLM1	0x2			/* priv level 1 */
#define PFM_PLM2	0x4			/* priv level 2 */
#define PFM_PLM3	0x8			/* user level (least privileged) */

/*
 * type used to describe a set of bits in the mask (container type)
 */
typedef unsigned long pfmlib_regmask_bits_t;

/*
 * how many elements do we need to represent all the PMCs and PMDs (rounded up)
 */
#if PFMLIB_MAX_PMCS > PFMLIB_MAX_PMDS
#define PFMLIB_REG_MAX	PFMLIB_MAX_PMCS
#else
#define PFMLIB_REG_MAX	PFMLIB_MAX_PMDS
#endif

#define __PFMLIB_REG_BV_BITS (sizeof(pfmlib_regmask_bits_t)<<3)
#define PFMLIB_BVSIZE(x) (((x)+(__PFMLIB_REG_BV_BITS)-1) / __PFMLIB_REG_BV_BITS)
#define PFMLIB_REG_BV PFMLIB_BVSIZE(__PFMLIB_REG_BV_BITS)

typedef struct {
	pfmlib_regmask_bits_t bits[PFMLIB_REG_BV];
} pfmlib_regmask_t;


#define PFMLIB_MAX_MASKS_PER_EVENT 32

/*
 * event description for pfmlib_input_param_t
 */
typedef struct {
	unsigned int	event;		/* event descriptor */
	unsigned int	plm;		/* event privilege level mask */
	unsigned long	flags;		/* per-event flag */
	unsigned int	unit_masks[PFMLIB_MAX_MASKS_PER_EVENT]; /* unit-mask identifiers */
	unsigned int	num_masks;	/* number of masks specified in 'unit_masks' */
	unsigned long	reserved[2];	/* for future use */
} pfmlib_event_t;

/*
 * type used to describe the value of a PMD counter
 * (not to be confused with the value of a PMC)
 */
typedef uint64_t pfmlib_counter_t;

/*
 * generic PMD type
 */
typedef pfmlib_counter_t pfmlib_pmd_t;

/*
 * generic register definition.
 */
typedef struct {
	unsigned int 		reg_num;	/* register index */
	unsigned int 		reg_evt_idx;	/* event idx in pfp_events */
	unsigned long long	reg_value;	/* register value */
	unsigned int		reg_pmd_num;	/* which counter is used */
	unsigned int		reg_reserved1;	/* for future use */
	unsigned long		reg_reserved[1];/* for future use */
} pfmlib_reg_t;
#define PFMLIB_REG_PMD_NONE	(~0)	/* PMC is not associated with counter */
#define PFMLIB_EVENT_NONE	(~0)	/* PMC is not associated with counter */

/*
 * generic PMC register definition.
 */
typedef pfmlib_reg_t	pfmlib_pmc_t;

/*
 * library generic input parameters for pfm_dispatch_event()
 */
typedef struct {
	unsigned int	 pfp_event_count;	 	/* how many events specified (input) */
	unsigned int	 pfp_dfl_plm;		 	/* default priv level : used when event.plm==0 */
	unsigned int     pfp_flags;		 	/* set of flags for all events used when event.flags==0*/
	unsigned int	 reserved1;			/* for future use */
	pfmlib_event_t	 pfp_events[PFMLIB_MAX_PMCS];	/* event descriptions */
	pfmlib_regmask_t pfp_unavail_pmcs;		/* bitmask of unavailable PMC registers */
	unsigned long	 reserved[6];			/* for future use */
} pfmlib_input_param_t;

/*
 * possible values for the flags (pfp_flags) in generic input parameters (apply to all events)
 */
#define PFMLIB_PFP_SYSTEMWIDE		0x1	/* indicate monitors will be used in a system-wide session */

/*
 * library generic output parameters from pfm_dispatch_event()
 */
typedef struct {
	unsigned int	pfp_pmc_count;		 	/* how many PMCS were setup in pfp_pmc[] */
	unsigned int	reserved1;			/* for future use */
	pfmlib_pmc_t	pfp_pmcs[PFMLIB_MAX_PMCS];	/* PMC registers number and values */
	unsigned long	reserved[7];			/* for future use */
} pfmlib_output_param_t;


/*
 * library configuration options
 */
typedef struct {
	unsigned int	pfm_debug:1;	/* set in debug  mode */
	unsigned int	pfm_verbose:1;	/* set in verbose mode */
	unsigned int	pfm_reserved:30;/* for future use */
} pfmlib_options_t;

extern int pfm_set_options(pfmlib_options_t *opt);
extern int pfm_initialize(void);

extern int pfm_list_supported_pmus(int (*pf)(const char *fmt,...));
extern int pfm_get_pmu_name(char *name, int maxlen);
extern int pfm_get_pmu_type(int *type);
extern int pfm_get_pmu_name_bytype(int type, char *name, size_t maxlen);
extern int pfm_is_pmu_supported(int type);
extern int pfm_force_pmu(int type);

extern int pfm_find_event(const char *str, unsigned int *idx);
extern int pfm_find_event_byname(const char *name, unsigned int *idx);
extern int pfm_find_event_bycode(int code, unsigned int *idx);
extern int pfm_find_event_bycode_next(int code, unsigned int start, unsigned int *next);
extern int pfm_find_event_mask(unsigned int event_idx, const char *str, unsigned int *mask_idx);
extern int pfm_get_max_event_name_len(size_t *len);
extern int pfm_get_max_name_len(size_t *len);

extern int pfm_get_num_events(unsigned int *count);
extern int pfm_get_num_event_masks(unsigned int event_idx, unsigned int *count);
extern int pfm_get_event_name(unsigned int idx, char *name, size_t maxlen);
extern int pfm_get_event_code(unsigned int idx, int *code);
extern int pfm_get_event_counters(unsigned int idx, pfmlib_regmask_t *counters);
extern int pfm_get_event_description(unsigned int idx, char **str);
extern int pfm_get_event_code_counter(unsigned int idx, unsigned int cnt, int *code);
extern int pfm_get_event_mask_name(unsigned int event_idx, unsigned int mask_idx, char *name, size_t maxlen);
extern int pfm_get_event_mask_description(unsigned int event_idx, unsigned int mask_idx, char **desc);

extern int pfm_print_event_info(const char *name, int (*pf)(const char *fmt,...));
extern int pfm_print_event_info_byindex(unsigned int idx, int (*pf)(const char *fmt,...));

extern int pfm_dispatch_events(pfmlib_input_param_t *p, void *model_in, pfmlib_output_param_t *q, void *model_out);

extern int pfm_get_impl_pmcs(pfmlib_regmask_t *impl_pmcs);
extern int pfm_get_impl_pmds(pfmlib_regmask_t *impl_pmds);
extern int pfm_get_impl_counters(pfmlib_regmask_t *impl_counters);
extern int pfm_get_num_pmds(unsigned int *num);
extern int pfm_get_num_pmcs(unsigned int *num);
extern int pfm_get_num_counters(unsigned int *num);

extern int pfm_get_hw_counter_width(unsigned int *width);
extern int pfm_get_version(unsigned int *version);
extern char *pfm_strerror(int code);
extern int pfm_get_cycle_event(unsigned int *ev);
extern int pfm_get_inst_retired_event(unsigned int *ev);

/*
 * Supported PMU family (must fit in 64 bits)
 */
#define PFMLIB_NO_PMU	 	 	0	/* unsupported */
#define PFMLIB_GEN_IA64_PMU	 	1	/* IA-64 architected PMU */
#define PFMLIB_ITANIUM_PMU	 	2	/* Itanium   */
#define PFMLIB_ITANIUM2_PMU 	 	3	/* Itanium 2 */
#define PFMLIB_MONTECITO_PMU 	 	4	/* dual-core Itanium 2 */
#define PFMLIB_AMD_X86_64_PMU		16	/* AMD X86_64 */
#define PFMLIB_GEN_IA32_PMU		63	/* IA-32 architected PMU */
#define PFMLIB_I386_P6_PMU		32	/* P6/Pentium M */

#define PFMLIB_GEN_MIPS64_PMU           64      /* 5KC,20KC,25KF */

/*
 * pfmlib error codes
 */
#define PFMLIB_SUCCESS		  0	/* success */
#define PFMLIB_ERR_NOTSUPP	 -1	/* function not supported */
#define PFMLIB_ERR_INVAL	 -2	/* invalid parameters */
#define PFMLIB_ERR_NOINIT	 -3	/* library was not initialized */
#define PFMLIB_ERR_NOTFOUND	 -4	/* object not found */
#define PFMLIB_ERR_NOASSIGN	 -5	/* cannot assign events to counters */
#define PFMLIB_ERR_FULL	 	 -6	/* buffer is full (obsolete) */
#define PFMLIB_ERR_EVTMANY	 -7	/* event used more than once */
#define PFMLIB_ERR_MAGIC	 -8	/* invalid library magic number */
#define PFMLIB_ERR_FEATCOMB	 -9	/* invalid combination of features */
#define PFMLIB_ERR_EVTSET	-10	/* incompatible event sets */
#define PFMLIB_ERR_EVTINCOMP	-11	/* incompatible event combination */
#define PFMLIB_ERR_TOOMANY	-12	/* too many events */

#define PFMLIB_ERR_IRRTOOBIG	-13	/* code range too big */
#define PFMLIB_ERR_IRREMPTY	-14	/* empty code range */
#define PFMLIB_ERR_IRRINVAL	-15	/* invalid code range */
#define PFMLIB_ERR_IRRTOOMANY	-16	/* too many code ranges */
#define PFMLIB_ERR_DRRINVAL	-17	/* invalid data range */
#define PFMLIB_ERR_DRRTOOMANY	-18	/* too many data ranges */
#define PFMLIB_ERR_BADHOST	-19	/* not supported by host CPU */
#define PFMLIB_ERR_IRRALIGN	-20	/* bad alignment for code range */
#define PFMLIB_ERR_IRRFLAGS	-21	/* code range missing flags */

#define __PFMLIB_REGMASK_EL(g)		((g)/__PFMLIB_REG_BV_BITS)
#define __PFMLIB_REGMASK_MASK(g)	(((pfmlib_regmask_bits_t)1) << ((g) % __PFMLIB_REG_BV_BITS))

static inline int
pfm_regmask_isset(pfmlib_regmask_t *h, unsigned int b)
{
	if (b >= PFMLIB_REG_MAX)
		return 0;
	return (h->bits[__PFMLIB_REGMASK_EL(b)] & __PFMLIB_REGMASK_MASK(b)) != 0;
}

static inline int
pfm_regmask_set(pfmlib_regmask_t *h, unsigned int b)
{
	if (b >= PFMLIB_REG_MAX)
		return PFMLIB_ERR_INVAL;

	h->bits[__PFMLIB_REGMASK_EL(b)] |=  __PFMLIB_REGMASK_MASK(b);

	return PFMLIB_SUCCESS;
}

static inline int
pfm_regmask_clr(pfmlib_regmask_t *h, unsigned int b)
{
	if (h == NULL || b >= PFMLIB_REG_MAX)
		return PFMLIB_ERR_INVAL;

	h->bits[__PFMLIB_REGMASK_EL(b)] &= ~ __PFMLIB_REGMASK_MASK(b);

	return PFMLIB_SUCCESS;
}

static inline int
pfm_regmask_weight(pfmlib_regmask_t *h, unsigned int *w)
{
	unsigned int pos;
	unsigned int weight = 0;

	if (h == NULL || w == NULL)
		return PFMLIB_ERR_INVAL;

	for (pos = 0; pos < PFMLIB_REG_BV; pos++) {
		weight += (unsigned int)pfmlib_popcnt(h->bits[pos]);
	}
	*w = weight;
	return PFMLIB_SUCCESS;
}

static inline int
pfm_regmask_eq(pfmlib_regmask_t *h1, pfmlib_regmask_t *h2)
{
	unsigned int pos;

	if (h1 == NULL || h2 == NULL)
		return 0;

	for (pos = 0; pos < PFMLIB_REG_BV; pos++) {
		if (h1->bits[pos] != h2->bits[pos]) return 0;
	}
	return 1;
}

static inline int
pfm_regmask_and(pfmlib_regmask_t *dst, pfmlib_regmask_t *h1, pfmlib_regmask_t *h2)
{
	unsigned int pos;
	if (dst == NULL || h1 == NULL || h2 == NULL)
		return PFMLIB_ERR_INVAL;

	for (pos = 0; pos < PFMLIB_REG_BV; pos++) {
		dst->bits[pos] = h1->bits[pos] & h2->bits[pos];
	}
	return PFMLIB_SUCCESS;
}

static inline int
pfm_regmask_or(pfmlib_regmask_t *dst, pfmlib_regmask_t *h1, pfmlib_regmask_t *h2)
{
	unsigned int pos;
	if (dst == NULL || h1 == NULL || h2 == NULL)
		return PFMLIB_ERR_INVAL;

	for (pos = 0; pos < PFMLIB_REG_BV; pos++) {
		dst->bits[pos] = h1->bits[pos] | h2->bits[pos];
	}
	return PFMLIB_SUCCESS;
}

static inline int
pfm_regmask_copy(pfmlib_regmask_t *dst, pfmlib_regmask_t *src)
{
	unsigned int pos;
	if (dst == NULL || src == NULL)
		return PFMLIB_ERR_INVAL;

	for (pos = 0; pos < PFMLIB_REG_BV; pos++) {
		dst->bits[pos] = src->bits[pos];
	}
	return PFMLIB_SUCCESS;
}

#ifdef __cplusplus /* extern C */
}
#endif

#endif /* __PFMLIB_H__ */
