#include <phidgets/quadservo.h>
#include <stdio.h>

#include <math.h>

#include <termios.h>
#include <term.h>
#include <curses.h>
#include <unistd.h> 

static struct termios initial_settings, new_settings;
static char peek_character = -1;

void init_keyboard();
void close_keyboard();
int kbhit();
int readch();
void fail(int r);
void fail_cleanup(PhidgetQuadServo* qsc, int r);

int main (int argc, char** argv)
{
  if (argc != 3)
  {
    fprintf(stderr, "usage: %s [serial] [servo (0-3)]\n", argv[0]);
    exit(-1);
  }

  init_keyboard();

  phidget_return ret = phidget_init();
  PhidgetQuadServo* qsc = phidget_new_PhidgetQuadServo();
  ret = phidget_quadservo_open(qsc, atoi(argv[1]), 3);
  if (ret != PHIDGET_RET_SUCCESS)
  {
    fprintf(stderr, "failed to open servo with serial %d.\n", atoi(argv[1]));
    fail_cleanup(qsc, 1);
  }

  PhidgetQuadServoSelector servo;
  switch (atoi(argv[2]))
  {
    case 0: servo = PHIDGET_QUADSERVO_MOTOR0; break;
    case 1: servo = PHIDGET_QUADSERVO_MOTOR1; break;
    case 2: servo = PHIDGET_QUADSERVO_MOTOR2; break;
    case 3: servo = PHIDGET_QUADSERVO_MOTOR3; break;
    default:
      fprintf(stderr, "servo %d out of range ([0-3]).\n", atoi(argv[2]));
      fail_cleanup(qsc, 2);
  }

  int low = 1000;
  int high = 2000;
  
  phidget_quadservo_set_servo_parameters(qsc, servo, low, high, 10.6);

  printf("press q/a or w/s to raise/lower the pulse limit on either side.\n");
  printf("initial: low = %d; high = %d (to quit, press 'z')\n", low, high);

  double i;
  bool running = true;
  while (running)
  {
    for (i = 0.0; running && i < 2 * M_PI; i += 0.01)
    {
      unsigned int pos = 0x7fffffff - cos(i) * 0x7fffffff;
      phidget_quadservo_set_single_position(qsc, servo, pos);
      if (kbhit()) {
        switch (readch()) {
          case 'a': low += 10; break;
          case 'A': low += 1; break;
          case 'q': low -= 10; break;
          case 'Q': low -= 1; break;
          case 's': high -= 10; break;
          case 'S': high -= 1; break;
          case 'w': high += 10; break;
          case 'W': high += 1; break;
          case 'z': running = false; break;
        }
        if (running)
        {
          printf("new: low = %d; high = %d (to quit, press 'z')\n", low, high);
          phidget_quadservo_set_servo_parameters(qsc, servo, low, high, 10.6);
        }
      }
    }
  }
  ret = phidget_quadservo_close(qsc);
  phidget_delete_PhidgetQuadServo(&qsc);
  ret = phidget_cleanup();

  close_keyboard();

  printf("settings: low = %d; high = %d\n", low, high);

  return 0;
}

void init_keyboard()
{
  tcgetattr(0,&initial_settings);
  new_settings = initial_settings;
  new_settings.c_lflag &= ~ICANON;
  new_settings.c_lflag &= ~ECHO;
  new_settings.c_lflag &= ~ISIG;
  new_settings.c_cc[VMIN] = 1;
  new_settings.c_cc[VTIME] = 0;
  tcsetattr(0, TCSANOW, &new_settings);
}

void close_keyboard()
{
  tcsetattr(0, TCSANOW, &initial_settings);
}

int kbhit()
{
  char ch;
  int nread;

  if(peek_character != -1)
    return 1;
  new_settings.c_cc[VMIN]=0;
  tcsetattr(0, TCSANOW, &new_settings);
  nread = read(0,&ch,1);
  new_settings.c_cc[VMIN]=1;
  tcsetattr(0, TCSANOW, &new_settings);

  if(nread == 1) {
    peek_character = ch;
    return 1;
  }
  return 0;
}

int readch()
{
  char ch;

  if(peek_character != -1) {
    ch = peek_character;
    peek_character = -1;
    return ch;
  }
  read(0,&ch,1);
  return ch;
}

void fail(int r)
{
  close_keyboard();
  exit(r);
}

void fail_cleanup(PhidgetQuadServo* qsc, int r)
{
  phidget_quadservo_close(qsc);
  phidget_delete_PhidgetQuadServo(&qsc);
  phidget_cleanup();
  fail(r);
}

/* COPYRIGHT --
 *
 * This file is part of libphidgets, a user-space library for phidgets.
 * libphidgets is (c) 2003-2005 Martin F. Krafft <krafft@ailab.ch>
 * and distributed under the terms of the Artistic Licence.
 * See the ./COPYING file in the source tree root for more information.
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES
 * OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
