////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////
#ifndef S11N_NS_MAP_H_INCLUDED
#define S11N_NS_MAP_H_INCLUDED 1

#include <list>
#include <iterator> // insert_iterator
#include <algorithm> // for_each()
#include <stdio.h> // snprintf()

#include <S11N_NS/functor.h> // copy_if, same_name

#include <S11N_NS/data_node_serialize.h> // core serialize funcs
#include <S11N_NS/data_node_functor.h> // data_node_child_serializer functor
#include <S11N_NS/data_node_algo.h> // dump_node_debug()
#include <S11N_NS/abstract_creator.h> // abstract_creator class

namespace S11N_NS {
/**
   The S11N_NS::map namespace defines functors and algorithms for
   working with std::map and std::pair containers.
*/
namespace map {

        
        /**

           serialize_streamable_map() is intended for use as a
           serialization proxy for the s11n framework. It
           de/serializes objects of type <tt>std::map&lt;X,Y&gt;</tt>
           into a destination node.

           NodeType must support:

           set( const std::string & key, const std::string & value )

           - MapType must conform to std::map conventions and
           it's key_type and mapped_type must both be Value Types
           which are i/ostreamable (this includes all PODs and
           std::string). Pointers as keys or values are not
           supported by this functor.

           Returns the number of values serialized into dest.
        */

        template <typename NodeType, typename MapType>
        size_t serialize_streamable_map( NodeType & dest, const MapType & src )
        {
                typedef typename MapType::value_type VT;
                typedef typename VT::first_type FT;
                typedef typename VT::second_type ST;
                VT p;
                size_t count=0;
                typename MapType::const_iterator it = src.begin();
                for( ; src.end() != it; ++it )
                {
                        ++count;
                        dest.set(
                                 S11N_NS::to_string<FT>( (*it).first ),
                                 S11N_NS::to_string<ST>( (*it).second )
                                 );
                }
                return count;
        }

        /**
           Exactly like serialize_streamable_map(dest,src) except that a subnode,
           named subnodename, of dest is created to store the data. 
        */
        template <typename NodeType, typename MapType>
        size_t serialize_streamable_map( NodeType & dest,
                                         const std::string & subnodename,
                                         const MapType & src )
        {
                return serialize_streamable_map( S11N_NS::create_child( dest, subnodename ), src );
        }

        /**
           This is the converse of serialize_streamable_map(). It tries to
           read in all properties stored in src and stick them into
           dest.

           NodeType must support begin() and end() and they must
           return iterators to pair&lt;X,Y&gt;, where X and Y must
           meet the same requirements as the key and value types for
           MapType in serialize_streamable_map(). MapType must support:

           void insert( MapType::value_type );

           (Duh.)

           Returns the number of values deserialized into dest.
        */
        template <typename NodeType, typename MapType>
        size_t deserialize_streamable_map( const NodeType & src, MapType & dest )
        {
                typedef typename MapType::value_type VT; // pair
                typedef typename VT::first_type T1;
                typedef typename VT::second_type T2;

                const T1 default1 = T1();
                const T2 default2 = T2();
                size_t count=0;
                typename NodeType::const_iterator it = src.begin();
                for( ; src.end() != it; ++it )
                {
                        ++count;
                        dest.insert( std::make_pair(
                                                    S11N_NS::from_string( (*it).first, default1 ),
                                                    S11N_NS::from_string( (*it).second, default2 )
                                                     ) );
                }
                return count;
        }

        /**
           Exactly like deserialize_streamable_map(dest,src) except
           that a subnode of dest, named subnodename, is sought to
           pull the data from.

           A return value of 0 indicates that either no child with
           the given name was found or that no deserializable items
           were in that node.
        */
        template <typename NodeType, typename MapType>
        size_t deserialize_streamable_map( const NodeType & src,
                                           const std::string & subnodename,
                                           MapType & dest )
        {
                const NodeType * ch = S11N_NS::find_child_by_name( src, subnodename );
                if( ! ch ) return 0;
                return deserialize_streamable_map<NodeType,MapType>( *ch, dest );
        }


        /**
           Serializes a std::pair-compatible type into a "custom"
           format, suitable for saving pairs in standard XML
           (de/serialize_streamable_map() can't do this when keys are
           not valid XML keys, e.g., numeric). Use
           deserialize_streamable_pair() to decode the data.

           The destination node gets these two properties:

           - first = src.first

           - second = src.second

           PairType must comply with:

           - first/second types must be i/o streamable (i.e.,
           convertable to strings).

           Returns true on success... and never fails. Honestly. It'll
           fail at compile-time if it's going to fail.


           use deserialize_streamable_pair() to convert them back to pairs,
           or fish out the "first" and "second" properties manually.
        */
        template <typename NodeType, typename PairType>
        bool serialize_streamable_pair( NodeType & dest, const PairType & src )
        {
                typedef typename PairType::first_type FT;
                typedef typename PairType::second_type ST;
                dest.set( "first", src.first );
                dest.set( "second", src.second );
                return true;
        }

        /**
           The quasi-counterpart of serialize_streamable_pair(). It's
           non-conventional args and return type are a result of
           map::value_type having a const .first element, which
           prohibits us assigning to it. See deserialize_pair() for
           more info on that.
        */
        template <typename PairType, typename NodeType>
        PairType deserialize_streamable_pair( const NodeType & src  )
        {
                typedef typename PairType::first_type T1;
                typedef typename PairType::second_type T2;
                T1 default1 = T1();
                T2 default2 = T2();
                return std::make_pair( src.get( "first", default1 ),
                                       src.get( "second", default2 )
                                       );
        }


        /**
           Similar to serialize_streamable_map(), but puts each key/value
           pair into it's own node, using serialize_streamable_pair(). The
           end effect is that it's output is more verbose, but may be
           compatible with more file formats, regardless of the actual
           key type. e.g., numeric keys are supported by standard XML
           (though they are by the s11n XML parsers), and this algorithm
           structures the data such that this is not a problem.

           Returns the number of pairs stored.

           MapType must meet these conditions:

           value_type must be a pair containing i/ostreamable types
           (e.g. PODs/strings).

        */
        template <typename NodeType, typename MapType>
        size_t serialize_streamable_map_pairs( NodeType & dest, const MapType & src )
        {
                // dest.impl_class( class_name_of_MapType );
                typedef typename MapType::value_type VT;
                typedef typename VT::first_type FT;
                typedef typename VT::second_type ST;
                size_t count=0;
                typename MapType::const_iterator it = src.begin();
                for( ; src.end() != it; ++it )
                {
                        ++count;
                        serialize_streamable_pair( create_child( dest, "pair" ), *it );
                }
                return count;
        }

        /**
           The load-time counterpart to serialize_streamable_map_pairs().
        */
        template <typename NodeType, typename MapType>
        size_t deserialize_streamable_map_pairs( const NodeType & src, MapType & dest )
        {
                typedef typename MapType::value_type VT; // pair
                typedef typename VT::first_type T1;
                typedef typename VT::second_type T2;
                typedef std::list<const NodeType *> ChList;
                typedef typename ChList::const_iterator ChIt;

                static const T1 default1 = T1();
                static const T2 default2 = T2();

                ChList namedch;
                if( 0 == S11N_NS::find_children_by_name( src, "pair", namedch ) ) return 0;

                size_t count=0;
                const NodeType * ch = 0;
                ChIt it = namedch.begin();
                VT p = VT(default1,default2);
                for( ; namedch.end() != it; ++it )
                {
                        ++count;
                        ch = *it;
                        dest.insert( deserialize_streamable_pair<VT>( *ch ) );
                }
                return count;
        }


        /**
           serialize_pair() can serialize any std::pair type which
           meets these conditions:

           - PairType must be registed with s11n.
           
           PairType's first_type and second_type types must:

           - both be Serializables.  They may be pointer or value
           types.


           ACHTUNG: never pass the same destination container to the
           operators more than once or you will get duplicate and/or
           incorrect data.
        */
        template <typename NodeType, typename PairType>
        bool serialize_pair( NodeType & dest, const PairType & src )
        {
                dest.impl_class( classname<PairType>() );
                NodeType * ch = & create_child( dest, "first" );
                if( ! serialize( *ch, src.first ) )
                {
                        CERR << "serialize_pair: first child failed serialize!\n";
                        return false;
                }
                ch = & create_child( dest, "second" );
                if( ! serialize( *ch, src.second ) )
                {
                        CERR << "serialize_pair: second child failed serialize!\n";
                        return false;
                }
                return true;
        }

        /**
           The counterpart to serialize_pair().

           Note: std::map&lt;X,Y&gt;::value_type is not the same as pair&lt;X,Y&gt;,
           but is pair&lt;const X,Y&gt;, so you cannot simply iterate over a map and
           pass each pair to this function, because this function cannot assign
           to the first element of such a pair.
        */
        template <typename NodeType, typename PairType>
        bool deserialize_pair( const NodeType & src, PairType & dest )
        {
                //CERR << "deserialize_pair: deserialize " << src.impl_class() << "\n";
                typedef typename PairType::first_type FT;
                typedef typename PairType::second_type ST;
                const NodeType * ch = 0;
                typedef pointer_stripper<PairType> PS;

                // Note: the abstract_creator code below is simply
                // to treat pointer and value types identically
                // with this same code base. See it's docs for what
                // it does (or doesn't do, in the case of reference
                // types).

                typedef S11N_NS::abstract_creator<FT> AC1st;
                typedef S11N_NS::abstract_creator<ST> AC2nd;
                //////////////////////////////////////// .first
                ch = find_child_by_name( src, "first" );
                if( ! ch )
                {
                        CERR << "deserialize_pair: deserialize: no 'first' node found!\n";
                        return false;
                }
                FT f; // value of .first
                if( ! AC1st::create( f, ch->impl_class() ) )
                {
                        CERR << "Internal error: could not create first element."
                             <<"type='"<<ch->impl_class()<<"'!\n";
                        return false;
                }
                if( ! deserialize( *ch, f ) )
                {
                        CERR << "pair deserialize(..., first ) failed.\n";
                        AC1st::release( f );
                        return false;
                }
                //////////////////////////////////////// .second:
                ST s; // value of .second
                ch = find_child_by_name( src, "second" );
                if( ! ch )
                {
                        CERR << "deserialize_pair: deserialize: no 'second' node found!\n";
                        AC1st::release( f );
                        return false;
                }
                if( ! AC2nd::create( s, ch->impl_class() ) )
                {
                        CERR << "Internal error: could not create second element."
                             <<"type='"<<ch->impl_class()<<"'!\n";
                        AC1st::release( f );
                        return false;
                }
                if( ! deserialize( *ch, s ) )
                {
                        CERR << "deserialize_pair(): deserialize((..., second ) failed.\n";
                        AC1st::release( f );
                        AC2nd::release( s );
                        return false;
                }
                dest.first = f;
                dest.second = s;
                return true;
        }



        /**
           pair_serializer_proxy is a Serializable Proxy for std::pairs.
        */
        struct pair_serializer_proxy
        {
                /**
                   See serialize_pair().
                */
                template <typename NodeType, typename PairType>
                bool operator()( NodeType & dest, const PairType & src ) const
                {
                        return serialize_pair( dest, src );
                }
                /**
                   See deserialize_pair().
                */
                template <typename NodeType, typename PairType>
                bool operator()( const NodeType & src, PairType & dest ) const
                {
                        return deserialize_pair( src, dest );
                }
        };

        

        /**
           Serialize the given map into dest. MapType's pairs must be
           Serializable and must contain Serializable types, but their
           "pointerness" is irrelevant.

           ACHTUNG: never pass the same destination container to the
           operators more than once or you will get duplicate and/or
           incorrect data.

           See deserialize_map() for important info.

        */
        template <typename NodeType, typename MapType>
        bool serialize_map( NodeType & dest, const MapType & src )
        {
                typedef typename MapType::const_iterator CIT;
                CIT b = src.begin(), e = src.end();
                NodeType * ch = 0;
                for( ; e != b; ++b )
                {
                        ch = new NodeType;
                        ch->name( "pair" );
                        if( ! serialize_pair( *ch, *b ) )
                        {
                                delete( ch );
                                CERR << "map_serializer_proxy: child failed serialize.\n";
                                return false;
                        }
                        dest.children().push_back( ch );
                }
                return true;
        }

        /**
           Identical to the two-argument form, but creates a subnode of dest,
           named subnodename, and serializes to that node.
        */
        template <typename NodeType, typename MapType>
        bool serialize_map( NodeType & dest,
                            const std::string & subnodename,
                            const MapType & src )
        {
                NodeType * ch = new NodeType;
                if( ! serialize_map<NodeType,MapType>( *ch, src ) )
                {
                        delete( ch );
                        return false;
                }
                dest.children().push_back( ch );
                return true;
        }

        /**
           The counterpart of serializer_map(), deserializes src into the
           given map. MapType must be Serializable and contain pairs
           which themselves are Serializables... ad inifinitum..

           Minor caveat:

           This operation will only work with maps containing std::pair
           types, not map-like classes which use a different pair
           type. :( The reason is that map&lt;X,Y&gt;::value_type is
           not a pair of (X,Y), but (const Y,Y), which means we cannot
           use the map's value_type for a deser operation because we
           cannot assign to it's .first element (i.e., can't
           deserialize it). To get around that we "manually" create a
           new std::pair type using map's key_type and mapped_type
           typedefs, which "loses" the constness for use so we can
           assign to the first_type during deserialization, and then
           insert that pair into the deserializing map.


         */
        template <typename NodeType, typename MapType>
        bool deserialize_map( const NodeType & src, MapType & dest )
        {
                typedef typename NodeType::child_list_type::const_iterator CIT;
                //typedef typename SerializableType::value_type VT;
                // ^^^ no, because VT::first_type is const!
                // Thus we hand-create a compatible pair type:
                typedef typename MapType::key_type KType;
                typedef typename MapType::mapped_type VType;
                typedef std::pair< KType, VType > PairType;
                PairType pair;
                CIT b = src.children().begin(), e = src.children().end();
                const NodeType *ch = 0;
                for( ; e != b ; ++b )
                {
                        ch = *b;
                        if( ! deserialize_pair( *ch, pair ) )
                        {
                                CERR << "map_serializer_proxy: child failed deser.\n";
                                return false;
                        }
                        dest.insert( pair );
                }
                return true;
        }


        /**
           Identical to the two-argument form, but tries to deserialize
           from a subnode of src named subnodename. If no such node is found
           then false is returned.
        */
        template <typename NodeType, typename MapType>
        bool deserialize_map( const NodeType & src,
                              const std::string & subnodename,
                              MapType & dest )
        {
                const NodeType * ch = S11N_NS::find_child_by_name( src, subnodename );
                if( ! ch ) return false;
                return deserialize_map<NodeType,MapType>( *ch, dest );
        }


        /**
           A proxy which can serialize std::maps which contain Streamable
           Types.

           It uses de/serialize_streamable_map(), so see those
           functions for details.
        */
        struct streamable_map_serializer_proxy
        {
                /**
                   Serializes src to dest.

                   ACHTUNG: never pass the same destination container
                   to this operator more than once or you will get
                   duplicate and/or incorrect data.
                */
                template <typename NodeType, typename SerializableType>
                bool operator()( NodeType & dest , const SerializableType & src ) const
                {
                        dest.impl_class( class_name<SerializableType>::name() );
                        //S11N_NS::serialize_streamable_map_pairs( dest, src );
                        serialize_streamable_map( dest, src );
                        return true;
                }

                /**
                   Deserializes dest from src.
                */
                template <typename NodeType, typename SerializableType>
                bool operator()( const NodeType & src , SerializableType & dest ) const
                {
                        //S11N_NS::deserialize_streamable_map_pairs( src, dest );
                        deserialize_streamable_map( src, dest );
                        return true;
                }
        };



        /**
           map_serializer_proxy is a Serialization Proxy for std::maps.

           See de/serialize_map(): this functor simply wraps those.
        */
        struct map_serializer_proxy
        {

                /**
                   Serializes src into dest. Returns true on success,
                   false on error. Stops at the first child-serialize failure.
                */
                template <typename NodeType, typename MapType>
                bool operator()( NodeType & dest , const MapType & src ) const
                {
                        return serialize_map( dest, src );
                }
                /**
                   Deserializes src into dest. Returns true on success,
                   false on error. Stops at the first child-deserialize failure.
                */
                template <typename NodeType, typename MapType>
                bool operator()( const NodeType & src , MapType & dest ) const
                {
                        return deserialize_map( src, dest );
                }
        };





} } // namespace S11N_NS::map


#endif // S11N_NS_MAP_H_INCLUDED
