#ifndef PACKAGE_NAMESPACE_FILE_UTIL_H_INCLUDED
#define PACKAGE_NAMESPACE_FILE_UTIL_H_INCLUDED 1

#include <string>

namespace PACKAGE_NAMESPACE {

        /**
           CompressionPolicy describes the framework-wide policy
           regarding compression when writing to files. (Sorry, pure
           stream-based compression isn't yet supported.)

           Setting to GZip/BZipCompression only enables compression if
           HAVE_ZLIB or HAVE_BZLIB ars set during library compilation,
           otherwise non-compressed streams are used in all cases.

           bzip compresses much better than gzip, but is notably
           slower. The speed should not make much of a difference
           unless you are working with very large data sets, and then
           you will notice that gzip is faster.

           Tip: you can use get_istream() on an arbitrary filename to
           get an appropriate input stream. Likewise, use
           get_ostream() to open writable a file with the
           policy-mandated stream type (if possible, defaulting to
           std::ofstream).
         */
        enum CompressionPolicy {
	        /** NoCompression implies just what it says - do not use compressed output streams. */
	        NoCompression = 0,
                /** GZipCompression means to use the PACKAGE_NAMESPACE::ogzstream class for output streams. */
        	GZipCompression = 1,
                /** BZipCompression means to use the PACKAGE_NAMESPACE::obzstream class for output streams. */
	        BZipCompression = 2
        };

        /**
           Sets the framework-wide CompressionPolicy preference.
           See the PACKAGE_NAMESPACE::CompressionPolicy enum.
         */
        void compression_policy( CompressionPolicy c );

        /**
           Returns the framework-wide CompressionPolicy preference.
           See the PACKAGE_NAMESPACE::CompressionPolicy enum.
         */
        CompressionPolicy compression_policy();

        /**
           Looks at the given file and tries to figure out what type
           of decompression stream, if any, should be used.
           It will return one of the following types:

           - std::ifstream
           - PACKAGE_NAMESPACE::gzstream
           - PACKAGE_NAMESPACE::bzstream

           It will return NULL if it cannot open filename, or a
           std::ifstream if it cannot figure out a decompression
           scheme.

           It will only return one of the compressed stream types if
           this code is built with the appropriate macros: HAVE_ZLIB
           and HAVE_BZLIB, and then linked against -lgz and/or -lbz2.
           Thus, if this code is not built with compression support it
           is possible that it returns a std::ifstream reading from a
           compressed file.

           If AsFilename is true, input is treated as a file,
           otherwise it is treated as a string, for which an
           stringstream is returned. Compressor streams are currently
           only supported when AsFile is true. This feature has subtle
           uses in writing more generic client code for handling arbitrary
           input streams.
        */
        std::istream * get_istream( const std::string & src, bool AsFile = true );


        /**
           Returns one of the following types of ostreams, depending
           on compression_policy() and compile-time library settings
           of HAVE_ZLIB and HAVE_BZLIB.

           - PACKAGE_NAMESPACE::ogzstream
           - PACKAGE_NAMESPACE::obzstream
           - std::ofstream

           Note that this function only uses filename to pass to the
           stream's constructor, and does no checking of the file.

           The caller is responsible for deleting the pointer.

           For determining the type of input stream for a file, 
           see PACKAGE_NAMESPACE::get_istream().
        */
        std::ostream * get_ostream( const std::string & filename );



        
        /**
           Returns the first 'bytes' bytes from the given file,
           assuming the file exists and can be read. It stops at the
           first newline character unless read_past_nl is true.

           On error an empty string is returned.

           This function is primarily intended to be used for looking
           for magic cookies. It is also sometimes useful, for
           example, for checking the type of a file or grabbing, e.g.,
           the CVSROOT out of CVS/Root.

           Results are undefined with binary files.

           If this library is compiled with HAVE_ZLIB/HAVE_BZLIB set
           to true then this function supports transparent
           decompression of gzipped/bzipped files.

           It caches requests, so subsequent lookups are fast.
         */
        std::string bytes_from_file( const std::string & fn, unsigned int bytes = 128, bool read_past_nl = false );

}; // namespace PACKAGE_NAMESPACE

#endif // PACKAGE_NAMESPACE_FILE_UTIL_H_INCLUDED
