#ifndef PACKAGE_NAMESPACE_PROPERTY_SERVER_H_INCLUDED
#define PACKAGE_NAMESPACE_PROPERTY_SERVER_H_INCLUDED 1
#include <string>

#include <PACKAGE_NAMESPACE/to_string.h> // to/from_string()
#include <PACKAGE_NAMESPACE/phoenix.h> // phoenix class

namespace PACKAGE_NAMESPACE {

        /**
           property_server is an experimental class for tying
           arbitrary properties to arbitrary objects. It supports, as
           values, any data types which are i/ostreamable using their
           <code> >>istream</code> or <code>ostream<< </code>
           operators.

           License: Do As You Damned Well Please
           Author: stephan@s11n.net

        */

        template <typename ContextType, typename ObjIDType = ContextType *, typename KeyType = std::string>
        class property_server
        {
        public:
                /**
                   The "sharing context" used by this type.
                */
                typedef ContextType context_type;

                /** The type used to key objects to their properties. */
                typedef ObjIDType object_id_type;

                /** The key type used for properties. */
                typedef KeyType key_type;

                /** The underlying data store for properties. */
                typedef std::map<key_type,std::string> property_map_type;


                /**
                   Sets the property key to val for objid.
                */
                template <typename ValueType>
                static void set( const object_id_type objid, const key_type & key, const ValueType & val )
                {
                        object_map()[objid][key] = PACKAGE_NAMESPACE::to_string( val );
                }

                /**
                   Returns the property key for objid. If there is an error
                   converting the value to ValueType, or if the property does
                   not exist, defaultval is returned. This can be used to
                   implement custom error handling by passing a known-invalid
                   value as defaultval and interpretting it as an error
                   value. If there is *no* known-invalid combinations then you
                   can check twice with two different defaulvals. If
                   defaultval is returned both times then there is definately
                   an error, because valid data would only be interpretted one
                   way and you've checked against two different defaultvals.
                */
                template <typename ValueType>
                static ValueType get( const object_id_type objid, const key_type & key, const ValueType & defaultval )
                {
                        typename object_map_type::const_iterator it = object_map().find( objid );
                        if( object_map().end() == it ) return defaultval;
                        const property_map_type & pmap = it.second;
                        typename property_map_type::const_iterator pit = pmap.find( key );
                        if( pmap.end() == pit ) return defaultval;
                        return PACKAGE_NAMESPACE::from_string<ValueType>( pit.second, defaultval );
                }

                /**
                   Removes the given property from the given object.
                */
                static void unset( const object_id_type objid, const key_type & key )
                {
                        typename object_map_type::const_iterator it = object_map().find( objid );
                        if( object_map().end() == it ) return;
                        const property_map_type & pmap = it.second;
                        typename property_map_type::const_iterator pit = pmap.find( key );
                        if( pmap.end() == pit ) return;
                        pmap.erase( pit );
                }

                /**
                   Removes the property map for objid, freeing up any
                   resources it consumed.
                */
                static void clear( const object_id_type objid )
                {
                        typename object_map_type::iterator it = object_map().find( objid );
                        if( object_map().end() == it ) return;
                        object_map().erase( it );
                }

                /** Removes all properties for all objects. */
                static void clear_all()
                {
                        object_map().clear();
                }

                //      typedef void (change_listener_callback)( object_id_type, const key_type key & );
                //        void add_change_listener( change_listener_callback cb );

                /**
                   Returns a pointer to the given object's property map, or 0
                   if no properties have been set for the object (or if
                   clear(objid) has been called).

                   The caller does not own the returned pointer and (ideally)
                   should not modify the map directly (prefer get() and
                   set()). It is provided here primarily for serialization
                   purposes: so the map can be saved and loaded using
                   arbitrary serialization techniques.
                */
                static property_map_type * get_map( const object_id_type objid )
                {
                        typename object_map_type::iterator it = object_map().find( objid );
                        if( object_map().end() == it ) return 0;
                        property_map_type * pm = & ((*it).second);
                        return pm;
                }

        private:
                typedef property_server< ContextType, ObjIDType > this_type;
                typedef std::map<object_id_type,property_map_type> object_map_type;

                static object_map_type & object_map()
                {
                        return PACKAGE_NAMESPACE::phoenix<object_map_type,this_type>::instance();
                }

        };

} // namespace PACKAGE_NAMESPACE_PACKAGE_NAMESPACE

#endif // PROPERTY_SERVER_H_INCLUDED
