/* Copyright (C) 2005 Red Hat, Inc. */

struct dbase_file;
typedef struct dbase_file dbase_t;
#define DBASE_DEFINED

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "debug.h"
#include "handle.h"
#include "parse_utils.h"
#include "database_file.h"
#include "semanage_store.h"

/* Representation of the database once loaded in memory */
typedef struct cache_entry {
	record_t* data;
	struct cache_entry* prev;
	struct cache_entry* next;
} cache_entry_t;

/* FILE dbase */ 
struct dbase_file {

	/* Backing file suffix */
	const char* suffix;

	/* Base record table */
	record_table_t* rtable;

	/* FILE extension */
	record_file_table_t* rftable;

	/* In-memory representation (cache) */
	cache_entry_t* cache;
	cache_entry_t* cache_tail;

	size_t cache_sz;
	int cached;
	int modified;
};

static int construct_filename(
	semanage_handle_t* handle,
	dbase_file_t* dbase,
	char** filename) {

	const char* path = (handle->is_in_transaction)?
		semanage_path(SEMANAGE_TMP, SEMANAGE_TOPLEVEL):
		semanage_path(SEMANAGE_ACTIVE, SEMANAGE_TOPLEVEL);

	size_t fname_length = strlen(path) + strlen(dbase->suffix) + 2;

	char* fname = malloc(fname_length);
	if (!fname) {
		ERR(handle, "out of memory, could not construct filename");
		return STATUS_ERR;
	}
	snprintf(fname, fname_length, "%s/%s", path, dbase->suffix);

	*filename = fname;
	return STATUS_SUCCESS;
}


/* Helper for adding records to the cache */
static int dbase_file_cache_prepend(
	semanage_handle_t* handle, 
	dbase_file_t* dbase,
	record_t* data) {

	/* Initialize */
	cache_entry_t* entry =
		(cache_entry_t*) malloc(sizeof (cache_entry_t));
	if (entry == NULL)
		goto omem;
	entry->data = data;
	entry->prev = NULL;
	entry->next = dbase->cache;

	/* Link */
	if (dbase->cache != NULL)
		dbase->cache->prev = entry;
	if (dbase->cache_tail == NULL)
		dbase->cache_tail = entry;
	dbase->cache = entry;

	dbase->cache_sz++;
        return STATUS_SUCCESS;

        omem:
	ERR(handle, "out of memory, could not cache record");
        return STATUS_ERR;
}

static int dbase_file_cache(
	semanage_handle_t* handle,
	dbase_file_t* dbase) {

	record_t* process_record = NULL;
	int pstatus = STATUS_SUCCESS;

	parse_info_t* parse_info;
	char* fname = NULL;

	/* Already cached */
	if (dbase->cached)
		return STATUS_SUCCESS;

	dbase->cache_sz = 0;
	dbase->cache = NULL;
	dbase->cache_tail = NULL;

	if (construct_filename(handle, dbase, &fname) < 0)
		goto err;

	if (parse_init(handle, fname, NULL, &parse_info) < 0) 
		goto err;

	if (parse_open(handle, parse_info) < 0)
                goto err;

	/* Main processing loop */
	do {

		/* Create record */
		if (dbase->rtable->create(handle, &process_record) < 0)
			goto err;

		/* Parse record */
		pstatus = dbase->rftable->parse(handle, parse_info, process_record);

		/* Parse error */
		if (pstatus < 0)
			goto err;

		/* End of file */
		else if (pstatus == STATUS_NODATA) 
			break;

		/* Add record to list */
		if (dbase_file_cache_prepend(handle, dbase, process_record) < 0)
			goto err;
		process_record = NULL;

	} while (pstatus != STATUS_NODATA);

	dbase->rtable->free(process_record);
	parse_close(parse_info);
	parse_release(parse_info);
	free(fname);
	dbase->cached = 1;

	return STATUS_SUCCESS;

	err:
	ERR(handle, "could not cache file database");
	dbase->rtable->free(process_record);
	parse_close(parse_info);
	parse_release(parse_info);
	free(fname);
	return STATUS_ERR;
}

static void dbase_file_drop_cache(
	dbase_file_t* dbase) {

	if (!dbase->cached)
		return;

	cache_entry_t *prev, *ptr = dbase->cache;
	while (ptr != NULL) {
		prev = ptr;
		ptr = ptr->next;
		dbase->rtable->free(prev->data);
		free(prev);
	}

	dbase->cached = 0;
	dbase->modified = 0;
}	

/* Flush database to file */
static int dbase_file_flush(
	semanage_handle_t* handle,
	dbase_file_t* dbase) {

	cache_entry_t* ptr;
	char* fname = NULL;
	FILE* str = NULL;

	if (!dbase->modified || !dbase->cached)
		return STATUS_SUCCESS;

	if (construct_filename(handle, dbase, &fname) < 0)
		goto err;

	str = fopen(fname, "w");
	if (!str) {
		ERR(handle, "could not open %s for writing: %s",
			fname, strerror(errno));
		goto err;
	}

	for (ptr = dbase->cache_tail; ptr != NULL; ptr = ptr->prev) {
		if (dbase->rftable->print(handle, ptr->data, str) < 0)
			goto err;
	}

	dbase->modified = 0;
	fclose(str);
	free(fname);
	return STATUS_SUCCESS;

	err:
	if (str != NULL)
		fclose(str);
		
	ERR(handle, "could not flush database to file");
	free(fname);
	return STATUS_ERR;
}

/* Check if modified */
static int dbase_file_is_modified(
	dbase_file_t* dbase) {

	return dbase->modified;
}

/* Helper for finding records in the cache */
static int dbase_file_cache_locate(
	semanage_handle_t* handle,
	dbase_file_t* dbase,
	record_key_t* key,
	cache_entry_t** entry) {

	cache_entry_t* ptr;

	if (dbase_file_cache(handle, dbase) < 0)
		goto err;

	for (ptr = dbase->cache; ptr != NULL; ptr = ptr->next) {
		if (! dbase->rtable->compare(ptr->data, key)) {
			*entry = ptr;
			return STATUS_SUCCESS;
		}
	}

	return STATUS_NODATA;

	err:
	ERR(handle, "could not complete cache lookup");
	return STATUS_ERR;
}

int dbase_file_init(
	semanage_handle_t* handle, 
	const char* suffix,
	record_table_t* rtable,
	record_file_table_t* rftable,
	dbase_file_t** dbase) {

	dbase_file_t* tmp_dbase = 
		(dbase_file_t*) malloc(sizeof(dbase_file_t));
	
	if (!tmp_dbase)
		goto omem;

	tmp_dbase->suffix = suffix;
	tmp_dbase->rtable = rtable;
	tmp_dbase->rftable = rftable;
	tmp_dbase->cache = NULL;
	tmp_dbase->cache_tail = NULL;
	tmp_dbase->cache_sz = 0;
	tmp_dbase->cached = 0;
	tmp_dbase->modified = 0;

	*dbase = tmp_dbase;
	
	return STATUS_SUCCESS;
	
	omem:
	ERR(handle, "out of memory, could not initialize file database");
	free(tmp_dbase);
	return STATUS_ERR;
}
	
/* Release dbase resources */
void dbase_file_release(
	dbase_file_t* dbase) {

	dbase_file_drop_cache(dbase);
	free(dbase);
}

static int dbase_file_exists(
	semanage_handle_t* handle,
	dbase_file_t* dbase,
	record_key_t* key,
	int* response) {

	cache_entry_t* entry;
	int status;

	status = dbase_file_cache_locate(handle, dbase, key, &entry);
	if (status < 0)
		goto err;

	*response = (status != STATUS_NODATA);
	return STATUS_SUCCESS;

	err:
	ERR(handle, "could not check if record exists");
	return STATUS_ERR;
}

static int dbase_file_add(
	semanage_handle_t* handle,
	dbase_file_t* dbase,
	record_key_t* key,
	record_t* data) {

	int status;
	cache_entry_t* entry;

	status = dbase_file_cache_locate(handle, dbase, key, &entry);
	if (status < 0)
		goto err;

	if (status != STATUS_NODATA) {
		ERR(handle, "record is already in the database");
		goto err;
	}

	if (dbase_file_cache_prepend(handle, dbase, data) < 0)
		goto err;

	dbase->modified = 1;
	return STATUS_SUCCESS;

	err:
	ERR(handle, "could not add record to the database");
	return STATUS_ERR;
}

static int dbase_file_set(
	semanage_handle_t* handle,
	dbase_file_t* dbase,
	record_key_t* key,
	record_t* data) {

	cache_entry_t* entry;
	int status;

	status = dbase_file_cache_locate(handle, dbase, key, &entry);
	if (status < 0)
		goto err;
        if (status == STATUS_NODATA) {
		ERR(handle, "record not found in the database");
		goto err;
	}
	else {
		dbase->rtable->free(entry->data);
		entry->data = data;
	}

	dbase->modified = 1;
        return STATUS_SUCCESS;

	err:
	ERR(handle, "could not set record value");
	return STATUS_ERR;
}


static int dbase_file_modify(
	semanage_handle_t* handle,
	dbase_file_t* dbase,
	record_key_t* key,
	record_t* data) {

	cache_entry_t* entry;
	int status;

	status = dbase_file_cache_locate(handle, dbase, key, &entry);
	if (status < 0)
		goto err;
	if (status == STATUS_NODATA) {
		if (dbase_file_cache_prepend(handle, dbase, data) < 0)
			goto err;
	}
	else {
		dbase->rtable->free(entry->data);
		entry->data = data;
	}

	dbase->modified = 1;
	return STATUS_SUCCESS;

	err:
	ERR(handle, "could not modify record value");
	return STATUS_ERR;
}

static int dbase_file_count(
        semanage_handle_t* handle,
        dbase_file_t* dbase,
        unsigned int* response) {

	*response = dbase->cache_sz;
	handle = NULL;
	return STATUS_SUCCESS;
}

static int dbase_file_query(
	semanage_handle_t* handle,
	dbase_file_t* dbase,
	record_key_t* key,
	record_t** response) {

	cache_entry_t* entry;
	int status;
       
	status = dbase_file_cache_locate(handle, dbase, key, &entry);
	if (status < 0 || status == STATUS_NODATA)
		goto err;

	if (dbase->rtable->clone(handle, entry->data, response) < 0)
		goto err;

	return STATUS_SUCCESS;

	err:
	ERR(handle, "could not query record value");
	return STATUS_ERR;
}

static int dbase_file_iterate(
	semanage_handle_t* handle,
	dbase_file_t* dbase, 
	int (*fn) (record_t* record, void* fn_arg),
	void* arg) {

	int rc;
	cache_entry_t* ptr;

	for (ptr = dbase->cache; ptr != NULL; ptr = ptr->next) {

		rc = fn(ptr->data, arg);
		if (rc < 0) 
			goto err;
		
		else if (rc > 1)
			break;
        }


        return STATUS_SUCCESS;

	err:
	ERR(handle, "could not iterate over records");
	return STATUS_ERR;
}

static int dbase_file_del(
	semanage_handle_t* handle,
	dbase_file_t* dbase,
	record_key_t* key) {

	cache_entry_t *ptr, *prev = NULL;

	for (ptr = dbase->cache; ptr != NULL; ptr = ptr->next) {
		if (! dbase->rtable->compare(ptr->data, key)) {
			if (prev != NULL)
				prev->next = ptr->next;
			else
				dbase->cache = ptr->next;

			if (ptr->next != NULL)
				ptr->next->prev = ptr->prev;
			else
				dbase->cache_tail = ptr->prev;

			dbase->rtable->free(ptr->data);
			dbase->cache_sz--;
			free(ptr);
			dbase->modified = 1;
			return STATUS_SUCCESS;
		}
		else
			prev = ptr;
	}

	handle = NULL;
	return STATUS_SUCCESS;
}

static int dbase_file_list(
	semanage_handle_t* handle,
	dbase_file_t* dbase,
	record_t*** records,
	size_t* count) {

	cache_entry_t* ptr;
	record_t** tmp_records = NULL;
	size_t tmp_count;
	int i = 0;

	tmp_count = dbase->cache_sz;
	if (tmp_count > 0) {
		tmp_records = (record_t**)
			calloc(tmp_count, sizeof (record_t*));

		if (tmp_records == NULL)
			goto omem;

		for (ptr = dbase->cache; ptr != NULL; ptr = ptr->next) {
			if (dbase->rtable->clone(handle, 
				ptr->data, &tmp_records[i]) < 0)
				goto err;
			i++;
		}
	}

	*records = tmp_records;
	*count = tmp_count;
	return STATUS_SUCCESS;

	omem:
	ERR(handle, "out of memory");

	err:
	for (; i >= 0; i--)
		dbase->rtable->free(tmp_records[i]);
	free(tmp_records);
	ERR(handle, "could not allocate record array");
	return STATUS_ERR;
}

static record_table_t* dbase_file_get_rtable(
	dbase_file_t* dbase) {

	return dbase->rtable;
}


/* FILE dbase - method table implementation */
dbase_table_t SEMANAGE_FILE_DTABLE = {

	/* Cache/Transactions */
	.cache = dbase_file_cache,
	.drop_cache = dbase_file_drop_cache,
	.flush = dbase_file_flush,
	.is_modified = dbase_file_is_modified,

	/* Database API */
	.iterate = dbase_file_iterate,
	.exists = dbase_file_exists, 
	.list = dbase_file_list,   
	.add = dbase_file_add,
	.set = dbase_file_set,
	.del = dbase_file_del, 
	.modify = dbase_file_modify, 
	.query = dbase_file_query, 
	.count = dbase_file_count, 

	/* Polymorphism */
	.get_rtable = dbase_file_get_rtable
};
