////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007 Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

#ifndef SFML_FONTMANAGER_HPP
#define SFML_FONTMANAGER_HPP

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Graphics/Image.hpp>
#include <SFML/Graphics/Rect.hpp>
#include <SFML/System/NonCopyable.hpp>
#include <map>
#include <string>


namespace sf
{
namespace priv
{
////////////////////////////////////////////////////////////
/// FontManager manages font library and user defined fonts
////////////////////////////////////////////////////////////
class FontManager : NonCopyable
{
public :

    ////////////////////////////////////////////////////////////
    /// Defines a bitmap font (texture and character sizes)
    ////////////////////////////////////////////////////////////
    struct Font
    {
        struct Character
        {
            IntRect   Rect;
            FloatRect Coord;
            int       Advance;
        };

        Image                        Texture;    ///< Texture associated to font
        unsigned int                 CharSize;   ///< Characters size in the bitmap font
        std::map<wchar_t, Character> Characters; ///< Bounding box of each character
    };

    ////////////////////////////////////////////////////////////
    /// Get the unique instance of the class
    ///
    /// \return Reference to the sfFontManager instance
    ///
    ////////////////////////////////////////////////////////////
    static FontManager& GetInstance();

    ////////////////////////////////////////////////////////////
    /// Get a bitmap font from a font file (create it if doesn't exist)
    ///
    /// \param Filename : Path of the font file to load
    /// \param CharSize : Size of characters in bitmap - the bigger, the higher quality (16 by default)
    /// \param Charset :  Characters set to generate (empty by default - take the ASCII range [31, 255])
    ///
    /// \return Pointer to the bitmap font (default font is returned if an error occured)
    ///
    ////////////////////////////////////////////////////////////
    const Font& GetBitmapFont(const std::string& Filename, unsigned int CharSize = 16, std::wstring Charset = L"");

private :

    ////////////////////////////////////////////////////////////
    /// Default constructor
    ///
    ////////////////////////////////////////////////////////////
    FontManager();

    ////////////////////////////////////////////////////////////
    /// Destructor
    ///
    ////////////////////////////////////////////////////////////
    ~FontManager();

    ////////////////////////////////////////////////////////////
    /// Create the default font
    ////////////////////////////////////////////////////////////
    void CreateDefaultFont();

    ////////////////////////////////////////////////////////////
    // Types
    ////////////////////////////////////////////////////////////
    typedef std::map<std::string, Font> FontTable;

    ////////////////////////////////////////////////////////////
    // Member data
    ////////////////////////////////////////////////////////////
    FontTable myFonts; ///< Table of loaded bitmap fonts
};

} // namespace priv

} // namespace sf


#endif // SFML_FONTMANAGER_HPP
