////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007 Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#include <SFML/Graphics/RenderWindow.hpp>
#include <SFML/Graphics/Drawable.hpp>
#include <SFML/Graphics/GraphicsDevice.hpp>
#include <SFML/Graphics/Image.hpp>
#include <SFML/Graphics/OpenGL.hpp>
#include <iostream>


namespace sf
{
////////////////////////////////////////////////////////////
/// Default constructor
////////////////////////////////////////////////////////////
RenderWindow::RenderWindow() :
myBackgroundColor(Color(0, 0, 0, 255)),
myOptimizeStates (false)
{

}


////////////////////////////////////////////////////////////
/// Construct the window
////////////////////////////////////////////////////////////
RenderWindow::RenderWindow(VideoMode Mode, const std::string& Title, unsigned long WindowStyle, int AntialiasingLevel) :
myBackgroundColor(Color(0, 0, 0, 255)),
myOptimizeStates (false)
{
    Create(Mode, Title, WindowStyle, AntialiasingLevel);
}


////////////////////////////////////////////////////////////
/// Construct the window from an existing control
////////////////////////////////////////////////////////////
RenderWindow::RenderWindow(WindowHandle Handle, int AntialiasingLevel) :
myBackgroundColor(Color(0, 0, 0, 255)),
myOptimizeStates (false)
{
    Create(Handle, AntialiasingLevel);
}


////////////////////////////////////////////////////////////
/// Destructor
////////////////////////////////////////////////////////////
RenderWindow::~RenderWindow()
{
    // Nothing to do...
}


////////////////////////////////////////////////////////////
/// Draw something on the window
////////////////////////////////////////////////////////////
void RenderWindow::Draw(const Drawable& Object) const
{
    // Set our window as the current target for rendering
    if (SetCurrent())
    {
        // Save the current render states and set the SFML ones
        if (!myOptimizeStates)
        {
            static const GLbitfield States = GL_COLOR_BUFFER_BIT | GL_CURRENT_BIT | GL_ENABLE_BIT |
                                             GL_TEXTURE_BIT | GL_TRANSFORM_BIT | GL_VIEWPORT_BIT;
            GLCheck(glPushAttrib(States));
            SetRenderStates();
        }

        // Set the projection matrix corresponding to the current view
        GLCheck(glMatrixMode(GL_PROJECTION));
        GLCheck(glPushMatrix());
        GLCheck(glLoadMatrixf(myCurrentProjection));

        // Let the object draw itself
        Object.Draw(*this);

        // Restore the previous projection matrix
        GLCheck(glMatrixMode(GL_PROJECTION));
        GLCheck(glPopMatrix());

        // Restore render states
        if (!myOptimizeStates)
            GLCheck(glPopAttrib());
    }
}


////////////////////////////////////////////////////////////
/// Save the content of the window to an image
////////////////////////////////////////////////////////////
Image RenderWindow::Capture() const
{
    // Get the window dimensions
    const unsigned int Width  = GetWidth();
    const unsigned int Height = GetHeight();

    // Set our window as the current target for rendering
    if (SetCurrent())
    {
        // Get pixels from the backbuffer
        std::vector<Uint8> Pixels(Width * Height);
        glReadPixels(0, 0, Width, Height, GL_RGBA, GL_UNSIGNED_BYTE, &Pixels[0]);

        // Create an image from it and return it
        return Image(Width, Height, &Pixels[0]);
    }
    else
    {
        return Image(Width, Height, Color::White);
    }
}


////////////////////////////////////////////////////////////
/// Change the background color of the window
////////////////////////////////////////////////////////////
void RenderWindow::SetBackgroundColor(const Color& Col)
{
    myBackgroundColor = Col;
}


////////////////////////////////////////////////////////////
/// Change the current active view
////////////////////////////////////////////////////////////
void RenderWindow::SetView(const View* NewView)
{
    // Set our window as the current target for rendering
    if (SetCurrent())
    {
        // If a null pointer is passed, switch to default view
        if (NewView == NULL)
            NewView = &myDefaultView;

        // Compute the view rectangle coordinates
        float X = NewView->Rect.Left + NewView->Rect.GetWidth()  / 2;
        float Y = NewView->Rect.Top  + NewView->Rect.GetHeight() / 2;
        float HalfWidth  = NewView->Rect.GetWidth()  / (2 * NewView->Zoom);
        float HalfHeight = NewView->Rect.GetHeight() / (2 * NewView->Zoom);

        // Store the view rectangle for optimization purpose
        myCurrentRect.Left   = std::min(X - HalfWidth,  X + HalfWidth);
        myCurrentRect.Top    = std::min(Y - HalfHeight, Y + HalfHeight);
        myCurrentRect.Right  = std::max(X - HalfWidth,  X + HalfWidth);
        myCurrentRect.Bottom = std::max(Y - HalfHeight, Y + HalfHeight);

        // Update the projection matrix according to the new view, and save it
        GLCheck(glMatrixMode(GL_PROJECTION));
        GLCheck(glPushMatrix());
        GLCheck(glLoadIdentity());
        GLCheck(glOrtho(X - HalfWidth, X + HalfWidth, Y + HalfHeight, Y - HalfHeight, -1, 1));
        GLCheck(glGetFloatv(GL_PROJECTION_MATRIX, myCurrentProjection));
        GLCheck(glPopMatrix());
    }
}


////////////////////////////////////////////////////////////
/// Get current view rectangle
////////////////////////////////////////////////////////////
const FloatRect& RenderWindow::GetViewRect() const
{
    return myCurrentRect;
}


////////////////////////////////////////////////////////////
/// Tell SFML to optimize its calls to the graphics driver,
/// in case the user is not doing custom OpenGL calls
////////////////////////////////////////////////////////////
void RenderWindow::OptimizeForNonOpenGL(bool Optimize)
{
    myOptimizeStates = Optimize;
}


////////////////////////////////////////////////////////////
/// Called after the window has been created
////////////////////////////////////////////////////////////
void RenderWindow::OnCreate()
{
    // Set default OpenGL states
    SetRenderStates();

    // Setup the default view
    myDefaultView.Rect = FloatRect(0, 0, static_cast<float>(GetWidth()), static_cast<float>(GetHeight()));
    myDefaultView.Zoom = 1.f;
    SetView(NULL);
}


////////////////////////////////////////////////////////////
/// Called when the window displays its content on screen
////////////////////////////////////////////////////////////
void RenderWindow::OnDisplay()
{
    // Find which buffers we must clear
    GLbitfield                ClearBits  = GL_COLOR_BUFFER_BIT;
    if (GetDepthBits()   > 0) ClearBits |= GL_DEPTH_BUFFER_BIT;
    if (GetStencilBits() > 0) ClearBits |= GL_STENCIL_BUFFER_BIT;

    // Clear the color/depth/stencil buffers for next frame
    GLCheck(glClearColor(myBackgroundColor.r / 255.f,
                         myBackgroundColor.g / 255.f,
                         myBackgroundColor.b / 255.f,
                         myBackgroundColor.a / 255.f));
    GLCheck(glClear(ClearBits));
}


////////////////////////////////////////////////////////////
/// Called after an event has been received
////////////////////////////////////////////////////////////
void RenderWindow::OnEventReceived(const Event& EventReceived)
{
    // Adjust the viewport when the window is resized
    if (EventReceived.Type == Event::Resized)
    {
        if (SetCurrent())
            GLCheck(glViewport(0, 0, EventReceived.Size.Width, EventReceived.Size.Height));
    }
}


////////////////////////////////////////////////////////////
/// Set the OpenGL render states needed for the SFML rendering
////////////////////////////////////////////////////////////
void RenderWindow::SetRenderStates() const
{
    GLCheck(glAlphaFunc(GL_GREATER, 0));
    GLCheck(glDisable(GL_LIGHTING));
    GLCheck(glDisable(GL_DEPTH_TEST));
    GLCheck(glDisable(GL_CULL_FACE));
    GLCheck(glShadeModel(GL_SMOOTH));
    GLCheck(glClearDepth(1.f));
    GLCheck(glClearStencil(0));
}

} // namespace sf
