/* libspf - Sender Policy Framework library
*
*  ANSI C implementation of spf-draft-200405.txt
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*  Author: Sean Comeau   <scomeau@obscurity.org>
*
*  File:   spf.c
*  Desc:   Main library functionality
*
*  License:
*
*  The libspf Software License, Version 1.0
*
*  Copyright (c) 2004 James Couzens & Sean Comeau  All rights
*  reserved.
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*  1. Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*
*  2. Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the
*     distribution.
*
*  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
*  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
*  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
*  DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS MAKING USE OF THIS LICESEN
*  OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
*  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
*  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
*  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
*  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
*  SUCH DAMAGE.
*
*/

#include "../../config.h"       /* autoconf */
#include "main.h"               /* our header */
#include "util.h"               /* Utility functions */
#include "dns.h"                /* DNS functions */
#include "macro.h"              /* MACRO parsing functions */

#undef VERSION                  /* autoconf */

spf_config_t  confg;            /* global config struct */
u_int8_t      spf_rlevel;       /* global recursive counter */
int           h_errno;          /* error handling */


/* SPF_init
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/25/03
*
*  Desc:
*         Applies config options and returns allocated memory to a peer_info_t
*  structure.  Malloc returns NULL on a failure, and as such this function exits
*  with NULL upon failure to allocate memory which there by results in a
*  complete failure of the entire library.
*
*/
peer_info_t *SPF_init(const char *local, const char *rip, const char *expl,
  const char *tf, const char *guess, u_int32_t use_trust, u_int32_t use_guess)
{
  time_t        curtime;          /* time */
  char          *cur_utc_time;    /* time */

  peer_info_t   *peer_info;       /* peer info structure to be returned */

  /* spf_result struct including header texts */
  static spf_result_t spf_result[] =
  {
    { SIZEOF(HR_PASS),    HR_PASS,    SPF_PASS,     SIZEOF(HDR_PASS),     HDR_PASS,     '+'   },
    { SIZEOF(HR_NONE),    HR_NONE,    SPF_NONE,     SIZEOF(HDR_NONE),     HDR_NONE,     '\0'  },
    { SIZEOF(HR_S_FAIL),  HR_S_FAIL,  SPF_S_FAIL,   SIZEOF(HDR_S_FAIL),   HDR_S_FAIL,   '~'   },
    { SIZEOF(HR_H_FAIL),  HR_H_FAIL,  SPF_H_FAIL,   SIZEOF(HDR_H_FAIL),   HDR_H_FAIL,   '-'   },
    { SIZEOF(HR_ERROR),   HR_ERROR,   SPF_ERROR,    SIZEOF(HDR_ERROR),    HDR_ERROR,    '\0'  },
    { SIZEOF(HR_NEUTRAL), HR_NEUTRAL, SPF_NEUTRAL,  SIZEOF(HDR_NEUTRAL),  HDR_NEUTRAL,  '?'   },
    { SIZEOF(HR_UNKNOWN), HR_UNKNOWN, SPF_UNKNOWN,  SIZEOF(HDR_UNKNOWN),  HDR_UNKNOWN,  '\0'  },
    { SIZEOF(HR_UNMECH),  HR_UNMECH,  SPF_UNMECH,   SIZEOF(HDR_UNMECH),   HDR_UNMECH,   '\0'  }
  };

  xvprintf("Called with: (%s) (%s) (%s) (%s) (%s) %u:%u\n",
     local ? local : "NULL",
     rip   ? rip   : "NULL",
     expl  ? expl  : "NULL",
     tf    ? tf    : "NULL",
     guess ? guess : "NULL",
     use_trust, use_guess);

  /* allocate peer_info structure */
  peer_info = xmalloc(SIZEOF(peer_info_t));

  if (use_trust == TRUE)
  {
    peer_info->use_trust = TRUE;
  }
  else
  {
    peer_info->use_trust = FALSE;
  }

  if (use_guess == TRUE)
  {
    peer_info->use_guess = TRUE;
  }
  else
  {
    peer_info->use_guess = FALSE;
  }

  peer_info->spf_result   = spf_result;

  peer_info->ALL          = FALSE;
  peer_info->p            = NULL;

  peer_info->spf_ver      = 0;

  peer_info->helo         = NULL;
  peer_info->ehlo         = NULL;
  peer_info->from         = NULL;

  if (expl != NULL && (*expl && *(expl + 1)))
  {
    peer_info->explain    = xstrndup(expl, (strlen(expl) + 1));
  }
  else
  {
    peer_info->explain    = NULL;
  }

  if (guess != NULL && (*guess && *(guess + 1)))
  {
    peer_info->guess      = xstrndup(guess, (strlen(guess) + 1));
  }
  else
  {
    peer_info->guess      = xstrndup(SPF_GUESS, (SIZEOF(SPF_GUESS) + 1));
  }

  if (tf != NULL && (*tf && *(tf + 1)))
  {
    peer_info->trusted    = xstrndup(tf, (strlen(tf) + 1));
  }
  else
  {
    peer_info->trusted    = xstrndup(SPF_TRUSTED, (SIZEOF(SPF_TRUSTED) + 1));
  }

  peer_info->ptr_mhost    = NULL;
  peer_info->cur_dom      = NULL;
  peer_info->cur_eaddr    = NULL;
  peer_info->mta_hname    = xstrndup(local, (strlen(local) + 1));


  if (inet_pton(AF_INET, rip, &peer_info->addr) <= 0)
  {
    xeprintf("Unable to execute inet_print\n");
  }

  peer_info->r_ip         = xstrndup(rip, (strlen(rip) + 1));
  peer_info->r_vhname     = xstrndup(local, (strlen(local) + 1));

  snprintf(peer_info->ip_ver, IP_ADDR, "in-addr");

  memset(peer_info->local_part, '\0', LOCAL_PART);

  cur_utc_time = xmalloc(23);
  snprintf(cur_utc_time, 23, "%lu", time(&curtime));
  memcpy(peer_info->utc_time, cur_utc_time, 23);
  xfree(cur_utc_time);

  memset(peer_info->last_m, '\0', MAX_MECHANISM);
  memset(peer_info->error,  '\0', MAX_ERROR);

  /* localhost is exempt from SPF so automagical pass ;) */
  if ((strcmp(rip, "127.0.0.1") == 0) || (strcmp(rip, "localhost") == 0))
  {
    UTIL_assoc_prefix(peer_info, SPF_PASS, NULL);
  }
  else
  {
    UTIL_assoc_prefix(peer_info, SPF_NEUTRAL, NULL);
  }

  xprintf("libspf initialized succesfully. (%i bytes allocated)\n",
    SIZEOF(peer_info_t));

  return(peer_info);
}


/* SPF_close
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date: 12/25/03
*
*  Desc:
*       Free memory associated with passed peer_info_t structure.
*  peer_info _MUST_ be nulled, checks in the various MTA code rely
*  on it to be NULL for various checks.
*
*/
peer_info_t *SPF_close(peer_info_t *peer_info)
{
  if (peer_info == NULL)
  {
    xeprintf("peer structure null.  Aborting!\n");
    return(NULL);
  }

  xfree(peer_info->mta_hname);
  xfree(peer_info->helo);
  xfree(peer_info->from);
  xfree(peer_info->cur_dom);
  xfree(peer_info->r_ip);

  if (peer_info->ptr_mhost != NULL)
  {
    xfree(peer_info->ptr_mhost);
    peer_info = NULL;
  }

  xfree(peer_info->cur_eaddr);
  xfree(peer_info->trusted);
  xfree(peer_info->guess);
  xfree(peer_info->explain);
  xfree(peer_info->r_vhname);

  xfree(peer_info);
  peer_info = NULL;

  return(peer_info);
}


/* SPF_policy_main
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/10/03
*  Date:   01/24/04 by James <jcouzens@codeshare.ca>
*  Date:   05/02/04 by Teddy <teddy@teddy.ch>
*
*  Desc:
*          This calls only the recursive version SPF_policy_main_rec.
*  In earlier versions, SPF_policy_main was not able to handle CNAME
*
*/
SPF_RESULT SPF_policy_main(peer_info_t *peer_info)
{
  return SPF_policy_main_rec(peer_info, 0);
}


/* SPF_policy_main_rec
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:    12/10/03
*  Date:    01/24/04 by James <jcouzens@codeshare.ca>
*  Date:    05/02/04 by Teddy <teddy@teddy.ch>
*
*  Desc:
*          Fetches the SERVER_POLICY records from the FQDN's NS server
*  and returns an SPF_RESULT enumeration indicating the level of
*  compliance with an SPF policy (match, fail, error etc.. see spf.h)
*  Recursion inserted by Teddy
*
*/
SPF_RESULT SPF_policy_main_rec(peer_info_t *peer_info, int rec)
{
  char       *rr_data = NULL;      /* record */
  char       *domain  = NULL;      /* original domain */
  SPF_RESULT result;               /* result for recursive */

  /* localhost is exempt */
  if (peer_info->RES == SPF_PASS)
  {
    return(SPF_PASS);
  }

  if (spf_rlevel >= SPF_RECURSE)
  {
    snprintf(peer_info->error, MAX_ERROR, "INCLUDE loop, terminated.");
    UTIL_assoc_prefix(peer_info, SPF_UNKNOWN, NULL);
    return(SPF_UNKNOWN);
  }

  if (peer_info == NULL)
  {
    xeprintf("Error peer structure is NULL.  Aborting\n");
  }
  else if (peer_info->addr.s_addr <= 0)
  {
    xvprintf("(QID: %u) :: Error no idea who the remote peer is.  Abort!\n",
      spf_rlevel);

    UTIL_assoc_prefix(peer_info, SPF_ERROR, NULL);
    return(SPF_ERROR);
  }

  /* We check if domain name is CNAME */
  if ((rr_data = DNS_query(peer_info, peer_info->cur_dom, T_CNAME, NULL)) !=
    NULL)
  {
    /* See if maximum depth has been exceeded */
    if (MAX_CNAME > rec)
    {
      xvprintf("Domain (%s) is CNAME of (%s). Restarting SPF_policy_main.",
        peer_info->cur_dom, rr_data);

      domain = peer_info->cur_dom;
      peer_info->cur_dom = rr_data;
      result = SPF_policy_main_rec(peer_info, rec + 1);
      xfree(peer_info->cur_dom);
      peer_info->cur_dom = domain;
      return(result);
    }
    else
    {
      xvprintf("(QID: %u) :: Error: Maximum level (%d) )of recursion " \
        "reached. Abort!\n", spf_rlevel, MAX_CNAME);

      UTIL_assoc_prefix(peer_info, SPF_ERROR, NULL);
      xfree(rr_data);
      return(SPF_ERROR);
    }
  }

  if ((rr_data = DNS_query(peer_info, peer_info->cur_dom, T_TXT, NULL)) != NULL)
  {
    xprintf("(QID: %u) :: DNS_query returned with answer: (%s)\n",
      spf_rlevel, rr_data);

    UTIL_assoc_prefix(peer_info, SPF_NONE, NULL);
    SPF_parse_policy(peer_info, rr_data);
    xfree(rr_data);
  }
  else if (peer_info->RES == SPF_NONE && peer_info->use_trust == TRUE)
  {
    xvprintf("(QID: %u) :: Error obtaining record, trying Trusted Fwdr (%s)\n",
      spf_rlevel, peer_info->trusted);

    SPF_parse_policy(peer_info, peer_info->trusted);
  }

  if (peer_info->RES != SPF_PASS && peer_info->use_guess == TRUE)
  {
    xvprintf("(QID: %u) :: Attempting a best guess effort as a last resort "
      "using: (%s)\n", spf_rlevel, peer_info->guess);

    SPF_parse_policy(peer_info, peer_info->guess);
  }

  /* Didn't PASS and has a NULL explanation */
  if (peer_info->RES != SPF_PASS && peer_info->explain == NULL)
  {
    if ((peer_info->explain = MACRO_expand(peer_info, SPF_EXPLAIN)) == NULL)
    {
      xeprintf("Unable to obtain explanation\n");
    }
  }

  xprintf("(QID: %u) :: Return policy %i on mech: (%s) with outcome: (%s)\n",
    spf_rlevel, peer_info->RES, peer_info->last_m, peer_info->rs);

  UTIL_log_result(peer_info);
  return(peer_info->RES);
}


/*  SPF_parse_policy
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   12/19/03
*
*  Desc:
*          Parses the passed 'policy' and interprets the policies
*  contained within it to decide if based on various comparisons against
*  r_ip (remote peer's ipaddress) whether or not it meets the SPF
*  policy.
*
*/
SPF_BOOL SPF_parse_policy(peer_info_t *peer_info, const char *policy)
{
  SPF_MECHANISM   POLICY_TYPE;    /* SPF policy mechanism type */
  SPF_RESULT      PREFIX_TYPE;    /* SPF mechanism prefix type */
  SPF_BOOL        POLICY_MATCH;   /* T / F on a policy match */
  SPF_BOOL        POLICY_ERROR;   /* T /F for errors to return early */

  /* general use */
  policy_addr_t   *policy_ip;     /* ip address of a policy record */

  /* include */
  peer_info_t     *i_peer_info;   /* copy of peer_info with replaced domain */

  size_t          p_len;          /* length of passed policy string */
  int16_t         pos;            /* position in the policy */
  int16_t         s_pos;          /* position in a policy token */
  int16_t         i;

  char            *copy;          /* storage container for passed policy */
  char            *cp;            /* working pointer to passed policy */

  char            *token;         /* token pointer to piece of policy */
  char            *token_ptr;     /* working pointer for token */

  char            *tmp_ptr;       /* temporary utility pointer */
  char            *tmp_ptr2;      /* 2nd temporary utility pointer */
  char            *macro;         /* expanded macro storage container */

  u_int8_t        ver_c;          /* copy of spf version */

  POLICY_MATCH    = FALSE;
  POLICY_ERROR    = FALSE;
  PREFIX_TYPE     = SPF_NEUTRAL;

  policy_ip       = NULL;
  i_peer_info     = NULL;
  macro           = NULL;

  if ((policy == NULL) || (peer_info == NULL))
  {
    xeprintf("Unable to continue called with NULL structure\n");
    return(SPF_ERROR);
  }

  p_len = strlen(policy);

  xprintf("(QID: %u) :: about to parse (%s) of len: %i (%s)\n",
    spf_rlevel, policy, p_len, peer_info->spf_result[peer_info->RES].s);

  /* allocate memory and assign working pointer for policy */
  cp = copy = xstrndup(policy, (p_len + 1));

  i = strlen(cp);

  /* loops through the entire policy string until its exhausted */
  while (*cp)
  {
    while (*(cp + 1) && (pos = UTIL_index(cp, ' ')) == 0)
    {
      cp++;
    }

    if (!*(cp + 1))
    {
      xfree(copy);

      /* implicit return is neutral */
      if (peer_info->spf_ver == SPF_VERSION)
      {
        UTIL_assoc_prefix(peer_info, SPF_NEUTRAL, NULL);
        return(TRUE);
      }

      return(FALSE);
    }

    token = xstrndup(cp, (pos + 1));
    token_ptr = token;
    s_pos = 0;

    /* look for a mechanism modifier prefix */
    if (UTIL_is_spf_result(*token_ptr) ||
      (strncmp(token_ptr, "default", 7) == 0))
    {
      PREFIX_TYPE = UTIL_get_mech_prefix(peer_info, token_ptr);
      snprintf(peer_info->last_m, MAX_MECHANISM, "%s", token_ptr);

      if  (*token_ptr != 'd')
      {
        /* move ahead one because a prefix was specified */
        token_ptr++;
      }
    }
    else
    {
      snprintf(peer_info->last_m, MAX_MECHANISM, "%s", token_ptr);
      PREFIX_TYPE = SPF_PASS;
    }

    /* figure out what sort of mechanism we're working with */
    POLICY_TYPE = UTIL_get_policy_mech(token_ptr);

    xprintf("(QID: %u) :: SPF Policy Mechanism: %i (token: %s) (pos: %i)\n",
      spf_rlevel, POLICY_TYPE, token_ptr, pos);

    switch (POLICY_TYPE)
    {
      case NO_POLICY:
        break;
      case UNMECH:
        UTIL_assoc_prefix(peer_info, SPF_UNMECH, NULL);
        xfree(token);
        xfree(copy);
        return(FALSE);
      case VERSION:
        xprintf("(QID: %u) :: policy mechanism is version (%s)\n",
          spf_rlevel, token_ptr);

        if ((s_pos = UTIL_index(token_ptr, '=')) > 0)
        {
          token_ptr += s_pos + 4; /* skip =spf */

          if (atoi(token_ptr) > SPF_VERSION)
          {
            UTIL_assoc_prefix(peer_info, SPF_NONE, NULL);
            xfree(token);
            xfree(copy);
            return(FALSE);
          }

          peer_info->spf_ver = atoi(token_ptr);

          xvprintf("(QID: %u) :: SPF Version defined as: %u\n",
            spf_rlevel, peer_info->spf_ver);

        }
        else
        {
           xvprintf("(QID: %u) :: SPF version redefined! (%u)\n",
             spf_rlevel, peer_info->spf_ver);

           UTIL_assoc_prefix(peer_info, SPF_S_FAIL, NULL);
           xfree(token);
           xfree(copy);
           return(FALSE);
        }

        break;
      case ALL:
        xprintf("(QID: %u) :: policy mechanism is all (%s) policy: (%i)\n",
          spf_rlevel, token_ptr, POLICY_TYPE);

        UTIL_assoc_prefix(peer_info, PREFIX_TYPE, NULL);
        POLICY_MATCH = TRUE;
        peer_info->ALL = TRUE;

        break;
      case DEFAULT:
        xprintf("(QID: %u) :: policy mechanism is default (%s) policy: (%i)\n",
          spf_rlevel, token_ptr, POLICY_TYPE);

        POLICY_MATCH = TRUE;
        peer_info->ALL = TRUE;
        break;
      case INCLUDE:
        xprintf("(QID: %u) :: policy mechanism is include (%s)\n",
          spf_rlevel, token_ptr);

        if ((tmp_ptr = strstr(token_ptr, ":")) != NULL)
        {
          tmp_ptr++;

          ver_c = peer_info->spf_ver;
          if (UTIL_is_macro(tmp_ptr) == TRUE)
          {
            if ((macro = MACRO_expand(peer_info, tmp_ptr)) != NULL)
            {
              peer_info->cur_dom = xstrndup(macro, (strlen(macro) + 1));
              xfree(macro);
            }
          }
          else
          {
            peer_info->cur_dom = xstrndup(tmp_ptr, (strlen(tmp_ptr) + 1));
          }

          spf_rlevel++;
          peer_info->RES = SPF_NEUTRAL;
          PREFIX_TYPE = SPF_policy_main(peer_info);
          peer_info->spf_ver = ver_c; /* re-assign original proto ver */
          xvprintf("(QID: %u) :: include returned result %i (%s)\n",
            spf_rlevel, PREFIX_TYPE, peer_info->last_m);

          switch (PREFIX_TYPE)
          {
            case SPF_PASS:
              POLICY_MATCH = TRUE;
              break;
            case SPF_NONE:
              /* 'none' coming back from an include is a permanent error, and
               *  thus we return unknown
               */
              POLICY_ERROR = TRUE;
              PREFIX_TYPE = SPF_UNKNOWN;
              break;
            case SPF_S_FAIL:
              break;
            case SPF_H_FAIL:
              break;
            case SPF_ERROR:
              POLICY_ERROR = TRUE;
              break;
            case SPF_NEUTRAL:
              break;
            case SPF_UNKNOWN:
              POLICY_ERROR = TRUE;
              break;
            case SPF_UNMECH:
              break;
          } /* switch */

          if (POLICY_ERROR == TRUE)
          {
            UTIL_assoc_prefix(peer_info, PREFIX_TYPE, token_ptr);
            xfree(token);
            xfree(copy);
            return(FALSE);
          }
        }
        else
        {
          /* 'include' with no options returns unknown */
          POLICY_MATCH = TRUE;
          UTIL_assoc_prefix(peer_info, SPF_UNKNOWN, token_ptr);
          xfree(token);
          xfree(copy);
          return(TRUE);
        }
        break;

      case A:
        xprintf("(QID: %u) :: policy mechanism is A (%s)\n", spf_rlevel,
          token_ptr);

        if ((tmp_ptr = strstr(token_ptr, "/")) != NULL)
        {
          tmp_ptr++;
          POLICY_MATCH = UTIL_a_cmp(peer_info, token_ptr, atoi(tmp_ptr));
        }
        else
        {
          POLICY_MATCH = UTIL_a_cmp(peer_info, token_ptr, 32);
        }

        /* mechanism was supplied with a - indicating a failure on */
        if ((PREFIX_TYPE == SPF_H_FAIL) && (POLICY_MATCH == TRUE))
        {
          xvprintf("(QID: %u) :: Found a match on a negative prefix.  Halting"
            "parse.\n", spf_rlevel);

          UTIL_assoc_prefix(peer_info, SPF_H_FAIL, token_ptr);
          xfree(token);
          xfree(copy);
          return(FALSE);
        }

        break;

      case MX:
        xprintf("(QID: %u) :: policy mechanism is mx (%s)\n",
          spf_rlevel, token_ptr);

        /* we've been supplied an MX to use instead.. grab it */
        if ((tmp_ptr = strstr(token_ptr, ":")) != NULL)
        {
          tmp_ptr++;
          tmp_ptr2 = tmp_ptr;
        }
        else
        {
          tmp_ptr2 = peer_info->cur_dom;
        }

        if ((tmp_ptr = strstr(token_ptr, "/")) != NULL)
        {
          tmp_ptr++;
          POLICY_MATCH = UTIL_mx_cmp(peer_info, tmp_ptr2, atoi(tmp_ptr));
        }
        else
        {
          POLICY_MATCH = UTIL_mx_cmp(peer_info, tmp_ptr2, 32);
        }

        /* mechanism was supplied with a - indicating a failure on */
        if ((PREFIX_TYPE == SPF_H_FAIL) && (POLICY_MATCH == TRUE))
        {
          xvprintf("(QID: %u) :: Found a match on a negative prefix.  Halting"
            "parse.\n", spf_rlevel);

          UTIL_assoc_prefix(peer_info, SPF_H_FAIL, token_ptr);
          xfree(token);
          xfree(copy);
          return(FALSE);
        }

        break;
      case PTR:
        xprintf("(QID: %u) :: policy mechanism is ptr (%s)\n",
          spf_rlevel, token_ptr);

        POLICY_MATCH = UTIL_ptr_cmp(peer_info, token_ptr);

        /* mechanism was supplied with a - indicating a failure on */
        if ((PREFIX_TYPE == SPF_H_FAIL) && (POLICY_MATCH == TRUE))
        {
          xvprintf("(QID: %u) :: Found a match on a negative prefix.  Halting"
            "parse.\n", spf_rlevel);

          break;
        }

        break;
      case IP4:
        xprintf("(QID: %u) :: policy mechanism is ip4 (%s)\n",
          spf_rlevel, token_ptr);

        if ((policy_ip = UTIL_expand_ip(token_ptr)) == NULL)
        {
          xprintf("(QID: %u) :: ERROR: inet_pton unable to convert ip to "
            "binary\n", spf_rlevel);

          break;
        }

        xvprintf("(QID: %i) :: POL: %lu (%s) PEER: %lu (%s)\n",
          spf_rlevel, policy_ip->addr.s_addr, token_ptr, peer_info->addr.s_addr,
          peer_info->r_ip);
        POLICY_MATCH = UTIL_cidr_cmp(/*peer_info, */policy_ip, &peer_info->addr);
        xfree(policy_ip);

        /* mechanism was supplied with a - indicating a failure on */
        if ((PREFIX_TYPE == SPF_H_FAIL) && (POLICY_MATCH == TRUE))
        {
          xvprintf("(QID: %u) :: Found a match on a negative prefix.  Halting "
            "parse.\n", spf_rlevel);

          UTIL_assoc_prefix(peer_info, SPF_H_FAIL, token_ptr);
          xfree(token);
          xfree(copy);
          return(FALSE);
        }

        break;
      case IP6:
        xprintf("(QID: %u) :: policy mechanism is ip6 (%s)\n",
          spf_rlevel, token_ptr);
        break;
      case EXISTS:
        xprintf("(QID: %u) :: policy mechanism is exists (%s)\n",
          spf_rlevel, token_ptr);

        if ((tmp_ptr = strstr(token_ptr, ":")) != NULL)
        {
          tmp_ptr++;

          if ((macro = MACRO_expand(peer_info, tmp_ptr)) == NULL)
          {
            xvprintf("(QID: %u) :: Unable to expand macro (%s). "
              "Aborting.\n",  spf_rlevel, tmp_ptr);

            break;
          }

          if (DNS_query(peer_info, macro, T_A, NULL) == (char *)TRUE)
          {
            POLICY_MATCH = TRUE;
          }

          xfree(macro);
        }
        break;
      case REDIRECT:
        xprintf("(QID: %u) :: policy mechanism is redirect (%s)\n",
          spf_rlevel, token_ptr);

        if ((tmp_ptr = strstr(token_ptr, "=")) != NULL)
        {
          tmp_ptr++;

          ver_c = peer_info->spf_ver;
          if (UTIL_is_macro(tmp_ptr) == TRUE)
          {
            if ((macro = MACRO_expand(peer_info, tmp_ptr)) != NULL)
            {
              xfree(peer_info->cur_dom);
              peer_info->cur_dom = xstrndup(macro, (strlen(macro) + 1));
              xfree(macro);
            }
          }
          else
          {
            xfree(peer_info->cur_dom);
            peer_info->cur_dom = xstrndup(tmp_ptr, (strlen(tmp_ptr) + 1));
          }

          spf_rlevel++;
          peer_info->RES = SPF_NEUTRAL;
          PREFIX_TYPE = SPF_policy_main(peer_info);

          /* re-assign original proto ver */
          peer_info->spf_ver = ver_c;

          xvprintf("(QID: %u) :: redirect returned result %i (%s)\n",
            spf_rlevel, PREFIX_TYPE, peer_info->last_m);

          switch (PREFIX_TYPE)
          {
            case SPF_PASS:
              POLICY_MATCH = TRUE;
              break;
            case SPF_NONE:
              /* 'none' coming back from an redirect is a permanent error, and
               *  thus we return unknown
               */
              POLICY_ERROR = TRUE;
              PREFIX_TYPE = SPF_UNKNOWN;
              break;
            case SPF_S_FAIL:
              break;
            case SPF_H_FAIL:
              break;
            case SPF_ERROR:
              POLICY_ERROR = TRUE;
              break;
            case SPF_NEUTRAL:
              break;
            case SPF_UNKNOWN:
              POLICY_ERROR = TRUE;
              break;
            case SPF_UNMECH:
              break;
          } /* switch */

          if (POLICY_ERROR == TRUE)
          {
            UTIL_assoc_prefix(peer_info, PREFIX_TYPE, token_ptr);
            xfree(token);
            xfree(copy);
            return(FALSE);
          }
        }
        break;
      case EXPLAIN:
        xprintf("(QID: %u) :: policy mechanism is explain (%s)\n",
          spf_rlevel, token_ptr);

        if ((tmp_ptr = strstr(token_ptr, "=")) != NULL)
        {
          tmp_ptr++;

          if ((macro = MACRO_expand(peer_info, tmp_ptr)) == NULL)
          {
            xvprintf("(QID: %u) :: Unable to expand macro (%s). Aborting.\n",
              spf_rlevel, tmp_ptr);

            break;
          }

          peer_info->explain = xstrndup(macro, (strlen(macro) + 1));
          xfree(macro);
        }
        else
        {
          xvprintf("(QID: %u) :: EXPLAIN modifier must be accompanies " \
	    "by arguments and I found none.", spf_rlevel);

          /* 'include' with no options returns unknown */
          POLICY_MATCH = TRUE;
          UTIL_assoc_prefix(peer_info, SPF_UNKNOWN, token_ptr);
          xfree(token);
          xfree(copy);
          return(TRUE);
        }

       break;
    } /* switch */

    xfree(token);
    cp += pos + 1;  /* move over to the next mechanism */

    if ((POLICY_MATCH == TRUE) && (peer_info->spf_ver > 0))
    {
      /*peer_info->RES = SPF_PASS;*/
      UTIL_assoc_prefix(peer_info, PREFIX_TYPE, peer_info->last_m);
      peer_info->RES_P = peer_info->RES;
      xfree(copy);
      return(TRUE);
    }
  } /* while parsing policy */

  xfree(copy);
  return(FALSE);
}


/* SPF_result
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   02/02/04
*
*  Desc:
*         Returns a string of allocated memory detailing a response
*  that the MTA will return to the client based on the result
*  received from an SPF query.
*
*/
char *SPF_result(peer_info_t *peer_info)
{
  char  *buf;

  buf = xmalloc(MAX_HEADER);
  memset(buf, '\0', MAX_HEADER);

  switch (peer_info->RES)
  {
    case SPF_PASS:
      snprintf(buf, MAX_SMTP_RES, RES_PASS, peer_info->from,
        peer_info->r_ip);
      break;
    case SPF_NONE:
      snprintf(buf, MAX_SMTP_RES, RES_NONE, peer_info->from);
      break;
    case SPF_S_FAIL:
      snprintf(buf, MAX_SMTP_RES, RES_S_FAIL, peer_info->from,
        peer_info->r_ip);
      break;
    case SPF_H_FAIL:
      snprintf(buf, MAX_SMTP_RES, RES_H_FAIL, peer_info->from,
        peer_info->r_ip);
      break;
    case SPF_ERROR:
      snprintf(buf, MAX_SMTP_RES, RES_ERROR, peer_info->from);
      break;
    case SPF_NEUTRAL:
      snprintf(buf, MAX_SMTP_RES, RES_NEUTRAL, peer_info->r_ip,
        peer_info->from);
      break;
    case SPF_UNKNOWN:
      snprintf(buf, MAX_SMTP_RES, RES_UNKNOWN, peer_info->from);
      break;
    case SPF_UNMECH:
      snprintf(buf, MAX_SMTP_RES, RES_UNMECH, peer_info->from);
      break;
  }

  xprintf("response: (%s)\n", buf);

  return(buf);
}


/*  SPF_get_explain
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/25/04
*
*  Desc:
*         Takes the contents of a returned explain parse (if any),
*  allocates memory for it and returns that memory with the contents of
*  that string.  Calling function is required to free this memory.
*
*
*/
char *SPF_get_explain(peer_info_t *peer_info)
{
  char    *buf = NULL;

  if (peer_info->explain != NULL)
  {
    if ((buf = MACRO_expand(peer_info, SPF_EXPLAIN)) != NULL)
    {
      xprintf("Prepending explain: (%s)\n",
         buf);
       return(buf);
    }
  }

  return(NULL);
}


/*  SPF_build_header
*
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   01/25/04
*
*  Desc:
*         Allocates memory and builds a string which contains a
*  Received-SPF header.  The calling function is responsible to free
*  this memory.
*
*
*/
char *SPF_build_header(peer_info_t *peer_info)
{
  char  *buf;

  buf = xmalloc(MAX_HEADER);
  memset(buf, '\0', MAX_HEADER);

  switch (peer_info->RES)
  {
    case SPF_PASS:
      snprintf(buf, MAX_HEADER, peer_info->spf_result[SPF_PASS].h,
        peer_info->mta_hname, peer_info->from, peer_info->r_ip,
        peer_info->mta_hname, peer_info->r_ip, peer_info->from);
      break;
    case SPF_NONE:
      snprintf(buf, MAX_HEADER, peer_info->spf_result[SPF_NONE].h,
        peer_info->mta_hname, peer_info->from);
      break;
    case SPF_S_FAIL:
      snprintf(buf, MAX_HEADER, peer_info->spf_result[SPF_S_FAIL].h,
        peer_info->mta_hname, peer_info->from, peer_info->r_ip,
        peer_info->mta_hname, peer_info->r_ip, peer_info->from);
      break;
    case SPF_H_FAIL:
      snprintf(buf, MAX_HEADER, peer_info->spf_result[SPF_H_FAIL].h,
        peer_info->mta_hname, peer_info->from, peer_info->r_ip,
        peer_info->mta_hname, peer_info->r_ip, peer_info->from);
      break;
    case SPF_ERROR:
      snprintf(buf, MAX_HEADER, peer_info->spf_result[SPF_ERROR].h,
        peer_info->mta_hname, peer_info->from, peer_info->error);
      break;
    case SPF_NEUTRAL:
      snprintf(buf, MAX_HEADER, peer_info->spf_result[SPF_NEUTRAL].h,
        peer_info->mta_hname, peer_info->from, peer_info->r_ip);
      break;
    case SPF_UNKNOWN:
      snprintf(buf, MAX_HEADER, peer_info->spf_result[SPF_UNKNOWN].h,
        peer_info->mta_hname, peer_info->from, peer_info->cur_dom,
        peer_info->last_m);
      break;
    case SPF_UNMECH:
      snprintf(buf, MAX_HEADER, peer_info->spf_result[SPF_UNMECH].h,
        peer_info->last_m, peer_info->mta_hname, peer_info->from);
      break;
  }

  xvprintf("Prepending header string: (%s)\n", buf);

  return(buf);
}


/* SPF_smtp_helo
*
*  Author: Sean Comeau <scomeau@obscurity.org>
*          James Couzens <jcouzens@codeshare.ca>
*          Patrick Earl (http://patearl.net/)
*
*  Date:   12/10/03
*          12/25/03 (modified / renamed)
*          02/04/04 (modified)
*
*  Desc:
*          Fetches the HELO from MTA and places into the global
*  peer_info structure.
*
*/
SPF_BOOL SPF_smtp_helo(peer_info_t *peer_info, const char *s)
{

  xprintf("called with (%s)\n", s);

  if (peer_info->helo)
  {
    xfree(peer_info->helo);
  }

  peer_info->helo = xstrdup(s);
  peer_info->ehlo = peer_info->helo;

  return (strlen(peer_info->helo) > 0);
}


/* SPF_smtp_from
*
*  Author: Sean Comeau <scomeau@obscurity.org>
*          James Couzens <jcouzens@codeshare.ca>
*          Patrick Earl (http://patearl.net/)
*
*  Date:   12/10/03
*          12/25/03 James Couzens <jcouzens@codeshare.ca>   (rewrite)
*          02/04/04 Patrick Earl (http://patearl.net) (modified)
*  Date:   02/05/04 Teddy <teddy@teddy.ch>
*
*  Desc:
*          Fetches the FROM from MTA and places into the global
*  peer_info structure.
*
*
*/
SPF_BOOL SPF_smtp_from(peer_info_t *peer_info, const char *s)
{
  int       len;      /* utility */
  char      *pos;     /* position in string */

  xfree(peer_info->from);
  xfree(peer_info->cur_dom);

  if (s == NULL)
  {
    if (*(peer_info->helo) == '\0')
    {
      peer_info->from = xstrdup("unknown");
    }
    else
    {
      peer_info->from = xstrdup(peer_info->helo);
    }

    xvprintf("NULL or invalid MAIL FROM rcvd.  Working with %s from now on.\n",
      peer_info->from ? peer_info->from : peer_info->helo);

    return(FALSE);
  }

  peer_info->from = xstrdup(s);

  /* strips < and > from E-Mail-Address */
  if (*peer_info->from == 60) {
    pos = peer_info->from;
    *(peer_info->from + strlen(peer_info->from) - 1) = 0;
    peer_info->from++;
    peer_info->from = xstrdup(peer_info->from);
    xfree(pos);
  }

  if (*peer_info->from == '\0')
  {
    if (*(peer_info->helo) == '\0')
    {
      peer_info->from = xstrdup("unknown");
    }
    else
    {
      peer_info->from = xstrdup(peer_info->helo);
    }
  }

  pos = peer_info->from;

  xprintf("FROM: (%s) (called with: %s)\n", peer_info->from, s);

  if ((pos = strstr(pos, "@")))
  {
    len = (pos - peer_info->from);

    if (len >= LOCAL_PART)
    {
      xvprintf("%s is >= %s causing data overrun\n", len, LOCAL_PART);
      /*xeprintf("die\n");*/
    }

    /* Copy everything up to the @ into local_part.  Do not include the @. */
    memcpy(peer_info->local_part, peer_info->from, len);
    peer_info->local_part[len] = 0;

    pos++; /* skip past the @ */
    peer_info->cur_dom = xstrdup(pos);

    xprintf("CUR DOM: (%s)\n", peer_info->cur_dom);

  }
  else
  {
    strcpy(peer_info->local_part, "postmaster");
    peer_info->cur_dom = xstrdup(peer_info->from);
  }

  xvprintf("LOCAL: (%s) DOMAIN (%s) SENDER: (%s)\n", peer_info->local_part,
    peer_info->cur_dom, peer_info->from);

  return(TRUE);
}

/* end spf.c */
