/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

/**
 * @file SpiffToolbox.h
 * Interface of Spiff::Toolbox.
 */

#ifndef SPIFF_TOOLBOX_H
#define SPIFF_TOOLBOX_H



#include "SpiffDefines.h"
#include <string.h>

namespace Spiff {

/**
 * Provides common helper functions.
 */
namespace Toolbox {



/// @cond DOXYGEN_NON_API

/**
 * Compares the content of two strings.
 * Used with STL containers.
 */
struct SpiffStringCompare {

	/**
	 * Returns true if string <code>s1</code> is lower than
	 * <code>s2</code>, false otherwise.
	 *
	 * @return	<code>s1</code> &lt; <code>s2</code>
	 */
	bool operator()(const XML_Char * s1, const XML_Char * s2) const;

};

/// @endcond



/**
 * Duplicates the string (using new() not malloc())
 * and returns the duplicate. If <code>source</code>
 * is NULL the return value also is NULL.
 *
 * @param source  Source text
 * @return        Duplacated text or NULL
 */
XML_Char * newAndCopy(const XML_Char * source);

/**
 * Replaces the string in <c>*dest</c> by a duplicate of the string in <c>src</c>
 * (using new() not malloc()). The old string is deleted.
 *
 * @param dest	Destination text
 * @param src	Source text
 */
void deleteNewAndCopy(XML_Char ** dest, const XML_Char * src);

/**
 * Replaces the string in <c>dest</c> by a duplicate of the string in <c>src</c>
 * (using new() not malloc()). The old string is deleted. If <c>destOwnership</c>
 * is false the old string is not deleted. If <c>sourceCopy</c> is false only
 * <c>source</c>'s pointer is copied, not the string.
 *
 * @param dest			Destination text
 * @param destOwnership	Destination ownership flag
 * @param source		Source text
 * @param sourceCopy	Source copy flag
 */
void deleteNewAndCopy(const XML_Char * & dest, bool & destOwnership,
	const XML_Char * source, bool sourceCopy);

/**
 * Sets a pointer to <c>NULL</c> and returns the
 * original value. This helper is used for <i>stealing</i>
 * memory.
 *
 * @param dest	Destination
 * @return		Old value
 */
template <class T>
const T * getSetNull(const T * & dest) {
	const T * backup = dest;
	dest = NULL;
	return backup;
}


/**
 * Copies a string's content if owned or just it's address if not.
 *
 * @param dest       Reference of destination string
 * @param ownDest    Reference of destination owner flag
 * @param source     Source string
 * @param ownSource  Source owner flag
 */
void copyIfOwned(const XML_Char * & dest, bool & ownDest, const XML_Char * source, bool ownSource);



/**
 * Deletes the text behind <code>dest</code> if owned
 * and non-NULL.
 * NOTE: <code>dest</code> is not set to NULL after
 *
 * @param dest     Reference of string to delete
 * @param ownDest  Owner flag, false will prevent deletion
 */
void freeIfOwned(const XML_Char * & dest, bool ownDest);



/**
 * Calls atoi() on a limited number of characters.
 *
 * @param text	Text
 * @param len	Number of characters to read
 * @return		Result of atoi()
 */
int PORT_ANTOI(const XML_Char * text, int len);



}
}

#endif // SPIFF_TOOLBOX_H
