/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
/*
 * This code is based on public domain code
 * 
 */
 
#include "syncml.h"
#include "sml_error.h"

#include "syncml_internals.h"

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

/** @brief Function description
 * 
 * @param parameter This parameter does great things
 * @returns What you always wanted to know
 * 
 */
static const unsigned char pr2six[256] =
{
	/* ASCII table */
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 64, 64, 64, 63,
	 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 64, 64, 64, 64, 64, 64,
	 64,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
	 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 64, 64, 64, 64, 64,
	 64, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
	 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
	 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64
};

SmlBool smlBase64Decode(const char *input, char **output, unsigned int *outsize, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, input, output, outsize, error);
	smlAssert(output);
	smlAssert(outsize);
	
	if (!input) {
		*output = NULL;
		*outsize = 0;
		smlTrace(TRACE_EXIT, "%s", __func__);
		return TRUE;
	}
	
	if (smlBase64DecodeBinary(input, strlen(input), output, outsize, error)) {
		smlTrace(TRACE_EXIT, "%s", __func__);
		return TRUE;
	}
	
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlBase64DecodeBinary(const char *input, unsigned int size, char **output, unsigned int *outsize, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p, %p)", __func__, input, size, output, outsize, error);

	register const unsigned char *bufin;
	register unsigned char *bufout;
	register int nprbytes;
	
	if (!input || !size) {
		*output = NULL;
		*outsize = 0;
		goto out;
	}
	
	int len;
	for (len = 0; len < size; len++) {
		if (pr2six[(int)input[len]] > 63) {
			if (input[len] == '=' && (len == size - 1 || len == size -2))
				continue;
			smlErrorSet(error, SML_ERROR_GENERIC, "Invalid base64 input");
			goto error;
		}
	}
	
	*outsize = ((size * 3) / 4) + 1;
	if (input[size - 1] == '=')
		(*outsize)--;
	if (input[size - 2] == '=')
		(*outsize)--;
	
	*output = smlTryMalloc0(*outsize, error);
	if (!*output)
		goto error;
		
	bufout = (unsigned char *)*output;
	bufin = (const unsigned char *)input;
	nprbytes = *outsize - 1;
	
	while (nprbytes >= 3) {
		*(bufout++) = (unsigned char) (pr2six[*bufin] << 2 | pr2six[bufin[1]] >> 4);
		*(bufout++) = (unsigned char) (pr2six[bufin[1]] << 4 | pr2six[bufin[2]] >> 2);
		*(bufout++) = (unsigned char) (pr2six[bufin[2]] << 6 | pr2six[bufin[3]]);
		bufin += 4;
		nprbytes -= 3;
	}
	
	/* Note: (nprbytes == 1) would be an error, so just ingore that case */
	if (nprbytes > 0)
		*(bufout++) = (unsigned char) (pr2six[*bufin] << 2 | pr2six[bufin[1]] >> 4);
	
	if (nprbytes > 1)
		*(bufout++) = (unsigned char) (pr2six[bufin[1]] << 4 | pr2six[bufin[2]] >> 2);

	*(bufout) = '\0';
	
out:
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	*output = NULL;
	*outsize = 0;
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static const char basis_64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

SmlBool smlBase64Encode(const char *input, char **output, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, input, output, error);
	smlAssert(output);
	
	if (!input) {
		*output = NULL;
		smlTrace(TRACE_EXIT, "%s", __func__);
		return TRUE;
	}
	
	if (smlBase64EncodeBinary(input, strlen(input), output, error)) {
		smlTrace(TRACE_EXIT, "%s", __func__);
		return TRUE;
	}
	
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}
 
SmlBool smlBase64EncodeBinary(const char *input, unsigned int size, char **output, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p)", __func__, input, size, output, error);
	smlAssert(output);
	
	int i;
	char *p;
	
	if (!input) {
		*output = NULL;
		goto out;
	}
	
	int outsize = (((size + 2) / 3) * 4) + 1;
	
	*output = smlTryMalloc0(outsize, error);
	if (!*output)
		goto error;
	
	p = *output;
	for (i = 0; i < size - 2; i += 3) {
		*p++ = basis_64[(input[i] >> 2) & 0x3F];
		*p++ = basis_64[((input[i] & 0x3) << 4) | ((int) (input[i + 1] & 0xF0) >> 4)];
		*p++ = basis_64[((input[i + 1] & 0xF) << 2) | ((int) (input[i + 2] & 0xC0) >> 6)];
		*p++ = basis_64[input[i + 2] & 0x3F];
	}
	 
	if (i < size) {
		*p++ = basis_64[(input[i] >> 2) & 0x3F];
		if (i == (size - 1)) {
			*p++ = basis_64[((input[i] & 0x3) << 4)];
			*p++ = '=';
		} else {
			*p++ = basis_64[((input[i] & 0x3) << 4) | ((int) (input[i + 1] & 0xF0) >> 4)];
			*p++ = basis_64[((input[i + 1] & 0xF) << 2)];
		}
		*p++ = '=';
	}
	
	*p++ = '\0';

out:
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/*@}*/
