/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "syncml.h"

#include "syncml_internals.h"
#include "sml_manager_internals.h"
#include "sml_transport_internals.h"
#include "sml_command_internals.h"
#include "sml_elements_internals.h"

typedef struct managerSession {
	SmlLink *link;
	SmlSession *session;
	SmlTransport *transport;
	GList *objects;
} managerSession;

SmlObject *smlManagerObjectFindInternal(
		SmlManager *manager,
		SmlSession *session,
		SmlCommandType type,
		SmlLocation *target,
		SmlLocation *source,
		const char* contentType);

static managerSession *_manager_session_find(SmlManager *manager, SmlSession *session)
{
	smlAssert(session);
	smlAssert(manager);
	
	GList *s = NULL;
	for (s = manager->sessions; s; s = s->next) {
		managerSession *sess = s->data;
		if (sess->session == session)
			return sess;
	}
	return NULL;
}

static void _manager_session_free(managerSession *sess)
{
	smlAssert(sess);
	
	smlSessionUnref(sess->session);
	
	if (sess->link)
		smlLinkDeref(sess->link);
	
	while (sess->objects) {
		SmlObject *object = sess->objects->data;
		smlManagerObjectFree(object);
		sess->objects = g_list_delete_link(sess->objects, sess->objects);
	}
	
	g_free(sess);
}

static gboolean _manager_prepare_internal(GSource *source, gint *timeout_)
{
	*timeout_ = 1;
	return FALSE;
}

static gboolean _manager_check_internal(GSource *source)
{
	SmlManager *manager = *((SmlManager **)(source + 1));
	GList *s = NULL;
	for (s = manager->sessions; s; s = s->next) {
		managerSession *session = s->data;
		if (smlSessionCheck(session->session))
			return TRUE;
	}
	return FALSE;
}

static gboolean _manager_dispatch_internal(GSource *source, GSourceFunc callback, gpointer user_data)
{
	int max = 100;
	
	SmlManager *manager = user_data;
	GList *s = NULL;
	for (s = manager->sessions; s; s = s->next) {
		managerSession *session = s->data;
		while (smlSessionCheck(session->session) && max) {
			smlSessionDispatch(session->session);
			max--;
		}
	}
	
	return TRUE;
}

static void _smlManagerSendEvent(SmlManager *manager, SmlManagerEventType type, SmlSession *session, SmlCommand *command, SmlCommand *parent, SmlError *error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p, %p, %p)", __func__, manager, type, session, command, parent, error);
	
	SmlManagerEvent *event = smlTryMalloc0(sizeof(SmlManagerEvent), NULL);
	if (event) {
		event->type = type;
		
		if (session) {
			event->session = session;
			smlSessionRef(session);
		} else {
			event->session = NULL;
		}
		
		if (command) {
			event->command = command;
			smlCommandRef(command);
		} else {
			event->command = NULL;
		}
		
		if (parent) {
			event->parent = parent;
			smlCommandRef(parent);
		} else {
			event->parent = NULL;
		}
		
		if (error) {
			event->error = error;
			smlErrorRef(&error);
		}
		
		smlQueueSend(manager->userEventQueue, event);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void _smlManagerEventFree(SmlManagerEvent *event)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, event);
	smlAssert(event);
	
	if (event->session)
		smlSessionUnref(event->session);
	
	if (event->command)
		smlCommandUnref(event->command);
	
	if (event->parent)
		smlCommandUnref(event->parent);
	
	if (event->error)
		smlErrorDeref(&event->error);
	
	g_free(event);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/* If we return FALSE in this function, the current request will get
 * aborted */
static SmlBool _smlManagerDataHandler(SmlTransport *tsp, SmlLink *link, SmlTransportEventType type, SmlTransportData *data, SmlError *error, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %p, %p, %p)", __func__, tsp, link, type, data, error, userdata);
	smlAssert(userdata);
	SmlManager *manager = userdata;
	SmlParser *parser = NULL;
	SmlHeader *header = NULL;
	SmlCred *cred = NULL;
	SmlSession *session = NULL;
	SmlError *locerror = NULL;
	
	switch (type) {
		case SML_TRANSPORT_EVENT_CONNECT_DONE:
			/* Pass the connect through */
			_smlManagerSendEvent(manager, SML_MANAGER_CONNECT_DONE, NULL, NULL, NULL, NULL);
			break;
		case SML_TRANSPORT_EVENT_DISCONNECT_DONE:
			/* Pass the disconnect through */
			_smlManagerSendEvent(manager, SML_MANAGER_DISCONNECT_DONE, NULL, NULL, NULL, NULL);
			break;
		case SML_TRANSPORT_EVENT_DATA:
	
			parser = smlParserNew(data->type, 0, &locerror);
			if (!parser)
				goto session_error;
			
			/* start to parse the data */
			if (!smlParserStart(parser, data->data, data->size, &locerror))
				goto error_free_parser;
			
			/* Get the header of the message*/
			header = NULL;
			cred = NULL;
			if (!smlParserGetHeader(parser, &header, &cred, &locerror))
				goto error_free_parser;
			
			/* Find the session if available otherwise register it as new */
			session = smlManagerSessionFind(manager, header->sessionID);
			if (!session) {
				/* If we use the target and source from the header of the received message
				 * then we MUST interchange source and target */
				if (!(session = smlSessionNew(SML_SESSION_TYPE_SERVER, data->type, header->version, header->protocol, header->source, header->target, header->sessionID, header->messageID + 1, &locerror)))
					goto error_free_header;
				
				if (!smlManagerSessionAdd(manager, session, link, &locerror))
					goto error_free_header;
			}
			
			/* Find the manager session so that we can update the link. We
			 * have to do this since each time a transport with links receives
			 * a new data packets, it might create a new link (connectionless
			 * protocols). So we have to update the link to be able to answer
			 * the data */
			GList *s;
			for (s = manager->sessions; s; s = s->next) {
				managerSession *sess = s->data;
				if (sess->session == session) {
					if (sess->link)
						smlLinkDeref(sess->link);
					sess->link = link;
					if (link)
						smlLinkRef(sess->link);
					break;
				}
			}
			
			/* Now check if the header is valid etc */
			if (!smlSessionReceiveHeader(session, header, &locerror))
				goto error_free_header;
			
			/* Then check if we are allowed to authenticate */
			if (!smlManagerDispatchHeader(manager, session, header, cred, &locerror))
				goto error_free_header;
			
			smlHeaderFree(header);
			if (cred)
				smlCredUnref(cred);
			
			/* Now let the session handle the commands etc */
			if (!smlSessionReceiveBody(session, parser, &locerror))
				goto error_free_parser;
			
			/* Free the parser */
			smlParserFree(parser);
			break;
		case SML_TRANSPORT_EVENT_ERROR:
			/* Pass the error through */
			smlErrorDuplicate(&locerror, &error);
			goto transport_error;
			break;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_header:
	if (header)
		smlHeaderFree(header);
	if (cred)
		smlCredUnref(cred);
error_free_parser:
	smlParserFree(parser);
session_error:
	_smlManagerSendEvent(manager, SML_MANAGER_SESSION_ERROR, session, NULL, NULL, locerror);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&locerror));
	smlErrorDeref(&locerror);
	return FALSE;
	
transport_error:
	_smlManagerSendEvent(manager, SML_MANAGER_TRANSPORT_ERROR, NULL, NULL, NULL, locerror);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&locerror));
	smlErrorDeref(&locerror);
	return FALSE;
}

SmlManager *smlManagerNew(SmlTransport *tsp, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, tsp, error);
	smlAssert(tsp);		
	
	SmlManager *manager = smlTryMalloc0(sizeof(SmlManager), error);
	if (!manager)
		goto error;
		
	smlTransportSetEventCallback(tsp, _smlManagerDataHandler, manager);
	manager->transport = tsp;
	
	manager->context = g_main_context_new();
	if (tsp->context == NULL &&
            // tsp->type != SML_TRANSPORT_HTTP_SERVER &&
	    tsp->type != SML_TRANSPORT_HTTP_CLIENT)
	{
		// if the context is already set
		// then the transport is already running in asynchronous mode
		tsp->context = manager->context;
		g_main_context_ref(tsp->context);
	}
	
	manager->functions = smlTryMalloc0(sizeof(GSourceFuncs), error);
	if (!manager->functions)
		goto error_free_manager;
	
	manager->running_mutex = g_mutex_new();
	manager->running = g_cond_new();
	
	manager->userEventQueue = smlQueueNew(error);
	if (!manager->userEventQueue)
		goto error_free_manager;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return manager;

error_free_manager:
	g_free(manager);	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

void smlManagerFree(SmlManager *manager)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, manager);
	smlAssert(manager);
	smlAssert(manager->transport);
	
	while (manager->sessions) {
		managerSession *sess = manager->sessions->data;
		
		_manager_session_free(sess);
	
		manager->sessions = g_list_delete_link(manager->sessions, manager->sessions);
	}
		
	while (manager->objects) {
		SmlObject *object = manager->objects->data;
		
		smlManagerObjectFree(object);
		manager->objects = g_list_remove(manager->objects, object);
	}
	
	if (manager->userEventQueue) {
		SmlManagerEvent *event = NULL;
		while ((event = smlQueueTryPop(manager->userEventQueue)))
			_smlManagerEventFree(event);
		
		smlQueueFree(manager->userEventQueue);
		manager->userEventQueue = NULL;
	}

	if (manager->functions)
		g_free(manager->functions);
		
	if (manager->thread)
		smlThreadFree(manager->thread);
	
	if (manager->source)
		g_source_unref(manager->source);
	
	g_main_context_unref(manager->context);
	
	g_cond_free(manager->running);
	g_mutex_free(manager->running_mutex);
	
	g_free(manager);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlManagerSetEventCallback(SmlManager *manager, SmlManagerEventCb callback, void *userdata)
{
	smlAssert(manager);
	smlAssert(callback);
	
	manager->eventCallback = callback;
	manager->eventCallbackUserdata = userdata;
}

SmlBool smlManagerStart(SmlManager *manager, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, manager, error);
	smlAssert(manager);
    
	manager->functions->prepare = _manager_prepare_internal;
	manager->functions->check = _manager_check_internal;
	manager->functions->dispatch = _manager_dispatch_internal;
	manager->functions->finalize = NULL;
	
	manager->thread = smlThreadNew(manager->context, error);
	if (!manager->thread)
		goto error;
	smlThreadStart(manager->thread);
	
	manager->source = g_source_new(manager->functions, sizeof(GSource) + sizeof(SmlManager *));
	SmlManager **managerptr = (SmlManager **)(manager->source + 1);
	*managerptr = manager;
	g_source_set_callback(manager->source, NULL, manager, NULL);
	g_source_attach(manager->source, manager->context);
	g_main_context_ref(manager->context);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlManagerStop(SmlManager *manager)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, manager);
	smlAssert(manager);
	
	smlThreadStop(manager->thread);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}


void smlManagerRun(SmlManager *manager)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, manager);
	smlAssert(manager);
	
	g_mutex_lock(manager->running_mutex);
	g_cond_wait(manager->running, manager->running_mutex);
	g_mutex_unlock(manager->running_mutex);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlManagerQuit(SmlManager *manager)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, manager);
	smlAssert(manager);
	
	g_mutex_lock(manager->running_mutex);
	g_cond_signal(manager->running);
	g_mutex_unlock(manager->running_mutex);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlSession *smlManagerSessionFind(SmlManager *manager, const char *sessionID)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %s)", __func__, manager, sessionID);
	GList *s = NULL;
	for (s = manager->sessions; s; s = s->next) {
		managerSession *session = s->data;
		if (!strcmp(smlSessionGetSessionID(session->session), sessionID)) {
			smlTrace(TRACE_EXIT, "%s: FOUND %p", __func__, session);
			return session->session;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s: NOT FOUND", __func__);
	return NULL;
}

void smlManagerDispatch(SmlManager *manager)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, manager);
	smlAssert(manager);
	SmlManagerEvent *event = NULL;
	while ((event = smlQueueTryPop(manager->userEventQueue))) {
		smlAssert(manager->eventCallback);
		manager->eventCallback(manager, event->type, event->session, event->error, manager->eventCallbackUserdata);
		_smlManagerEventFree(event);
	}
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlBool smlManagerCheck(SmlManager *manager)
{
	return smlQueueCheck(manager->userEventQueue);
}

static void _event_callback(SmlSession *session, SmlSessionEventType type, SmlCommand *command, SmlCommand *parent, SmlStatus *reply, SmlError *error, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p, %p, %p, %p)", __func__, session, type, command, parent, reply, error, userdata);
	SmlManager *manager = userdata;
	SmlError *locerror = NULL;
	
	switch (type) {
		case SML_SESSION_EVENT_HEADER_REPLY:
			break;
		case SML_SESSION_EVENT_COMMAND:
			if (!smlManagerDispatchCommand(manager, session, command, &locerror)) {
				_smlManagerSendEvent(manager, SML_MANAGER_SESSION_WARNING, session, NULL, NULL, locerror);
				goto error;
			}
			break;
		case SML_SESSION_EVENT_CHILD_COMMAND:
			if (!smlManagerDispatchChildCommand(manager, session, parent, command, &locerror)) {
				_smlManagerSendEvent(manager, SML_MANAGER_SESSION_WARNING, session, NULL, NULL, locerror);
				goto error;
			}
			break;
		case SML_SESSION_EVENT_FINAL:
			/* Pass the final through */
			_smlManagerSendEvent(manager, SML_MANAGER_SESSION_FINAL, session, NULL, NULL, NULL);
			break;
		case SML_SESSION_EVENT_END:
			/* Pass the end through */
			_smlManagerSendEvent(manager, SML_MANAGER_SESSION_END, session, NULL, NULL, NULL);
			break;
		case SML_SESSION_EVENT_FLUSH:
			/* Pass the flush through */
			_smlManagerSendEvent(manager, SML_MANAGER_SESSION_FLUSH, session, NULL, NULL, NULL);
			break;
		case SML_SESSION_EVENT_ERROR:
			/* Pass the error through */
			smlErrorDuplicate(&locerror, &error);
			_smlManagerSendEvent(manager, SML_MANAGER_SESSION_ERROR, session, NULL, NULL, locerror);
			goto error;
			break;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
		
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&locerror));
	smlErrorDeref(&locerror);
}

static void _data_send_callback(SmlSession *session, SmlTransportData *data, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, data, userdata);
	managerSession *sess = userdata;
	
	if (!smlTransportSend(sess->transport, sess->link, data, NULL))
		printf("Error while sending\n");
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlBool smlManagerSessionAdd(SmlManager *manager, SmlSession *session, SmlLink *link, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, manager, session, link, error);
	
	if (smlSessionGetSessionID(session)) {
		//Check if sessionID exists
		smlTrace(TRACE_INTERNAL, "Checking if session ID %s already exists", smlSessionGetSessionID(session));
		
		if (smlManagerSessionFind(manager, smlSessionGetSessionID(session))) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Session ID already exists");
			goto error;
		}
		
		if (atoi(smlSessionGetSessionID(session)) > manager->lastSessionID)
			manager->lastSessionID = atoi(smlSessionGetSessionID(session));
	} else {
		manager->lastSessionID++;
		char *lastid = g_strdup_printf("%i", manager->lastSessionID);
		smlSessionSetSessionID(session, lastid);
		g_free(lastid);
	}
	
	managerSession *sess = smlTryMalloc0(sizeof(managerSession), error);
	if (!sess)
		goto error;
	
	sess->session = session;
	
	if (link) {
		sess->link = link;
		smlLinkRef(link);
	}
	
	sess->transport = manager->transport;
	
	manager->sessions = g_list_append(manager->sessions, sess);
	
	smlSessionSetEventCallback(session, _event_callback, manager);
	smlSessionSetDataCallback(session, _data_send_callback, sess);
	
	_smlManagerSendEvent(manager, SML_MANAGER_SESSION_NEW, session, NULL, NULL, NULL);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
		
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlManagerSessionRemove(SmlManager *manager, SmlSession *session)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, manager, session);
	
	GList *s;
	for (s = manager->sessions; s; s = s->next) {
		managerSession *sess = s->data;
		if (sess->session == session) {
			manager->sessions = g_list_remove(manager->sessions, sess);
			_manager_session_free(sess);
			smlTrace(TRACE_EXIT, "%s", __func__);
			return;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s: Not Found", __func__);
}

/** @brief Register an object with a session
 * 
 * A object waits at a certain location for incoming commands. So you can for example
 * register a object that listens for Get commands on the /test location.
 * 
 * Note that you cannot deregister an object from a already running session
 * 
 * @param session The session on which to listen
 * @param type The command type to listen to or SML_COMMAND_TYPE_UNKNOWN to listen for all
 * @param session If given, only listens for the commands on the given session. NULL for all
 * @param location The location where the object listens (This must match the Target of the command). NULL to listen for all targets
 * @param source The source of the command we want (This must match the Source of the command). NULL to listen for all sources
 * @param callback The callback that will receive the incoming commands.
 * @param childCallback The call that will receive the commands from the child commands. Set to NULL if the expected command has no childs
 * @param userdata The userdata for the callbacks
 * @param error A error struct
 * @return TRUE if successful, FALSE otherwise
 * 
 */
SmlBool smlManagerObjectRegister(SmlManager *manager, SmlCommandType type, SmlSession *session, SmlLocation *location, SmlLocation *source, const char *contentType, SmlCommandCb callback, SmlCommandCb childCallback, void *userdata, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p, %p, %s, %p, %p, %p, %p)", __func__, manager, type, session, location, source, contentType, callback, childCallback, userdata, error);
	smlAssert(manager);
	smlAssert(callback);

	/* first we have to check if this an update for an already registered object */
	/* this is important for actions which require more than one DsSession */
	
        SmlObject *object = smlManagerObjectFindInternal(
				manager, session, type,
				location, source, contentType);
	if (object != NULL && session != NULL)
	{
		/* check that it is not a global manager object */
		SmlObject *hObject = smlManagerObjectFindInternal(
                                	manager, NULL, type,
                                	location, source, contentType);
		if (hObject != NULL && hObject == object)
		{
			/* the found object is a global manager object */
			/* so we have to register a new object */
			object = NULL;
		}
	}
	if (object)
	{
		/* there is an object and so we reconfigure it */
		smlTrace(TRACE_INTERNAL, "prepare a reusable object");

		if (object->location)
			smlLocationUnref(object->location);
		if (object->source)
			smlLocationUnref(object->source);
		if (object->contentType)
			g_free(object->contentType);
	}
	else
	{
		/* let's create and register a new object */
		smlTrace(TRACE_INTERNAL, "create and register a new object");

		object = smlTryMalloc0(sizeof(SmlObject), error);
		if (!object)
			goto error;

		if (session) {
			managerSession *sess = _manager_session_find(manager, session);
			if (!sess) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Session not found");
				goto error_free_object;
			}
		
			sess->objects = g_list_append(sess->objects, object);
		} else {
			manager->objects = g_list_append(manager->objects, object);
		}
	}

	smlTrace(TRACE_INTERNAL, "configure registered object");

	object->type = type;
	
	if (location) {
		object->location = location;
		smlLocationRef(location);
	}
	
	if (source) {
		object->source = source;
		smlLocationRef(source);
	}
	
	if (contentType) {
		object->contentType = g_strdup(contentType);
	}
	
	object->commandCallback = callback;
	object->childCallback = childCallback;
	object->commandCallbackUserdata = userdata;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_object:
	smlManagerObjectFree(object);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlManagerObjectDeregister(SmlManager *manager, SmlCommandType type, SmlLocation *location, SmlLocation *source)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p)", __func__, manager, type, location, source);
	smlAssert(manager);
	
	GList *o2 = g_list_copy(manager->objects);
	GList *o = NULL;
	for (o = o2; o; o = o->next) {
		SmlObject *object = o->data;
		
		
		if (object->type != type)
			continue;
		
		if (!smlLocationCompare(NULL, object->location, NULL, location))
			continue;
		
		if (!smlLocationCompare(NULL, object->source, NULL, source))
			continue;
		
		smlManagerObjectFree(object);
		manager->objects = g_list_remove(manager->objects, object);
	}
	g_list_free(o2);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlManagerObjectFree(SmlObject *object)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, object);
	smlAssert(object);
	
	if (object->location)
		smlLocationUnref(object->location);
		
	if (object->source)
		smlLocationUnref(object->source);
	
	if (object->contentType)
		g_free(object->contentType);
	
	g_free(object);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlObject *smlManagerObjectFind(SmlManager *manager, SmlSession *session, SmlCommand *cmd)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, manager, session, cmd);
	smlAssert(manager);
	smlAssert(cmd);

	SmlObject *object = smlManagerObjectFindInternal(
				manager,
				session,
				cmd->type,
				cmd->target, cmd->source,
				cmd->private.alert.contentType);
	smlTrace(TRACE_EXIT, "%s(%p)", __func__, object);
	return object;
}

SmlObject *smlManagerObjectFindInternal(
		SmlManager *manager,
		SmlSession *session,
		SmlCommandType type,
		SmlLocation *target,
		SmlLocation *source,
		const char* contentType)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %d, %p, %p)", __func__, manager, session, type, target, source);
	smlAssert(manager);
	SmlObject *object = NULL;
	GList *o = NULL;
	
	/* We first search for the object in the session specific objects */
	if (session) {
		managerSession *sess = _manager_session_find(manager, session);
		if (sess) {
			for (o = sess->objects; o; o = o->next) {
				object = o->data;
				
				if (object->type != SML_COMMAND_TYPE_UNKNOWN && type != object->type)
					continue;
				
				if (!smlLocationCompare(NULL, object->location, NULL, target))
					continue;
				
				if (!smlLocationCompare(NULL, object->source, NULL, source))
					continue;
				
				smlTrace(TRACE_EXIT, "%s: FOUND (session): %p", __func__, object);
				return object;
			}
		}
	}
	
	for (o = manager->objects; o; o = o->next) {
		object = o->data;
		
		if (object->type != SML_COMMAND_TYPE_UNKNOWN && type != object->type)
			continue;
		
		if (type == SML_COMMAND_TYPE_ALERT && contentType) {
			if (object->contentType) {
				/* This item is a san 11 alert */
				if (!strcmp(contentType, object->contentType)) {
					smlTrace(TRACE_EXIT, "%s: FOUND SAN TARGET: %p", __func__, object);
					return object;
				}
			}
			
			continue;
		}
		
		if (!smlLocationCompare(NULL, object->location, NULL, target))
			continue;
		
		if (!smlLocationCompare(NULL, object->source, NULL, source))
			continue;
		
		/* Check if the target is a san target */
		if (object->contentType)
			continue;
		
		smlTrace(TRACE_EXIT, "%s: FOUND: %p", __func__, object);
		return object;
	}
	
	smlTrace(TRACE_EXIT, "%s: NOT FOUND", __func__);
	return NULL;
}

SmlTransport *smlManagerGetTransport(SmlManager *manager)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, manager);
	smlAssert(manager);
	
	SmlTransport *tsp = manager->transport;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, tsp);
	return tsp;
}

SmlBool smlManagerDispatchHeader(SmlManager *manager, SmlSession *session, SmlHeader *header, SmlCred *cred, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p, %p)", __func__, manager, session, header, cred, error);
	smlAssert(manager);
	smlAssert(session);
	smlAssert(header);
	
	//If not send command to default handler
	if (manager->headerCallback) {
		manager->headerCallback(session, header, cred, manager->headerCallbackUserdata);
	} else {
		smlTrace(TRACE_INTERNAL, "Header not handled!");
		SmlStatus *status = smlStatusNew(SML_ERROR_NOT_SUPPORTED, 0, header->messageID, header->source, header->target, SML_COMMAND_TYPE_HEADER, error);
		if (!status)
			goto error;
			
		if (!smlSessionSendReply(session, status, error)) {
			smlStatusUnref(status);
			goto error;
		}
		
		smlStatusUnref(status);
	}

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlManagerDispatchChildCommand(SmlManager *manager, SmlSession *session, SmlCommand *parent, SmlCommand *cmd, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p, %p)", __func__, manager, session, parent, cmd, error);
	smlAssert(manager);
	smlAssert(session);
	smlAssert(parent);
	
	//Check if a handler for this object at this path has been installed.
	SmlObject *object = smlManagerObjectFind(manager, session, parent);
	if (object) {
		//Check if a handler for this object at this path has been installed.
		if (!object->childCallback) {
			smlErrorSet(error, SML_ERROR_GENERIC, "No handler for the child was installed");
			goto error;
		}
		
		object->childCallback(session, cmd, object->commandCallbackUserdata);
	} else {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to find child command handler");
		
		SmlStatus *reply = smlCommandNewReply(cmd, SML_ERROR_NOT_FOUND, error);
		if (!reply)
			goto error;
		
		if (!smlSessionSendReply(session, reply, error)) {
			smlStatusUnref(reply);
			goto error;
		}
		
		smlStatusUnref(reply);
		
		goto error;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlManagerDispatchCommand(SmlManager *manager, SmlSession *session, SmlCommand *cmd, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, manager, session, cmd, error);
	smlAssert(manager);
	smlAssert(session);
	smlAssert(cmd);
	
	//Check if a handler for this object at this path has been installed.
	SmlObject *object = smlManagerObjectFind(manager, session, cmd);
	if (object) {
		//Check if a handler for this object at this path has been installed.
		if (!object->commandCallback) {
			smlErrorSet(error, SML_ERROR_GENERIC, "No handler for the child was installed");
			goto error;
		}
		
		object->commandCallback(session, cmd, object->commandCallbackUserdata);
	} else {
		const char *type = smlCommandTypeToString(cmd->type, NULL);
		const char *srcuri = (cmd->source && cmd->source->locURI) ? cmd->source->locURI : "NULL";
		const char *dsturi = (cmd->target && cmd->target->locURI) ? cmd->target->locURI : "NULL";
		smlErrorSet(error, SML_ERROR_NOT_FOUND, "Unable to find command handler (%s: %s -> %s)", (type) ? type : "UNKNOWN", srcuri, dsturi);
		
		SmlStatus *reply = smlCommandNewReply(cmd, SML_ERROR_NOT_FOUND, error);
		if (!reply)
			goto error;
		
		if (!smlSessionSendReply(session, reply, error)) {
			smlStatusUnref(reply);
			goto error;
		}
		
		smlStatusUnref(reply);
		
		goto error;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlManagerRegisterHeaderHandler(SmlManager *manager, SmlHeaderCb callback, SmlStatusReplyCb statuscb, void *userdata)
{
	smlAssert(manager);

	manager->headerCallback = callback;
	manager->headerStatusCallback = statuscb;
	manager->headerCallbackUserdata = userdata;
}
