#ifndef IMPLICATIONS_H
#define IMPLICATIONS_H

/*
 * Collection of tag implications and a a TagcollFilter to apply or compress
 * them
 * 
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma interface

#include <TagcollConsumer.h>
#include <TagcollFilter.h>

#include <OpSet.h>

#include <map>
#include <set>
#include <string>

// List of explicit implications that can be used to expand a tagged collection
// while it's being parsed
class ImplicationList : public TagcollConsumer<std::string>
{
protected:
	// DAG arcs: child -> {parents}
	typedef std::map< std::string, OpSet<std::string> > impl_t;
	impl_t implications;

	// Get the set of all tags seen when walking through all parent lists
	OpSet<std::string> getDestinations(const std::string& tag, const OpSet<std::string>& seen = OpSet<std::string>()) const throw ();

	// Return true if tag1 can reach tag2 walking through some path in its parent list
	bool reaches(const std::string& tag1, const std::string& tag2, const OpSet<std::string>& seen = OpSet<std::string>()) const throw ();

public:
	virtual ~ImplicationList() throw () {}

	virtual void consume(const std::string& item) throw () {}
	virtual void consume(const std::string& item, const OpSet<std::string>& tags) throw ();
	
	// Expand a single tag
	OpSet<std::string> expand(const std::string& tag) throw ();

	// Expand a full tagset
	OpSet<std::string> expand(const OpSet<std::string>& tags) throw ();

	// Remove unnecessary arcs from the dag
	void pack() throw ();
	
	// Output the fully expanded implication dag to a TagcollConsumer
	void outputFull(TagcollConsumer<std::string>& consumer) throw ();

	// Output the implication dag to a TagcollConsumer
	void output(TagcollConsumer<std::string>& consumer) throw ();
};

class ApplyImplications : public TagcollFilter<std::string>
{
protected:
	ImplicationList& implications;
	
public:
	ApplyImplications(ImplicationList& implications) throw ()
		: implications(implications) {}

	virtual void consume(const std::string& item) throw ()
	{
		consumer->consume(item);
	}

	virtual void consume(const std::string& item, const OpSet<std::string>& tags) throw ()
	{
		consumer->consume(item, implications.expand(tags));
	}
};

// TagcollConsumer that just outputs the input to a given file handle, removing
// the tags that would be re-added if expanding with the given explicit
// implications
class CompressImplications : public TagcollFilter<std::string>
{
protected:
	ImplicationList& implications;

public:
	CompressImplications(ImplicationList& implications) throw ()
		: implications(implications) {}

	virtual void consume(const std::string& item) throw ()
	{
		consumer->consume(item);
	}
	
	virtual void consume(const std::string& item, const OpSet<std::string>& tags) throw ();
};

// vim:set ts=4 sw=4:
#endif
