#ifndef TAGCOLL_OPSET_H
#define TAGCOLL_OPSET_H

/*
 * std::set with operators overridden with set operations
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

/*
 * The reason for not using a plain std::set is that libtagcoll involves a lot
 * of set operations, and overridden operators greatly help having cleaner code
 */

/* TODO: replace + with | and ^ with &, since logical operators better
 * correspond to set operations */

#pragma interface

#include <set>

namespace Tagcoll
{

/**
 * OpSet is the standard std::set extended with set operations.
 * Example:
 * \code
 *  OpSet<string> myfavs;
 *  OpSet<string> yourfavs;
 *  myfavourite += "pear";
 *  myfavourite += "banana";
 *  yourfavourite += "apple";
 *  yourfavourite += "pear";
 *  OpSet<string> ourfavs = myfavs ^ yourfavs;
 *  OpSet<string> interesting = myfavs + yourfavs;
 *  OpSet<string> myonlyfavs = myfavs - yourfavs;
 *  for (OpSet<string>::const_iterator i = ourfavs.begin();
 *       i != ourfavs.end(); i++)
 *     cout << *i << endl;
 * \endcode
 */
template<class T>
class OpSet : public std::set<T>
{
public:
	using std::set<T>::begin;
	using std::set<T>::end;

	// Return true if the tag set contains tag, else false
	bool contains(const T& item) const throw () { return find(item) != end(); }

	// Return true if the tag set contains ts, else false
	bool contains(const OpSet<T>& ts) const throw ();

	// Find the distance between two tagsets
	// The distance between A and B is defined by infinity if the intersection
	// between A and B is empty, else it is | (A + B) - (A ^ B) |
	// That is, the cardinality of ((A <union> B) <difference> (A <intersection> B))
	// Return -1 if the distance is infinity
	int distance(const OpSet<T>& ts) const throw ();

	/**
	 * Singleton union
	 * @return the set union of this set and the singleton set {tag}
	 */
	OpSet<T> operator+(const T& tag) const throw ();

	/**
	 * Singleton union
	 * @return the set union of this set and the singleton set {tag}
	 */
	OpSet<T>& operator+=(const T& ts) throw ();

	/**
	 * Set union
	 * @return the set union of this set and the set ts
	 */
	OpSet<T> operator+(const OpSet<T>& ts) const throw ();

	/**
	 * Singleton union
	 * @return the set union of this set and the singleton set {tag}
	 */
	OpSet<T>& operator+=(const OpSet<T>& ts) throw ();

	/**
	 * Singleton difference
	 * @return the set difference of this set and the singleton set {tag}
	 */
	OpSet<T> operator-(const T& tag) const throw ();

	/**
	 * Singleton difference
	 * @return the set difference of this set and the singleton set {tag}
	 */
	OpSet<T>& operator-=(const T& tag) throw ();

	/**
	 * Set difference
	 * @return the set difference of this set and the set ts
	 */
	OpSet<T> operator-(const OpSet<T>& ts) const throw ();

	/**
	 * Set difference
	 * @return the set difference of this set and the set ts
	 */
	OpSet<T>& operator-=(const OpSet<T>& ts) throw ();

	/**
	 * Set intersection
	 * @return the set intersection of this set and the set ts
	 */
	OpSet<T> operator^(const OpSet<T>& ts) const throw ();

	/**
	 * Set intersection
	 * @return the set intersection of this set and the set ts
	 */
	OpSet<T>& operator^=(const OpSet<T>& ts) throw ();
};

};

// vim:set ts=4 sw=4:
#endif
