#ifndef TAGCOLL_TAGCOLLECTION_H
#define TAGCOLL_TAGCOLLECTION_H

/*
 * Core tagged collection handling
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma interface

#include <string>
#include <set>
#include <map>
#include <list>

#include <tagcoll/OpSet.h>
#include <tagcoll/TagcollConsumer.h>
#include <tagcoll/TagcollChange.h>

// Definition of tag implication:
//   tag1 is implied by tag2 if all items associated to tag2 are also
//   associated to tag1
// Also said:
//   tag1 implies tag2 if all items associated to tag1 are also associated
//   to tag2

namespace Tagcoll
{

// Container for bookmarks stored for computation
template<class ITEM, class TAG>
class TagCollection : public TagcollConsumer<ITEM, TAG>
{
protected:
	class TagContainer : public std::map<TAG, int>
	{
	public:
		void add(const TAG& tag, int card = 1) throw ();
		void del(const TAG& tag, int card = 1) throw ();
	};

	// Tags in this collection, with their cardinality
	TagContainer tags;
	
	// Tag sets in this collection, with their cardinality
	typedef std::map<OpSet<TAG>, OpSet<ITEM> > tagsets_t;
	tagsets_t tagsets;

	// Items that have no tags in this collection
	OpSet<ITEM> untagged;

	// Get the list of tags that imply one of the tags in `tags'
	OpSet<TAG> getImplyingOneOf(const OpSet<TAG>& tags) const throw ();
	
public:
	// Represent a change to the TagCollection
	class Change : public std::map< ITEM, OpSet<TAG> > {};
	
	TagCollection() throw () {}

	TagCollection(const TagCollection& tc) throw ()
		: tags(tc.tags), tagsets(tc.tagsets), untagged(tc.untagged) {}

	virtual ~TagCollection() throw () {}

	TagCollection& operator=(const TagCollection& tc) throw ()
	{
		tags = tc.tags;
		tagsets = tc.tagsets;
		untagged = tc.untagged;
		return *this;
	}

	// Iterators support
	typedef typename tagsets_t::const_iterator const_iterator;
	typedef typename tagsets_t::iterator iterator;

	iterator begin() throw () { return tagsets.begin(); }
	iterator end() throw () { return tagsets.end(); }
	const_iterator begin() const throw () { return tagsets.begin(); }
	const_iterator end() const throw () { return tagsets.end(); }

	// Get the number of different tags in this collection
	int tagCount() const throw () { return tags.size(); }

	// Get the number of different tag sets in this collection
	int tagsetCount() const throw () { return tagsets.size(); }

	// Get the number of untagged items in this collection
	int untaggedCount() const throw () { return untagged.size(); }

	// Get the total number of items in this collection
	int totalCount() const throw ();

	// Get the set of untagged items in this collection
	OpSet<ITEM> getUntaggedItems() const throw () { return untagged; }

	// Get the set of items with the given tagset
	OpSet<ITEM> getItemsForTagset(const OpSet<TAG>& ts) const throw ();

	// Get the set of tags for the given item
	// Warning: it iterates over all tagsets to find out which one is attached
	// to the given item
	OpSet<TAG> getTagsetForItem(const ITEM& item) const throw ();

	// Get the set of all tags in this collection
	OpSet<TAG> getAllTags() const throw ();

	// Get the set of all tags in this collection that appear in tagsets
	// containing `ts'
	OpSet<TAG> getCompanionTags(const OpSet<TAG>& ts) const throw ();

	// Get the set of all items in this collection
	OpSet<ITEM> getAllItems() const throw ();

	// Get the set of all items in this collection whose tagsets contain `ts'
	OpSet<ITEM> getCompanionItems(const OpSet<TAG>& ts) const throw ();

	// Get the set of all items in this collection whose tagsets contain `ts'
	std::map< ITEM, OpSet<TAG> > getCompanionItemsAndTagsets(const OpSet<TAG>& ts) const throw ();

	// Get the list of tagsets related to the given one, with distance > 0 and <= maxdistance
	std::list< OpSet<TAG> > getRelatedTagsets(const OpSet<TAG>& ts, int maxdistance = 1) const throw ();

	// Apply a Change to the collection; return a reverse change that can be
	// reused to undo the operation
	Change applyChange(const Change& change) throw ();
	TagcollChange<ITEM, TAG> applyChange(const TagcollChange<ITEM, TAG>& change) throw ();

	// Add an untagged item to the collection
	void add(const ITEM& item) throw ();

	// Add a set of untagged items to the collection
	void add(const OpSet<ITEM>& items) throw ();

	// Add an item with the given tagset to the tagged collection
	void add(const OpSet<TAG>& tagset, const ITEM& item) throw ();

	// Add a set of items with the given tagset to the tagged collection
	void add(const OpSet<TAG>& tagset, const OpSet<ITEM>& items) throw ();

	// Add an untagged item to the collection
	void consume(const ITEM& item) throw () { add(item); }

	// Add a set of untagged items to the collection
	void consume(const OpSet<ITEM>& items) throw () { add(items); }

	// Add an item with the given tagset to the tagged collection
	void consume(const ITEM& item, const OpSet<TAG>& tagset) throw () { add(tagset, item); }

	// Add a set of items with the given tagset to the tagged collection
	void consume(const OpSet<ITEM>& items, const OpSet<TAG>& tagset) throw () { add(tagset, items); }

	// Return a tagged collection with all tagsets of this one that contain the
	// tag `tag', but with the tag removed
	TagCollection<ITEM, TAG> getChildCollection(const TAG& tag) const throw ();

	// Return a tagged collection with all tagsets of this one that are
	// nonempty when stripped by the tag `tag' and all tags that imply it 
	TagCollection<ITEM, TAG> getCollectionWithoutTags(const OpSet<TAG>& tag) const throw ();

	// Return the tagged collection with all tagsets of this one that do not
	// contain the tag `tag'
	TagCollection<ITEM, TAG> getCollectionWithoutTagsetsHaving(const TAG& tag) const throw ();

	// Return the tagged collection with all tagsets of this one that do not
	// contain the tags in `tags'
	TagCollection<ITEM, TAG> getCollectionWithoutTagsetsHavingAnyOf(const OpSet<TAG>& tag) const throw ();

	// Return the tag with maximum cardinality
	TAG findTagWithMaxCardinalityNotIn(const OpSet<TAG>& tags, int* card = 0) const throw ();
		
	// Return the list of tags that the given tag implies
	OpSet<TAG> getImpliedBy(const TAG& tag) const throw ();
	
	// Return a collection where equivalent tags are merged.
	// Equivalent tags are tags which are attached to the same set of items
	// Merging two equivalent tags A and B is done renaming both of them in the
	// tag "A, B"
	void mergeEquivalentTags() throw ();

	// Remove all the tags with cardinality less than `card'
	void removeTagsWithCardinalityLessThan(int card) throw ();
	
	// Output the contents of the collection to a TagcollConsumer
	void output(TagcollConsumer<ITEM, TAG>& cons) const throw ();
};

};

// vim:set ts=4 sw=4:
#endif
