/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*
 * TODO: 1. NO PEGAR O DOBRO DE REGISTROS NA VOLTA (MOVE PREVIOUS) NO FUTURO!!!!
 *
 */

#include "TePGInterface.h"
#include "TePGUtils.h"
#include <stdexcept>
#include <string>



#include "TeUtils.h"

/*
 *	Read/write mode flags for inversion (large object) calls
 */

#define INV_WRITE		0x00020000
#define INV_READ		0x00040000

/*
 * Read buffer size
 */
#define BUFFSIZE        1024

/*
 * PostgreSQL Geometric Types
 */
typedef struct
{
	double		x, y;
} BIN_PG_POINT;

using namespace std;

/*******************************
 * FUNES AUXILIARES          *
 *******************************/
// This function is the same parse_hex function in PostGIS rc3 source code.
inline unsigned char parse_hex(char *str)
{
	//do this a little brute force to make it faster

	unsigned char		result_high = 0;
	unsigned char		result_low = 0;

	switch (str[0])
	{
		case '0' :
			result_high = 0;
			break;
		case '1' :
			result_high = 1;
			break;
		case '2' :
			result_high = 2;
			break;
		case '3' :
			result_high = 3;
			break;
		case '4' :
			result_high = 4;
			break;
		case '5' :
			result_high = 5;
			break;
		case '6' :
			result_high = 6;
			break;
		case '7' :
			result_high = 7;
			break;
		case '8' :
			result_high = 8;
			break;
		case '9' :
			result_high = 9;
			break;
		case 'A' :
			result_high = 10;
			break;
		case 'B' :
			result_high = 11;
			break;
		case 'C' :
			result_high = 12;
			break;
		case 'D' :
			result_high = 13;
			break;
		case 'E' :
			result_high = 14;
			break;
		case 'F' :
			result_high = 15;
			break;
	}
	switch (str[1])
	{
		case '0' :
			result_low = 0;
			break;
		case '1' :
			result_low = 1;
			break;
		case '2' :
			result_low = 2;
			break;
		case '3' :
			result_low = 3;
			break;
		case '4' :
			result_low = 4;
			break;
		case '5' :
			result_low = 5;
			break;
		case '6' :
			result_low = 6;
			break;
		case '7' :
			result_low = 7;
			break;
		case '8' :
			result_low = 8;
			break;
		case '9' :
			result_low = 9;
			break;
		case 'A' :
			result_low = 10;
			break;
		case 'B' :
			result_low = 11;
			break;
		case 'C' :
			result_low = 12;
			break;
		case 'D' :
			result_low = 13;
			break;
		case 'E' :
			result_low = 14;
			break;
		case 'F' :
			result_low = 15;
			break;
	}
	return (unsigned char) ((result_high<<4) + result_low);
}


/*******************************
 * Definicao de TePGConnection *
 *******************************/
bool TePGConnection::open(const string& str_connection)
{
	close();

	pg_connection_ = PQconnectdb(str_connection.c_str());

	if(this->state())
	{
		int_connection_ = 1;
		return true;
	}
	else
		return false;
}

void TePGConnection::close(void)
{
 if(int_connection_)
   {
    if(pg_connection_)
      {
       PQfinish(pg_connection_);
      }
    pg_connection_ = 0;
    int_connection_ = 0;
   }
 return;
}


int TePGConnection::exec_cmd(const string& str_qry)
{
 PGresult *result_temp;
 int resp = -1;
 result_temp = PQexec(pg_connection_, str_qry.c_str());

    switch (PQresultStatus(result_temp))
	{
		case PGRES_EMPTY_QUERY: // The string sent to the backend was empty.
				break;
		case PGRES_COMMAND_OK: // Successful completion of a command returning no data
			    resp = atoi(PQcmdTuples(result_temp));
				last_oid_ = PQoidValue(result_temp);
				break;
		case PGRES_TUPLES_OK: // The query successfully executed
				break;

		case PGRES_COPY_OUT: // Copy Out (from server) data transfer started
		case PGRES_COPY_IN: // Copy In (to server) data transfer started
				break;

		case PGRES_BAD_RESPONSE: // The server's response was not understood
		case PGRES_NONFATAL_ERROR: // TODO: Is this one really an error?
		case PGRES_FATAL_ERROR:
			throw runtime_error("The server's response was not understood");

		default:
			throw logic_error("Internal lib TePGInterface error: ");
	}

 PQclear(result_temp);
 result_temp = 0;
 return resp;
}

Oid TePGConnection::insertBlob(char* buff, const int& blobSize)
{
	if(!buff)
		return 0;

	Oid obj_oid = lo_creat(pg_connection_, INV_READ | INV_WRITE);

	int fd  = lo_open(pg_connection_, obj_oid, INV_READ);

	if(fd < 0)
		return 0;

	int nbwrite = 0;

	int size_tmp = blobSize;

	while(size_tmp > 0)
	{
		if((size_tmp - BUFFSIZE) >= 0)
			nbwrite = lo_write(pg_connection_, fd, buff, BUFFSIZE);				
		else
			nbwrite = lo_write(pg_connection_, fd, buff, size_tmp);

		if(nbwrite <= 0)
			return 0;

		buff += nbwrite;
		size_tmp -= nbwrite;
	}

	if(lo_close(pg_connection_, fd) == 0)
		return obj_oid;

	return 0;
}

int TePGConnection::getBlob(const Oid& blobId, char *&buff)
{
	int fd = lo_open(pg_connection_, blobId, INV_READ);

	if(fd < 0)
		return -1;

	int len = lo_lseek(pg_connection_, fd, 0, SEEK_END);

	lo_lseek(pg_connection_, fd, 0, SEEK_SET);

	if(len <= 0)
		return len;

	if(!buff)
		buff = new char[len];

	char* buffAux = buff;

	int nbread = 0;

	while((nbread = lo_read(pg_connection_, fd, buffAux, BUFFSIZE)) > 0)
		buffAux += nbread;

	return len;
}

bool TePGConnection::beginTransaction(void)
{
	if(inTransaction_)
		return true;

	if(exec_cmd("BEGIN TRANSACTION") == 0)
	{
		inTransaction_ = true;

		return true;
	}

	return false;
}


bool TePGConnection::commitTransaction(void)
{
	if(exec_cmd("COMMIT TRANSACTION") == 0)
	{
		inTransaction_ = false;

		return true;
	}

	return false;
}

bool TePGConnection::rollBackTransaction(void)
{
	if(exec_cmd("ROLLBACK TRANSACTION") == 0)
	{
		inTransaction_ = false;

		return true;
	}

	return false;
}


/******************************
 * Definicao de TePGRecordset *
 ******************************/
const int TePGRecordset::int_bof_ = -1;
int TePGRecordset::openRecordSets_ = 0;
set<int> TePGRecordset::freeCursorNumbers_;
//const int TePGRecordset::numRowsToRetrieve_ = 2000;

TePGRecordset::TePGRecordset(const string& str_qry, TePGConnection* con_x,
			                 const TeCursorType& cursorType,
				             const TeCursorLocation& cursorLocation,
				             const TeCursorDataType& cursorDataType,
					         const int& numRowsToRetrieve)
 : pg_recordset_(0), conn_(con_x), cursorDeclared_(false), int_index_current_(-1),
   int_eof_(-1), cursorType_(cursorType), cursorLocation_(cursorLocation), 
   cursorDataType_(cursorDataType), numRowsToRetrieve_(numRowsToRetrieve)
{
	// endianness test
	if(((unsigned int) 1) == htonl((unsigned int) 1))
		isLittleEndian_ = false;	
	else
		isLittleEndian_ = true;

	if(freeCursorNumbers_.empty())
	{	// Se no houver cursores livres aberto => criar novo id
		++openRecordSets_;
		recordSetId_ = openRecordSets_;
	}
	else
	{
		set<int>::iterator it = freeCursorNumbers_.begin();
		recordSetId_ = *it;
		freeCursorNumbers_.erase(it);
	}

	if(!str_qry.empty() && conn_)
		open(str_qry, conn_, cursorType_, cursorLocation_, cursorDataType_);
}

TePGRecordset::~TePGRecordset()
{	
	// Deixa o id livre
	freeCursorNumbers_.insert(recordSetId_);

	close();
}

void TePGRecordset::close(void)
{
	if(cursorDeclared_ && conn_)
	{
		string sqlCloseCursor  = "CLOSE CURSOR_";
	           sqlCloseCursor += Te2String(recordSetId_);

		conn_->exec_cmd(sqlCloseCursor.c_str());
	}	

	freeRecordSet();

	cursorDeclared_ = false;

	conn_ = 0;

	return;
}

bool TePGRecordset::moveFirst(void)
{
	if(!cursorDeclared_)
		return false;

	if(cursorLocation_ == TeCLIENTESIDE)
	{
		if(recordCount() > 0)
		{
			int_index_current_ = 0;
			return true;
		}
		else
			return false;
	}

	firstFetch = false;
	lastFetch = false;

	string moveFirstSQL  = "MOVE ABSOLUTE 0 IN CURSOR_";
		   moveFirstSQL += Te2String(recordSetId_);

	int res = conn_->exec_cmd(moveFirstSQL.c_str());

	if(res == 0)
	{
		freeRecordSet();

		string fetchSQL  = "FETCH FORWARD " + Te2String(numRowsToRetrieve_);
	           fetchSQL += " FROM CURSOR_";
		       fetchSQL += Te2String(recordSetId_);

		pg_recordset_ = conn_->exec(fetchSQL.c_str());

		if(PQresultStatus(pg_recordset_) == PGRES_TUPLES_OK)
		{
			if(recordCount() > 0)
			{
				int_eof_ = recordCount();
				int_index_current_ = 0;
				return true;
			}
		}
		
		freeRecordSet();
		return false;
	}
 
	return false;
}

bool TePGRecordset::movePrevious(void)
{	
	//if(!cursorDeclared_ || (cursorType_ == TeUNIDIRECTIONAL))
	if(!cursorDeclared_)
		return false;

	if(cursorLocation_ == TeCLIENTESIDE)
	{
		if(bof())
			return false;
		else
		{
			--int_index_current_;
			return true;
		}
	}

	if(recordCount() > 0)
	{
		string move = "";
		string fetchSQL = "";

		if(!movingBackward_)
		{
			 if(bof())
				 return false;
		
			--int_index_current_;

			if(bof())
			{
				movingBackward_ = true;

				int val = recordCount() - 1;

				if(lastFetch)
				{
					++val;
				}

				move = " MOVE BACKWARD "+Te2String(val)+" FROM CURSOR_"+Te2String(recordSetId_)+";";

				fetchSQL = move;
				fetchSQL += " FETCH BACKWARD ";
				fetchSQL += Te2String(numRowsToRetrieve_);
				fetchSQL += " FROM CURSOR_";
				fetchSQL += Te2String(recordSetId_);
			}
			else
			{
				return true;
			}
		}
		else
		{
			if(eof())
				 return false;
		
			++int_index_current_;

			if(eof())
			{	
				fetchSQL  = "FETCH BACKWARD ";
				fetchSQL += Te2String(numRowsToRetrieve_);
				fetchSQL += " FROM CURSOR_";
				fetchSQL += Te2String(recordSetId_);
			}
			else
			{
				return true;
			}
		}
		pg_recordset_ = conn_->exec(fetchSQL.c_str());

		if(PQresultStatus(pg_recordset_) == PGRES_TUPLES_OK)
		{			
			firstFetch = false;
			lastFetch = false;
			
			if(this->recordCount() > 0)
			{
				int_eof_ = recordCount();
				int_index_current_ = 0;

				if(this->recordCount() > 0 != numRowsToRetrieve_)
				{
					firstFetch = true;
				}


				return true;
			}
			
		}

		freeRecordSet();
 	}

	return false;
}

bool TePGRecordset::moveNext(void)
{
	if(!cursorDeclared_)
		return false;

	if(cursorLocation_ == TeCLIENTESIDE)
	{
		if(eof())
			return false;

		++int_index_current_;

		return true;
	}

	if((recordCount() > 0))
	{
		string move = "";
		string fetchSQL = "";

		if(!movingBackward_)
		{	
			if(!eof())
			{
				++int_index_current_;

				if(eof())
				{
					fetchSQL  = "FETCH FORWARD ";
					fetchSQL += Te2String(numRowsToRetrieve_);
					fetchSQL += " FROM CURSOR_";
					fetchSQL += Te2String(recordSetId_);
				}
				else
				{
					return true;
				}
			}
		}
		else
		{
			if(!bof())
			{
				--int_index_current_;

				if(bof())
				{
					movingBackward_ = false;	
					int val = recordCount() -1;
					if(firstFetch)
					{
						++val;
					}
					move = " MOVE FORWARD "+Te2String(val)+" FROM CURSOR_"+Te2String(recordSetId_)+";";

					fetchSQL = move;

					fetchSQL += "FETCH FORWARD ";
					fetchSQL += Te2String(numRowsToRetrieve_);
					fetchSQL += " FROM CURSOR_";
					fetchSQL += Te2String(recordSetId_);
				}
				else
				{
					return true;
				}
			}
		}
		freeRecordSet();

		pg_recordset_ = conn_->exec(fetchSQL.c_str());

		if(PQresultStatus(pg_recordset_) == PGRES_TUPLES_OK)
		{
			firstFetch = false;
			lastFetch = false;

			if(recordCount() > 0)
			{
				int_eof_ = recordCount();
				int_index_current_ = 0;

				if(this->recordCount() > 0 != numRowsToRetrieve_)
				{
					lastFetch = true;
				}
				
				return true;
			}
		}
		
		freeRecordSet();
	}

	return false;
}

bool TePGRecordset::moveLast(void)
{
	//if(!cursorDeclared_ || (cursorType_ == TeUNIDIRECTIONAL))
	if(!cursorDeclared_)
		return false;

	if(cursorLocation_ == TeCLIENTESIDE)
	{
		if(recordCount() > 0)
		{
			int_index_current_ = recordCount() - 1;
			return true;
		}
		
		return false;
	}

	firstFetch = false;
	lastFetch = false;

	freeRecordSet();

	string fetchSQL  = "FETCH LAST FROM CURSOR_";
		   fetchSQL += Te2String(recordSetId_);

	pg_recordset_ = conn_->exec(fetchSQL.c_str());

	if(PQresultStatus(pg_recordset_) == PGRES_TUPLES_OK)
	{
		if(this->recordCount() > 0)
		{
			int_eof_ = recordCount();
			int_index_current_ = 0;
			return true;
		}		
	}
	
	freeRecordSet();
 
	return false;
	
}

bool TePGRecordset::moveTo(const int& lin_number)
{
	//if(!cursorDeclared_ || (cursorType_ == TeUNIDIRECTIONAL))
	if(!cursorDeclared_)
		return false;

	if(cursorLocation_ == TeCLIENTESIDE)
	{
		if((lin_number > int_bof_) && (lin_number < int_eof_))
		{
			int_index_current_ = lin_number;
			return true;
		}
		
		return false;		
	}

	firstFetch = false;
	lastFetch = false;

	freeRecordSet();

	string fetchSQL  = "FETCH ABSOLUTE ";
		   fetchSQL += Te2String(lin_number);
	       fetchSQL += " FROM CURSOR_";
		   fetchSQL += Te2String(recordSetId_);

	pg_recordset_ = conn_->exec(fetchSQL.c_str());

	if(PQresultStatus(pg_recordset_) == PGRES_TUPLES_OK)
	{
		if(this->recordCount() > 0)
		{
			int_eof_ = recordCount();
			int_index_current_ = 0;
			return true;
		}
	}

	freeRecordSet();
 
	return false;	
}

bool TePGRecordset::open(const string& str_qry, TePGConnection* con_x,
			      const TeCursorType& cursorType,
				  const TeCursorLocation& cursorLocation,
				  const TeCursorDataType& cursorDataType,
				  const int& numRowsToRetrieve)
{
	close();

	if(str_qry.empty() || !con_x)
		return false;

	cursorType_ = cursorType;
	cursorLocation_ = cursorLocation;
	cursorDataType_ = cursorDataType;
	numRowsToRetrieve_ = numRowsToRetrieve;
	cursorDeclared_ = false;
	conn_ = con_x;
	movingBackward_ = false;
	lastFetch = false;
	firstFetch = false;

	string sqlCursor  = "DECLARE CURSOR_";
		   sqlCursor += Te2String(recordSetId_);
		   sqlCursor += (cursorDataType_  == TeBINARYCURSOR) ? " BINARY " : " ";
		   //sqlCursor += (cursorType_ == TeUNIDIRECTIONAL) ? " NO " : " ";
		   sqlCursor += "SCROLL CURSOR WITH HOLD FOR ";
		   sqlCursor += str_qry;	

	pg_recordset_ = conn_->exec(sqlCursor.c_str());

	if(PQresultStatus(pg_recordset_) == PGRES_COMMAND_OK)
	{
		cursorDeclared_ = true;

		string numRows = (cursorLocation_ == TeSERVERSIDE) ? Te2String(numRowsToRetrieve_) : string("ALL");
		
		string fetchSQL  = "FETCH FORWARD " + numRows;
	           fetchSQL += " FROM CURSOR_";
		       fetchSQL += Te2String(recordSetId_);

		pg_recordset_ = conn_->exec(fetchSQL.c_str());

		if(PQresultStatus(pg_recordset_) == PGRES_TUPLES_OK)
		{
			if(recordCount() > 0)
			{
				int_eof_ = recordCount();
				int_index_current_ = 0;				
			}

			return true;
		}
	}
	
	freeRecordSet();

	return false;
}

int TePGRecordset::getBytea(const int& field_num, char*& buff) const
{
	if(cursorDataType_ == TeTEXTCURSOR)
	{
		size_t newLen;

		unsigned char* ptData = PQunescapeBytea((unsigned char*)(value(field_num)), &newLen);

		if(newLen <= 0)
			return newLen;
		
		if(!buff)
			buff = new char[newLen];

		memcpy(buff, ptData, newLen);

		TePGConnection::freeMem(ptData);

		return newLen;
	}
	else	// TeBINARYCURSOR
	{
		char* ptData = value(field_num);

		unsigned int numBytes;
		memcpy(&numBytes, ptData, sizeof(int));
		numBytes = ntohl(numBytes);

		if(!buff)
			buff = new char[numBytes];

		memcpy(buff, ptData + sizeof(int), numBytes);

		return numBytes;
	}
}

void TePGRecordset::getPGLine2D(const int& field_num, TeLine2D& l) const
{
	char* polygon = value(field_num);

	if(cursorDataType_ == TeBINARYCURSOR)
	{
		unsigned int numPts;
		memcpy(&numPts, polygon, sizeof(int));
		numPts = ntohl(numPts);

		l.reserve(numPts);

		// POINT ARRAY IS SUPPOSED ALWAYS IN BIG ENDIAN
		BIN_PG_POINT* pts = (BIN_PG_POINT*)(polygon + sizeof(int));		

		// endianness test
		if(isLittleEndian_)
		{
			union
			{
				double dWord_;
				unsigned int aux_[2];
			} swapx1, swapy1, swapx2, swapy2;

			// little-endian
			for(unsigned int i = 0; i < numPts ; ++i)
			{
				swapx1.dWord_ = pts[i].x;
				swapy1.dWord_ = pts[i].y;

				swapx2.aux_[1] = ntohl(swapx1.aux_[0]);
				swapx2.aux_[0] = ntohl(swapx1.aux_[1]);

				swapy2.aux_[1] = ntohl(swapy1.aux_[0]);
				swapy2.aux_[0] = ntohl(swapy1.aux_[1]);

				l.add(TeCoord2D(swapx2.dWord_, swapy2.dWord_));
			}
			
		}
		else
		{
			// big-endian!
			for(unsigned int i = 0; i < numPts ; ++i)
				l.add(TeCoord2D(pts[i].x, pts[i].y));
			
		}
	}
	else	//TeTEXTCURSOR
		l = PgGeomPolygon2Te(polygon);
}

char* TePGRecordset::getWKBHeader(char* v, unsigned char &byteOrder, unsigned int &wkbType, unsigned int &numGeometries) const
{
	const int byteOrderPlusGeomType = sizeof(unsigned char) + sizeof(unsigned int);	

	if(cursorDataType_ == TeTEXTCURSOR)
	{
		unsigned char header [byteOrderPlusGeomType];

		for (int t = 0; t < byteOrderPlusGeomType; ++t)
		{
			header[t] =(unsigned char) parse_hex(v) ;
			v += 2;
		}

		memcpy(&byteOrder, header, sizeof(unsigned char));		
		memcpy(&wkbType, header + sizeof(unsigned char), sizeof(unsigned int));
	}
	else
	{
		memcpy(&byteOrder, v, sizeof(unsigned char));		
		memcpy(&wkbType, v + sizeof(unsigned char), sizeof(unsigned int));

		v += byteOrderPlusGeomType;
	}	
	

	// 0 = Big Endian (wkbXDR) e 1 = Little Endian (wkbNDR)
	if((byteOrder == 0) && isLittleEndian_)
	{
		wkbType = ntohl(wkbType);
	}
	else if((byteOrder == 1) && !isLittleEndian_)
	{
			wkbType = htonl(wkbType);
	}	

	numGeometries = 0;

	if(wkbType > 1 && wkbType <= 7)
	{
		if(cursorDataType_ == TeTEXTCURSOR)
		{
			unsigned char qtd[sizeof(unsigned int)];

			for(unsigned int t = 0; t < sizeof(unsigned int); ++t)
			{
				qtd[t] = (unsigned char)parse_hex(v);
				v += 2;
			}

			memcpy(&numGeometries, qtd, sizeof(unsigned int));
		}
		else
		{
			memcpy(&numGeometries, v, sizeof(unsigned int));
			v += sizeof(unsigned int);
		}

		// 0 = Big Endian (wkbXDR) e 1 = Little Endian (wkbNDR)
		if((byteOrder == 0) && isLittleEndian_)
		{
			numGeometries = ntohl(numGeometries);	
		}
		else if((byteOrder == 1) && !isLittleEndian_)
		{
			numGeometries = htonl(numGeometries);
		}
	}

	return v;	
}

char* TePGRecordset::getWKBPoint(char* v, TeCoord2D& c) const
{
	unsigned char byteOrder;
	unsigned int wkbType;
	unsigned int numGeometries;

	v = getWKBHeader(v, byteOrder, wkbType, numGeometries);
	
	if(wkbType != 1)
		throw logic_error("Binary data doesn't supported!");
		

	union
	{
		double dWord_;
		unsigned int aux_[2];
	} swapx1, swapy1;

	if(cursorDataType_ == TeTEXTCURSOR)
	{

		const int double2Size = 2 * sizeof(double);
		unsigned char data[double2Size];
		
		for(int t = 0; t < double2Size; ++t) // len/2
		{
			data[t] = (unsigned char)parse_hex(v);
			v += 2;
		}
		

		memcpy(&swapx1.dWord_, data , sizeof(double));
		memcpy(&swapy1.dWord_, data + sizeof(double), sizeof(double));
	}
	else
	{
		memcpy(&swapx1.dWord_, v , sizeof(double));
		memcpy(&swapy1.dWord_, v + sizeof(double), sizeof(double));

		v += (sizeof(double) + sizeof(double)); // x + y
	}

	// 0 = Big Endian (wkbXDR)
	if((byteOrder == 0) && isLittleEndian_)
	{
		union
		{
			double dWord_;
			unsigned int aux_[2];
		} swapx2, swapy2;

		swapx2.aux_[1] = ntohl(swapx1.aux_[0]);
		swapx2.aux_[0] = ntohl(swapx1.aux_[1]);

		swapy2.aux_[1] = ntohl(swapy1.aux_[0]);
		swapy2.aux_[0] = ntohl(swapy1.aux_[1]);	
		
		c.x(swapx2.dWord_);
		c.y(swapy2.dWord_);

		return v;
	}
	else if((byteOrder == 1) && !isLittleEndian_)
	{
		union
		{
			double dWord_;
			unsigned int aux_[2];
		} swapx2, swapy2;

		swapx2.aux_[1] = htonl(swapx1.aux_[0]);
		swapx2.aux_[0] = htonl(swapx1.aux_[1]);

		swapy2.aux_[1] = htonl(swapy1.aux_[0]);
		swapy2.aux_[0] = htonl(swapy1.aux_[1]);
		
		c.x(swapx2.dWord_);
		c.y(swapy2.dWord_);

		return v;

	}

	c.x(swapx1.dWord_);
	c.y(swapy1.dWord_);

	return v;
}

char* TePGRecordset::getWKBLinearRing(char* v, int byteOrder, TeLine2D &line) const
{
	unsigned int numPoints;
	
	const int size2Double = sizeof(double) + sizeof(double); // x + y

	if(cursorDataType_ == TeTEXTCURSOR)
	{
		unsigned char data[size2Double];

		for(unsigned int n = 0; n < sizeof(unsigned int); ++n)
		{
			data[n] = (unsigned char)parse_hex(v);
			v += 2;
		}

		memcpy(&numPoints, data , sizeof(unsigned int));
	}
	else
	{
		memcpy(&numPoints, v , sizeof(unsigned int));
		v += sizeof(unsigned int);
	}

	// 0 = Big Endian (wkbXDR) e 1 = Little Endian (wkbNDR)
	if((byteOrder == 0) && isLittleEndian_)
	{
		numPoints = ntohl(numPoints);	
	}
	else if((byteOrder == 1) && !isLittleEndian_)
	{
		numPoints = htonl(numPoints);
	}

	for(unsigned int i = 0; i < numPoints; ++i)
	{
		union
		{
			double dWord_;
			unsigned int aux_[2];
		} swapx1, swapy1;

		if(cursorDataType_ == TeTEXTCURSOR)
		{
			unsigned char data[size2Double];

			for(int t = 0; t < size2Double; ++t) // len/2
			{
				data[t] = (unsigned char)parse_hex(v);
				v += 2;
			}	

			memcpy(&swapx1.dWord_, data , sizeof(double));
			memcpy(&swapy1.dWord_, data + sizeof(double), sizeof(double));
		}
		else
		{
			memcpy(&swapx1.dWord_, v , sizeof(double));
			memcpy(&swapy1.dWord_, v + sizeof(double), sizeof(double));
			v += size2Double;
		}

		// 0 = Big Endian (wkbXDR)
		if((byteOrder == 0) && isLittleEndian_)
		{
			union
			{
				double dWord_;
				unsigned int aux_[2];
			} swapx2, swapy2;

			swapx2.aux_[1] = ntohl(swapx1.aux_[0]);
			swapx2.aux_[0] = ntohl(swapx1.aux_[1]);

			swapy2.aux_[1] = ntohl(swapy1.aux_[0]);
			swapy2.aux_[0] = ntohl(swapy1.aux_[1]);	
			
			line.add(TeCoord2D(swapx2.dWord_, swapy2.dWord_));			
		}
		else if((byteOrder == 1) && !isLittleEndian_)	//1 = Little Endian (wkbNDR)
		{
			union
			{
				double dWord_;
				unsigned int aux_[2];
			} swapx2, swapy2;

			swapx2.aux_[1] = htonl(swapx1.aux_[0]);
			swapx2.aux_[0] = htonl(swapx1.aux_[1]);

			swapy2.aux_[1] = htonl(swapy1.aux_[0]);
			swapy2.aux_[0] = htonl(swapy1.aux_[1]);
			
			line.add(TeCoord2D(swapx2.dWord_, swapy2.dWord_));
		}
		else
		{
			line.add(TeCoord2D(swapx1.dWord_, swapy1.dWord_));
		}
	}

	return v;

}

char* TePGRecordset::getWKBLine(char* v, TeLine2D& l) const
{
	unsigned char byteOrder;
	unsigned int wkbType;
	unsigned int numPoints;

	v = getWKBHeader(v, byteOrder, wkbType, numPoints);
	
	if(wkbType != 2)
		throw logic_error("Binary data doesn't supported!");

	if(cursorDataType_ == TeTEXTCURSOR)
	{
		// Volta duas vezes o nmero de bytes do nmero de geometrias lidos ao chamar getWKBHeader
		v = getWKBLinearRing(v - (sizeof(unsigned int) + sizeof(unsigned int)), byteOrder, l);
	}
	else
	{
		// Volta o nmero de bytes do nmero de geometrias lidos ao chamar getWKBHeader
		v = getWKBLinearRing(v - sizeof(unsigned int), byteOrder, l);
	}

	return v;

}

char* TePGRecordset::getWKBPolygon(char* v, TePolygon& p) const
{
	unsigned char byteOrder;
	unsigned int wkbType;
	unsigned int numRings;

	v = getWKBHeader(v, byteOrder, wkbType, numRings);
	
	if(wkbType != 3)
		throw logic_error("Binary data doesn't supported!");

	for(unsigned int i = 0; i < numRings; ++i)
	{
		TeLine2D line;
		v = getWKBLinearRing(v, byteOrder, line);
		p.add(line);
	}

	return v;

}

char* TePGRecordset::getWKBMultiPoint(char* v, TePointSet &ps) const
{
	unsigned char byteOrder; 
	unsigned int wkbType; 
	unsigned int num_wkbPoints; 

	v = getWKBHeader(v, byteOrder, wkbType, num_wkbPoints);

	if(wkbType != 4)
		throw logic_error("Binary data doesn't supported!");

	for(unsigned int i = 0; i < num_wkbPoints; ++i)
	{
		TeCoord2D coord;
		v = getWKBPoint(v, coord);
		ps.add(coord);
	}

	return v;
}

char* TePGRecordset::getWKBMultiLine(char* v, TeLineSet &ls) const
{
	unsigned char byteOrder; 
	unsigned int wkbType; 
	unsigned int num_wkbLineStrings; 

	v = getWKBHeader(v, byteOrder, wkbType, num_wkbLineStrings);

	if(wkbType != 5)
		throw logic_error("Binary data doesn't supported!");

	for(unsigned int i = 0; i < num_wkbLineStrings; ++i)
	{
		TeLine2D line;
		v = getWKBLine(v, line);
		ls.add(line);
	}

	return v;
}

char* TePGRecordset::getWKBMultiPolygon(char* v, TePolygonSet &ps) const
{
	unsigned char byteOrder;
	unsigned int wkbType;
	unsigned int num_wkbPolygons;

	v = getWKBHeader(v, byteOrder, wkbType, num_wkbPolygons);

	if(wkbType != 6)
		throw logic_error("Binary data doesn't supported!");

	for(unsigned int i = 0; i < num_wkbPolygons; ++i)
	{
		TePolygon poly;
		v = getWKBPolygon(v, poly);
		ps.add(poly);
	}

	return v;
}

void TePGRecordset::getWKBGeomColl(char* v, TeMultiGeometry &mg) const
{
	unsigned char byteOrder; 
	unsigned int wkbType; 
	unsigned int num_wkbGeometries;

	v = getWKBHeader(v, byteOrder, wkbType, num_wkbGeometries);

	if(wkbType != 7)
		throw logic_error("Binary data doesn't supported!");

	for(unsigned int i = 0; i < num_wkbGeometries; ++i)
	{
		unsigned int geomType;
		unsigned char geomByteOrder;
		unsigned int num_geometries;

		getWKBHeader(v, geomByteOrder, geomType, num_geometries);

		if(geomType == 1)
		{
			TeCoord2D coord;

			v = getWKBPoint(v, coord);

			mg.addGeometry(coord);
		}
		else if(geomType == 2)
		{
			TeLine2D line;

			v = getWKBLine(v, line);

			mg.addGeometry(line);
		}
		else if(geomType == 3)
		{
			TePolygon poly;

			v = getWKBPolygon(v, poly);

			mg.addGeometry(poly);
		}
		else if(geomType == 4)
		{
			TePointSet pointSet;

			v = getWKBMultiPoint(v, pointSet);

			for(unsigned int a = 0; a < pointSet.size(); ++a)
			{
				TePoint point = pointSet[a];
				mg.addGeometry(point);
			}
		}
		else if(geomType == 5)
		{
			TeLineSet lineSet;

			v = getWKBMultiLine(v, lineSet);

			for(unsigned int a = 0; a < lineSet.size(); ++a)
			{
				TeLine2D coord = lineSet[a];

				mg.addGeometry(coord);
			}
		}
		else if(geomType == 6)
		{
			TePolygonSet polygonSet;
			
			v = getWKBMultiPolygon(v, polygonSet);
			
			for(unsigned int a = 0; a < polygonSet.size(); ++a)
			{
				TePolygon poly = polygonSet[a];

				mg.addGeometry(poly);
			}
		}
		else
		{
			throw logic_error("The data couldn't be decoded as a valid WKB geometry!");
		}
	}

}


void TePGRecordset::getPGISPoint(const int& field_num, TePoint& p) const
{
	TeCoord2D c;
	getWKBPoint(value(field_num), c);
	p.add(c);
}


void TePGRecordset::getPGISLine(const int& field_num, TeLine2D& l) const
{
	getWKBLine(value(field_num), l);		
}


void TePGRecordset::getPGISPolygon(const int& field_num, TePolygon& p) const
{
	getWKBPolygon(value(field_num), p);		
}


void TePGRecordset::getPGISMultiPoint(const int& field_num, TePointSet& ps) const
{
	getWKBMultiPoint(value(field_num), ps);		
}


void TePGRecordset::getPGISMultiLine(const int& field_num, TeLineSet& ls) const
{
	getWKBMultiLine(value(field_num), ls);		
}


void TePGRecordset::getPGISMultiPolygon(const int& field_num, TePolygonSet& ps) const
{
	getWKBMultiPolygon(value(field_num), ps);	
}


void TePGRecordset::getPGISGeomColl(const int& field_num, TeMultiGeometry& m) const
{
	getWKBGeomColl(value(field_num), m);
}



char* TePGRecordset::getData(const int& field_num)
{
	data_ = "";

	if(cursorDataType_ == TeTEXTCURSOR)
		return value(field_num);
	else
	{
		// chamar o conversor de tipo de dado de binrio p/ string p/ cada tipo!
		switch(fieldType(field_num))
		{
			case PG_BOOL_TYPE			:
			case PG_BYTEA_TYPE			:
			case PG_CHAR_TYPE			:
			case PG_INT8_TYPE			:
			case PG_INT2_TYPE			: break;

			case PG_INT4_TYPE			: data_ = Te2String(getInt(field_num));
										  break;

			case PG_TEXT_TYPE			:
			case PG_OID_TYPE			:
			case PG_POINT_TYPE			:
			case PG_LSEG_TYPE			:
			case PG_PATH_TYPE			:
			case PG_BOX_TYPE			:
			case PG_POLYGON_TYPE		:
			case PG_LINE_TYPE			:
			case PG_PG_FLOAT4_TYPE		: break;

			case PG_FLOAT8_TYPE			: data_ = Te2String(getDouble(field_num));
										  break;

			case PG_CIRCLE_TYPE			:
			case PG_MONEY_TYPE			:
			case PG_BPCHAR_TYPE			:
			case PG_VARCHAR_TYPE		:
			case PG_DATE_TYPE			:
			case PG_TIME_TYPE			:
			case PG_TIMESTAMP_TYPE		:
			case PG_TIMESTAMPTZ_TYPE	:
			case PG_INTERVAL_TYPE		:
			case PG_TIMETZ_TYPE			:
			case PG_BIT_TYPE			:
			case PG_VARBIT_TYPE			:
			case PG_NUMERIC_TYPE		:
			default						: break;
		}
		
		return (char*)data_.c_str();
	}
}

int TePGRecordset::fieldSize(const int& field_num)
{
 Oid field_t = this->fieldType(field_num);
 Oid field_m = PQfmod(pg_recordset_, field_num);
 PGresult *result_temp;
 char str_int[5];   //integer part if DECIMAL(p,s) or NUMERIC(p,s)
 //int size = -1;
 int size = 0;
 char str_field_t[50];
 char str_field_m[50];
 sprintf(str_field_t,"%d", field_t);
 sprintf(str_field_m,"%d", field_m);
 string str_qry = "SELECT format_type(";
        str_qry += str_field_t;
        str_qry += ",";
        str_qry += str_field_m;
        str_qry += ")";
 switch(field_t)
       {
        case 1043:   //VARCHAR(s) or CHARACTER VARYING(s)
        case 1700:   //NUMERIC(p,s) or DECIMAL(p,s)
                     result_temp = PQexec(conn_->c_ptr(), str_qry.c_str());
                     if(PQresultStatus(result_temp) == PGRES_TUPLES_OK)
                       {
                        char *lin = PQgetvalue(result_temp, 0, 0);  //Don't free lin because the return of PQgetvalue is in the struct PGResult
                        int i = 0;
                        while(*lin != '\0' && *lin != ',')
                             {
                              if(*lin >= '0' && *lin <= '9' && i < 4)
                                {
                                 str_int[i] = *lin;
                                 i++;
                                }
                              lin++;
                             }
                        str_int[i]='\0';
                        size = atoi(str_int);
                       }
                     PQclear(result_temp);
                     break;
        default:     //The size of fixed size or for other variable size -1
                     return PQfsize(pg_recordset_, field_num);
       }
 result_temp = 0;
 return size;
}

int TePGRecordset::fieldSizeFractionaryPart(const int& field_num)
{
 Oid field_t = this->fieldType(field_num);
 Oid field_m = PQfmod(pg_recordset_, field_num);
 PGresult *result_temp;
 char str_frac[5];   //fractionary part if DECIMAL(p,s) or NUMERIC(p,s)
 //int size = -1;
 int size = 0;
 char str_field_t[50];
 char str_field_m[50];
 sprintf(str_field_t,"%d", field_t);
 sprintf(str_field_m,"%d", field_m);
 string str_qry = "SELECT format_type(";
        str_qry += str_field_t;
        str_qry += ",";
        str_qry += str_field_m;
        str_qry += ")";
 switch(field_t)
       {
        case 1700:   //NUMERIC(p,s) or DECIMAL(p,s)
                     result_temp = PQexec(conn_->c_ptr(), str_qry.c_str());
                     if(PQresultStatus(result_temp) == PGRES_TUPLES_OK)
                       {
                        char *lin = PQgetvalue(result_temp, 0, 0);  //Don't free lin because the return of PQgetvalue is in the struct PGResult
                        int i = 0;
                        while(*lin != '\0' && *lin != ',')
                             {
                              lin++;
                             }
                        if(*lin == ',')
                          {
                           lin++;
                          }
                        while(*lin != '\0')
                             {
                              if(*lin >= '0' && *lin <= '9' && i < 4)
                                {
                                 str_frac[i] = *lin;
                                 i++;
                                }
                              lin++;
                             }
                        str_frac[i]='\0';
                        size = atoi(str_frac);
                       }
                     PQclear(result_temp);
                     break;
        default:     //Others doesn't have a fractionary part
                     break;

       }
 result_temp = 0;
 return size;
}

void TePGRecordset::freeRecordSet()
{
	if(pg_recordset_)
		PQclear(pg_recordset_);


	pg_recordset_ = 0;
	int_index_current_ = -1;
	int_eof_ = -1;
	return;
}





