/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeProxMatrixImplementation.h
    This file contains structures and definitions about different representations of proximity matrix 
*/

#ifndef TeProxMatrixImplementation_H
#define TeProxMatrixImplementation_H

#include "TeNeighbours.h"
#include <graph.h>

//! An abstract class to represent proximity matrix   
class TeProxMatrixImplementation  
{

protected:
	
	//! Type of the representation
	string type_;
	
	//! Empty constructor 
	TeProxMatrixImplementation(){}

public:

	//! Verify if two objects are neighbour or connected
	bool isConnected (const string& object_id1, const string& object_id2) 
	{
		TeProxMatrixAttributes attr; 
		return getConnectionAttributes (object_id1, object_id2, attr);
	}

	//! Connect two objects
	virtual void connectObjects (const string& /* object_id1 */, const string& /* object_id2 */, const TeProxMatrixAttributes& /* attr */){}

	//! Disconnect two objects
	virtual bool disconnectObjects (const string& /* object_id1 */, const string& /* object_id2 */) {return false;}

	//! Remove an object
	virtual bool removeObject (const string& /* object_id */)	{return false;}

	//! Get connection attributes
	virtual bool getConnectionAttributes (const string& /* object_id1 */, const string& /* object_id2 */, TeProxMatrixAttributes& /* attr */)	{return false;}

	//! Set connection attributes
	virtual bool setConnectionAttributes (const string& /* object_id1 */, const string& /* object_id2 */, const TeProxMatrixAttributes& /*attr*/) {return false;}

	//! Get the neighbours of an object
	virtual bool getNeighbours (const string& /* object_id */, TeNeighbours& /* neigh */)	{return false;}

	//! Get the obj-th neighbour of an object 
	virtual bool getNeighbours (int /* obj */, string& /* object_id */, TeNeighbours& /* neigh */) {return false;}  	

	//! Get the neighbours of an object
	virtual bool getNeighboursNeighbours (const string& /* object_id */, TeNeighbours& /* neigh */, int /* max_order */ = 2)	{return false;}

	//! Return the number of objects
	virtual int  NumberOfObjects () {return 0;}

	//! Return the type of the representation 
	string Type() {return type_;}

	//! Create a copy
	virtual TeProxMatrixImplementation* CreateCopy (){return 0;}
	
	//! Verify if is equal
	virtual bool IsEqual (TeProxMatrixImplementation& other)
	{
		if (type_ == other.type_) 
			return true; 
		return false;
	}

	//! Equal operator
	virtual bool operator== (const TeProxMatrixImplementation& /* other */) const { return true;}
		
	
	//! Save the proximity matrix in a text file
	virtual bool SaveTextFile (string& /* name */) {return false;}

	//! Save the proximity matrix in a GAL format text file 
	virtual bool SaveGALFile (string& /* name */){return false;}

	//! Save the proximity matrix in a GWT format text file 
	virtual bool SaveGWTFile (string& /* name */){return false;}

	//! List the proximity matrix on command prompt
	virtual void List (){}

	//! Destructor
	virtual ~TeProxMatrixImplementation(){}
};


//! A class to represent proximity matrix utilising the Breymann graph
class TeProxMatrixGraphBreymann : public  TeProxMatrixImplementation
{
private:
	string type_;
	br_stl::Graph<string, TeProxMatrixAttributes> graph_;

	typedef map<string, int> Object_id_map_type;
	Object_id_map_type  map_;

	typedef Object_id_map_type::iterator map_iterator;
	typedef Object_id_map_type::const_iterator map_const_iterator;

	bool getNeighboursNeighbours (const string& object_id, TeNeighbours& neigh, int max_order = 2, int current_order = 1);

public:

	//! Empty constructor
	TeProxMatrixGraphBreymann () : graph_(true) //false means undirected graph, has to be true to allow normalization
	{
		type_ = "Breymann";
	} 
																	 
	//! Copy constructor
	TeProxMatrixGraphBreymann(TeProxMatrixGraphBreymann& imp);

	//! Connect two objects
	virtual void connectObjects (const string& object_id1, const string& object_id2, const TeProxMatrixAttributes& attr);
	
	//! Get connection attributes 
	virtual bool getConnectionAttributes (const string& object_id1, const string& object_id2, TeProxMatrixAttributes& attr);
	
	//! Set connection attributes 
	virtual bool setConnectionAttributes (const string& object_id1, const string& object_id2, const TeProxMatrixAttributes& attr);

	//! Get the neighbours of an object
	virtual bool getNeighbours (const string& object_id, TeNeighbours& neigh);

	//! Get the obj-th neighbour of an object 
	virtual bool getNeighbours (int obj, string& object_id, TeNeighbours& neigh); 
	
	//! Get the neighbours of an object
	virtual bool getNeighboursNeighbours (const string& object_id, TeNeighbours& neigh, int max_order = 2);

	//! Return the number of the objects
	virtual int  NumberOfObjects () {return graph_.size();}

	//! Create a copy
	virtual TeProxMatrixImplementation* CreateCopy ();

	//! Verify if is equal
	virtual bool IsEqual (TeProxMatrixImplementation& other) 
	{
		if (type_ == other.Type()) 
			return (*this == (TeProxMatrixGraphBreymann&)other); 
		return false;
	}

	//! Assignment operator 
	TeProxMatrixGraphBreymann& operator= (TeProxMatrixGraphBreymann& imp); 
	
	//! Destructor
	virtual ~TeProxMatrixGraphBreymann() {}

	//! Save the proximity matrix in a text file 
	virtual bool SaveTextFile (string& name);

	//! Save the proximity matrix in a GAL text file 
	virtual bool SaveGALFile (string& name);

	//! Save the proximity matrix in a GWT text file 
	virtual bool SaveGWTFile (string& name);

	//! List proximity matrix on command prompt
	virtual void List ();
};


//! An abstract factory of proximity matrix representations  
class TeProxMatrixAbstractFactory  
{
public:

	static TeProxMatrixImplementation* MakeConcreteImplementation (const string impl_type = "BREYMANN")
	{
		if (impl_type ==  "Breymann") 
				return new TeProxMatrixGraphBreymann;

		return new TeProxMatrixGraphBreymann;
	}
	
};

#endif
