/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/


#include "TeSTElement.h"
#include "TeQuerier.h"
#include "TeQuerierParams.h"


TeSTElement::TeSTElement (const string& objId) 
{
	object_id_ = objId;
	time_ = TeTimeInterval();
	querier_ = 0;
}

TeSTElement::TeSTElement(TeTheme* theme, const string& objId)
{
	TeQuerierParams params; 
	params.setParams(theme, objId);
	
	object_id_ = objId;
	time_ = TeTimeInterval();
	querier_ = new TeQuerier(params);
}
		
TeSTElement::TeSTElement(const TeSTElement& other) 
{	
	object_id_		= other.object_id_;
	time_			= other.time_;
	instances_		= other.instances_;
	if(other.querier_)
		querier_ = new TeQuerier(other.querier_->params());
	else
		querier_ = 0;
}

TeSTElement::~TeSTElement()
{	
	if(querier_)
		delete (querier_);
}
			
TeSTElement&					
TeSTElement::operator= (const TeSTElement& other)
{	
	if ( this != &other )
	{
		object_id_		= other.object_id_;
		time_			= other.time_;
		instances_		= other.instances_;
		
		if(other.querier_)
			querier_ = new TeQuerier(other.querier_->params());
		else
			querier_ = 0;
	}
	return *this;
}

void 
TeSTElement::querier(TeQuerier* qu) 
{ 
	if(querier_)
		delete (querier_);

	querier_ = qu; 
}

bool 
TeSTElement::insertSTInstance (TeSTInstance& object)
{
	if(object.objectId() != this->objectId())
		return false;

	instances_.push_back (object);
	return true;
}
	
bool	
TeSTElement::build(bool loadGeometries, bool loadAllAttributes, vector<string> attrNames, int slide)
{
	if(!querier_)
		return false;

	TeQuerierParams* param;
	if(loadAllAttributes)
		param = new TeQuerierParams(loadGeometries, loadAllAttributes);
	else
		param = new TeQuerierParams(loadGeometries, attrNames);

	param->setParams(querier_->params().theme(), objectId());

	querier_->refresh(*param);
	delete param;

	return(buildImpl(slide));
}

bool 
TeSTElement::build(TeGroupingAttr& groupAttr, bool loadGeometries, int slide)
{
	if(!querier_)
		return false;

	TeQuerierParams param(loadGeometries, groupAttr); 
	param.setParams(querier_->params().theme(), objectId());

	querier_->refresh(param);

	return(buildImpl(slide));
}

bool 
TeSTElement::buildImpl(int slide)
{
	int dt = CLOCKS_PER_SEC/4, steps = 0;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;

	if(!querier_->loadInstances(slide))
		return false;

	int tot = querier_->numElemInstances();
	if(TeProgress::instance())
	{
		string caption = "Building data";
		TeProgress::instance()->setCaption(caption.c_str());
		string msg = "Building in progress. Please, wait!";
		TeProgress::instance()->setMessage(msg);
		TeProgress::instance()->setTotalSteps(tot);
		t2 = clock();
		t0 = t1 = t2;
	}
	
	TeSTInstance obj;
	while(querier_->fetchInstance(obj)) 
	{
		this->insertSTInstance(obj);
		obj.clear();
		if(TeProgress::instance())
		{
			steps++;
			t2 = clock();
			if (int(t2-t1) > dt)
			{
				t1 = t2;
				if(TeProgress::instance()->wasCancelled())
					return false;
				
				if((int)(t2-t0) > dt2)
					TeProgress::instance()->setProgress(steps);
			}
		}
	}
	if(TeProgress::instance())
		TeProgress::instance()->reset();

	querier_->clear(); //clear instances 
	return true;
}

TeSTElement::iterator 
TeSTElement::begin()     
{ 
	return iterator(instances_.begin());
}

TeSTElement::iterator 
TeSTElement::end()       
{
	return iterator(instances_.end());
}



