/*   tubo.h */

/*  A program independent forking object module for gtk based programs.
 *  
 *  Copyright (C) 2000-2005  Edscott Wilson Garcia under GNU GPL
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
/* public stuff */

/* Tubo() returns void pointer to tubo object 
*  usage: see example below */
#include <glib.h>
#include <sys/types.h>
#include <unistd.h>

/**
 * tubo_t:
 *
 **/
#define tubo_t void 

/**
 * TuboCancel:
 * @forkObject: Pointer to pointer to the remote interprocess object
 * returned by @TuboO, @TuboIO or @Tubo_full. 
 * @cleanup: User function to execute once the remote process is
 * terminated. If this value is NULL, it is ignored.
 * @user_data: User data pointer to send to @cleanup function.
 *
 * Returns: NULL pointer.
 *
 *
 **/
tubo_t *TuboCancel(tubo_t *forkObject, void (*cleanup) (void *), void *user_data);

/**
 * TuboPID:
 * @forkObject: Pointer to pointer to the remote interprocess object
 * returned by @TuboO, @TuboIO or @Tubo_full. 
 *
 * Returns: System PID for the remote child process.
 *
 *
 **/

pid_t TuboPID(tubo_t *forkObject);



/**
 * TuboO:
 * @fork_function: Pointer to function to execute after forking.
 * @fork_function_data: Pointer to date to be sent to @fork_function.
 * If this value is NULL, it is ignored.
 * @fork_finished_function: Pointer to function to execute when remote 
 * process has terminated.
 * @operate_stdout: Pointer to function to process output produced by 
 * remote process towards stdout. Set to NULL to ignore output.
 * @operate_stderr: Pointer to function to process output produced by 
 * remote process towards stderr. Set to NULL to ignore output.
 * @user_data: Pointer to user_data passed to @fork_finished_function, 
 * @operate_stdin, and @operate_stdout.
 * 
 * Returns: Pointer to the remote interprocess object.
 *
 **/
tubo_t *
TuboO(	   void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t, void *), 
	   int (*operate_stdout) (int, void *, void *), 
	   int (*operate_stderr) (int, void *, void *),
	   void *user_data);
/**
 * TuboIO:
 * @fork_function: Pointer to function to execute after forking.
 * @fork_function_data: Pointer to date to be sent to @fork_function.
 * If this value is NULL, it is ignored.
 * @fork_finished_function: Pointer to function to execute when remote 
 * process has terminated.
 * @operate_stdin: Pointer to the file descriptor to be used by the parent
 * process to write to stdin at the remote child process. If this value 
 * is NULL, then it is ignored.
 * @operate_stdout: Pointer to function to process output produced by 
 * remote process towards stdout. Set to NULL to ignore output.
 * @operate_stderr: Pointer to function to process output produced by 
 * remote process towards stderr. Set to NULL to ignore output.
 * @user_data: Pointer to user_data passed to @fork_finished_function, 
 * @operate_stdin, and @operate_stdout.
 *
 * Returns: Pointer to the remote interprocess object.
 *
 **/

tubo_t *
TuboIO(	   void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t, void *), 
	   int *operate_stdin, 
	   int (*operate_stdout) (int, void *, void *), 
	   int (*operate_stderr) (int, void *, void *),
	   void *user_data);


/**
 * Tubo_full:
 * @fork_function: Pointer to function to execute after forking.
 * @fork_function_data: Pointer to date to be sent to @fork_function.
 * If this value is NULL, it is ignored.
 * @fork_finished_function: Pointer to function to execute when remote 
 * process has terminated.
 * @operate_stdin: Pointer to the file descriptor to be used by the parent
 * process to write to stdin at the remote child process. If this value 
 * is NULL, then it is ignored.
 * @operate_stdout: Pointer to function to process output produced by 
 * remote process towards stdout. Set to NULL to ignore output.
 * @operate_stderr: Pointer to function to process output produced by 
 * remote process towards stderr. Set to NULL to ignore output.
 * @user_data: Pointer to user_data passed to @fork_finished_function, 
 * @operate_stdin, and @operate_stdout.
 * @read_delay: Millisecond delay before reading. Usually just set this 
 * value to zero to use the default delay (150). If you set to very low values,
 * you will not be able to handle more than a couple processes simultaneously.
 * 
 *
 * Returns: Pointer to the remote interprocess object.
 *
 **/



tubo_t *
Tubo_full(	   void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t, void *), 
	   int *operate_stdin, 
	   int (*operate_stdout) (int, void *, void *), 
	   int (*operate_stderr) (int, void *, void *),
	   void *user_data, 
	   int read_delay);


/**
 * Tubo_full_nonblock:
 * @fork_function: Pointer to function to execute after forking.
 * @fork_function_data: Pointer to date to be sent to @fork_function.
 * If this value is NULL, it is ignored.
 * @fork_finished_function: Pointer to function to execute when remote 
 * process has terminated.
 * @operate_stdin: Pointer to the file descriptor to be used by the parent
 * process to write to stdin at the remote child process. If this value 
 * is NULL, then it is ignored.
 * @operate_stdout: Pointer to function to process output produced by 
 * remote process towards stdout. Set to NULL to ignore output.
 * @operate_stderr: Pointer to function to process output produced by 
 * remote process towards stderr. Set to NULL to ignore output.
 * @user_data: Pointer to user_data passed to @fork_finished_function, 
 * @operate_stdin, and @operate_stdout.
 * @read_delay: Millisecond delay before reading. Usually just set this 
 * value to zero to use the default delay (150). If you set to very low values,
 * you will not be able to handle more than a couple processes simultaneously.
 * 
 *
 * Returns: Pointer to the remote interprocess object.
 *
 **/



tubo_t *
Tubo_full_blockread(	   void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t, void *), 
	   int *operate_stdin, 
	   int (*operate_stdout) (int, void *, void *), 
	   int (*operate_stderr) (int, void *, void *),
	   void *user_data, 
	   int read_delay);



/***********************************************************/
