
/******************************************************************************
 ******************************************************************************/

/** \file yada_pgsql.c
 * yada postgresql module
 *
 * $Id: yada_pgsql.c 142 2006-07-26 16:13:13Z grizz $
 */

/******************************************************************************
 * L I C E N S E **************************************************************
 ******************************************************************************/

/*
 * Copyright (c) 2003, 2004 dev/IT - http://www.devit.com
 *
 * This file is part of yada.
 *
 * Yada is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Yada is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with yada; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/******************************************************************************
 * I N C L U D E S ************************************************************
 ******************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <dlfcn.h>

#include <errno.h>

#include "_yada.h"
#include "common.h"
#include "libpq-fe.h"

/******************************************************************************
 * D E F I N E S **************************************************************
 ******************************************************************************/

/* A result structure, since libpq returns results as a table, we must track
 * the table row position for 'popping' from the table sequentially.
 */
struct _yada_pg_s_result
{
  PGresult *res;
  unsigned long row_count;
  unsigned long cur_row;
};

/******************************************************************************
 * T Y P E D E F S ************************************************************
 ******************************************************************************/

struct yada_modpriv_t
{
  PGconn *db;
};

/******************************************************************************
 * F U N C T I O N S **********************************************************
 ******************************************************************************/

/******************************************************************************/
/** notice handler to stop libpq from writing to stderr
 */

void yada_pgsql_noticer(void *arg, const char *message)
{
}

/******************************************************************************/
/** connect to postgresql database
 * @param dbstr hostname:port:databasename
 * @param yada_user username for db access
 * @param yada_pass password for db access
 */

static int yada_pgsql_connect(yada_t *_yada, char *yada_user, char *yada_pass)
{
  int i;
  char *fsep;
  char *dbargs[] = {0, 0, 0};

  
  fsep = dbargs[0] = strdup(_yada->dbstr);
  for(i = 1; i < 3 && (fsep = strchr(fsep, ':')); i++)
    {
    *fsep++ = 0;
    dbargs[i] = fsep;
    }

  _yada->_mod->db = PQsetdbLogin(dbargs[0], dbargs[1], NULL, NULL, dbargs[2],
   yada_user, yada_pass);

  free(dbargs[0]);

  if(PQstatus(_yada->_mod->db) != CONNECTION_OK)
    {
    _yada_set_err(_yada, errno, PQerrorMessage(_yada->_mod->db));
    return(0);
    }

  PQsetNoticeProcessor(_yada->_mod->db, yada_pgsql_noticer, 0);

  return(1);
}

/******************************************************************************/
/** disconnect from database
 */

static void yada_pgsql_disconnect(yada_t *_yada)
{

  if(!_yada->_mod->db)
    return;

  PQfinish(_yada->_mod->db);
  _yada->_mod->db = 0;
}

/******************************************************************************/
/** escape string
 */

char *yada_pgsql_escstr(char *src, int slen, char *dest, int *dlen)
{
  if(!slen)
    slen = strlen(src);

  /* no dest specified, alloc for it */
  if(!dest)
    if(!(dest = malloc((slen << 1) + 1)))
      return(0);

  *dlen = PQescapeString(dest, src, slen);
  return(dest);
}

/******************************************************************************/

static int yada_pgsql__exec(yada_t *_yada, char *sqlstr, int sqlstr_len)
{
  char pg_dealloc = 1;
  char *pgquery;
  PGresult *pg_res;
  int pgtouples;


  if(!sqlstr_len)
    {
    sqlstr_len = strlen(sqlstr);
    pgquery = sqlstr;
    pg_dealloc = 0;
    }
  else
    {
    if(!(pgquery = malloc(sizeof(char) + sqlstr_len)))
      {
      _yada_set_err(_yada, YADA_ENOMEM, "Cannot allocate memory for query!");
      return(-1);
      }
    pgquery[sqlstr_len] = '\0';
    memcpy(pgquery, sqlstr, sqlstr_len);
    }

  pg_res = PQexec(_yada->_mod->db, pgquery);
  
  /* just return 1 */
  if(PQresultStatus(pg_res) == PGRES_COMMAND_OK)
    pgtouples = 1;
  else if(PQresultStatus(pg_res) == PGRES_TUPLES_OK)
    pgtouples = PQntuples(pg_res);
  /* an error, since we don't use COPY */
  else
    {
    _yada_set_err(_yada, PQresultStatus(pg_res), PQerrorMessage(_yada->_mod->db));
    PQclear(pg_res);
    if(pg_dealloc == 1)
      free(pgquery);
    return(-1);
    }

  PQclear(pg_res);

  if(pg_dealloc == 1)
    free(pgquery);
  
  return(pgtouples);
}

/******************************************************************************/

static yada_rc_t* yada_pgsql__query(yada_t *_yada, char *sqlstr, int sqlstr_len)
{
  int pgtouples;
  char pg_dealloc = 1;
  char *pgquery;
  yada_rc_t *_yrc;
  struct _yada_pg_s_result *y_res;
  PGresult *pg_res;


  if(!sqlstr_len)
    {
    sqlstr_len = strlen(sqlstr);
    pgquery = sqlstr;
    pg_dealloc = 0;
    }
  else
    {
    if(!(pgquery = malloc(sizeof(char) + sqlstr_len)))
      {
      _yada_set_err(_yada, YADA_ENOMEM, "Cannot allocate memory for query!");
      return(0);
      }
    pgquery[sqlstr_len] = '\0';
    memcpy(pgquery, sqlstr, sqlstr_len);
    }

  pg_res = PQexec(_yada->_mod->db, pgquery);
  if(pg_dealloc == 1)
    free(pgquery);
  
  if(pg_res)
    {
    if(PQresultStatus(pg_res) == PGRES_TUPLES_OK)
      pgtouples = PQntuples(pg_res);
    else if(PQresultStatus(pg_res) == PGRES_COMMAND_OK)
      pgtouples = 0;
    else
      {
      _yada_set_err(_yada, PQresultStatus(pg_res), PQerrorMessage(_yada->_mod->db));
      PQclear(pg_res);
      return(0);
      }

    /* no results */
    if(pgtouples < 1 && PQresultStatus(pg_res) == PGRES_TUPLES_OK)
      {
      if(!(_yrc = _yada_rc_new(_yada)))
        {
        _yada_set_yadaerr(_yada, YADA_ENOMEM);
        PQclear(pg_res);
        return(0);
        }

      _yrc->t = YADA_RESULT;
      _yrc->data = 0;

      PQclear(pg_res);
      return(_yrc);
      }
    }
  else
    {
    _yada_set_err(_yada, errno, PQerrorMessage(_yada->_mod->db));
    return(0);
    }

  /* return results */
  if(!(_yrc = _yada_rc_new(_yada)))
    {
    _yada_set_yadaerr(_yada, YADA_ENOMEM);
    PQclear( pg_res );
    return(0);
    }

  if(!(y_res = malloc(sizeof(struct _yada_pg_s_result))))
    {
    _yada_set_yadaerr(_yada, YADA_ENOMEM);
    PQclear(pg_res);
    return(0);
    }

  y_res->res = pg_res;
  y_res->row_count = pgtouples;
  y_res->cur_row = 0;

  _yrc->t = YADA_RESULT;
  _yrc->data = y_res;
  return(_yrc);
}

/******************************************************************************/

static void yada_pgsql_free_result(yada_t *_yada, yada_rc_t *_yrc)
{
  struct _yada_pg_s_result *y_res;


  if(!(y_res = _yrc->data))
    return;

  PQclear(y_res->res);
  free(y_res);
}

/******************************************************************************/

static void yada_pgsql__destroy(yada_t *_yada)
{
  yada_pgsql_disconnect(_yada);
}

/******************************************************************************/
/** fetch result row into bound vars
 * @param yada yada struct
 * @param rrc result rc
 * @param brc bound vars rc
 */

int yada_pgsql_fetch(yada_t *_yada, yada_rc_t *rrc, yada_rc_t *brc)
{
  int i, di = 0;
  long pg_fcount;
  struct _yada_pg_s_result *y_res;
  yada_bindset_t *bindset;


  if(!(y_res = rrc->data))
    return(0);

  /* last result */
  if(y_res->row_count <= y_res->cur_row)
    return(0);

  bindset = brc->data;

  pg_fcount = PQnfields(y_res->res);
  
  for(i = 0; i < bindset->eles; i++, di++)
    {

    /************************************************/
    /* bound variables */
    if(bindset->ele[i].t > 0)
      {
      if( di >= pg_fcount )
        {
        *(char *)bindset->ele[i].ptr = 0;

        if(bindset->ele[i].t == 'b')
          i++;
        continue;
        }

      switch(bindset->ele[i].t)
        {
      case 'b':
        memcpy(bindset->ele[i].ptr, PQgetvalue(y_res->res, y_res->cur_row, di),
         PQgetlength(y_res->res, y_res->cur_row, di));
        /* set len var */
        *(unsigned long *)bindset->ele[++i].ptr =
         PQgetlength(y_res->res, y_res->cur_row, di);
        break;
      case 'd':
        *(int *)bindset->ele[i].ptr =
         atoi(PQgetvalue(y_res->res, y_res->cur_row, di));
        break;
      case 'l':
        *(long long *)bindset->ele[i].ptr =
         atoll(PQgetvalue(y_res->res, y_res->cur_row, di));
        break;
      case 's':
      case 'e':
      case 'v':
        /* +1 for terminating null */
        memcpy(bindset->ele[i].ptr, PQgetvalue(y_res->res, y_res->cur_row, di),
         PQgetlength(y_res->res, y_res->cur_row, di) + 1);
        break;
        }
      continue;
      }

    /************************************************/
    /* bound pointers */
    if(di >= pg_fcount)
      {
      *(char **)bindset->ele[i].ptr = 0;

      if(bindset->ele[i].t == -'b')
        i++;
      continue;
      }

    switch(-bindset->ele[i].t)
      {
    case 'd':
      bindset->ele[i].var.i = atoi(PQgetvalue(y_res->res, y_res->cur_row, di));
      *(int **)bindset->ele[i].ptr = &(bindset->ele[i].var.i);
      break;
    case 'l':
      bindset->ele[i].var.l = atoll(PQgetvalue(y_res->res, y_res->cur_row, di));
      *(long long **)bindset->ele[i].ptr = &(bindset->ele[i].var.l);
      break;
    case 's':
    case 'e':
    case 'v':
      if(PQgetisnull(y_res->res, y_res->cur_row, di))
        {
        *(char **)bindset->ele[i].ptr = 0;
        break;
        }

      *(char **)bindset->ele[i].ptr =
       PQgetvalue(y_res->res, y_res->cur_row, di);
      break;
    case 'b':
      if(PQgetisnull(y_res->res, y_res->cur_row, di))
        {
        *(char **)bindset->ele[i].ptr = 0;
        *(unsigned long *)bindset->ele[++i].ptr = 0;
        break;
        }

      *(char **)bindset->ele[i].ptr =
       PQgetvalue(y_res->res, y_res->cur_row, di);
      /* set len var */
      *(unsigned long *)bindset->ele[++i].ptr =
       PQgetlength(y_res->res, y_res->cur_row, di);
      break;
      }
    } /* for(eles) */

  y_res->cur_row++;
  return(1);
}

/******************************************************************************/
/** starts a transaction
 */

static int yada_pgsql_trx(yada_t *_yada, int flags)
{
  int pg_stat;
  PGresult *pg_res;


  pg_res = PQexec(_yada->_mod->db, "BEGIN");
  pg_stat = PQresultStatus(pg_res);
  PQclear(pg_res);

  /* return 0 if no error */
  return(pg_stat != PGRES_COMMAND_OK);
}

/******************************************************************************/
/** commits a transaction
 */

static int yada_pgsql_commit(yada_t *_yada)
{
  int pg_stat;
  PGresult *pg_res;


  pg_res = PQexec(_yada->_mod->db, "COMMIT");
  pg_stat = PQresultStatus(pg_res);
  PQclear(pg_res);

  /* return 0 if no error */
  return(pg_stat != PGRES_COMMAND_OK && pg_stat != PGRES_TUPLES_OK);
}

/******************************************************************************/
/** aborts a transaction
 */

static int yada_pgsql_rollback(yada_t *_yada, int flags)
{
  int pg_stat;
  PGresult *pg_res;


  pg_res = PQexec(_yada->_mod->db, "ROLLBACK");
  pg_stat = PQresultStatus(pg_res);
  PQclear(pg_res);

  /* return 0 if no error */
  return(pg_stat != PGRES_COMMAND_OK && pg_stat != PGRES_TUPLES_OK);
}

/******************************************************************************/
/** get the last insert id
 * @returns last insert id, or 0 on failure
 */

static uint64_t yada_pgsql_insert_id(yada_t *_yada, char *table, char *col)
{
  /* this needs to be user set */
  return(0);
}

/******************************************************************************/
/** create and init yada struct
 * @returns non zero on success, 0 on error
 */

int yada_mod_init(yada_t *_yada)
{
  if(!(_yada->_mod = calloc(1, sizeof(yada_modpriv_t))))
    return(0);

  /* yada module base functions */
  _yada->type_id = YADA_PGSQL;
  _yada->connect = yada_pgsql_connect;
  _yada->disconnect = yada_pgsql_disconnect;

  /* native prepare funtions */
  _yada->prepare = _yada_prepare;
  _yada->preparef = _yada_preparef;

  /* yada compat prepare funtions */
  _yada->yprepare = _yada_prepare;
  _yada->ypreparef = _yada_preparef;

  _yada->xprepare = _yada_xprepare;

  _yada->execute = _yada_execute;
  _yada->xexecute = _yada_xexecute;

  _yada->query = _yada_query;
  _yada->xquery = _yada_xquery;

  /* yada util functions */
  _yada->escstr = yada_pgsql_escstr;
  _yada->dumpexec = _yada_dumpexec;

  /* yada compat bind functions */
  _yada->bind = _yada_bind;
  _yada->fetch = yada_pgsql_fetch;

  /* transaction functions */
  _yada->trx = yada_pgsql_trx;
  _yada->commit = yada_pgsql_commit;
  _yada->rollback = yada_pgsql_rollback;

  _yada->insert_id = yada_pgsql_insert_id;

  /* private interfaces */
  _yada->_priv->exec = yada_pgsql__exec;
  _yada->_priv->query = yada_pgsql__query;
  _yada->_priv->destroy = yada_pgsql__destroy;

  _yada->_priv->free_rc[YADA_STATEMENT] = _yada_free_stmt;
  _yada->_priv->free_rc[YADA_RESULT] = yada_pgsql_free_result;
  _yada->_priv->free_rc[YADA_BINDSET] = yada_free_bindset;

  return(1);
}

/******************************************************************************
 ******************************************************************************/

