/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: freeq.c,v 1.9.2.1 2003/04/11 19:15:54 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <string.h>
#include <zorp/freeq.h>
#include <zorp/log.h>

#define FREE_QUEUE_SIZE 16
#define FREE_QUEUE_NEXT_SLOT(p) ((p < FREE_QUEUE_SIZE - 1) ? p + 1 : 0)

static ZFreeQueue *freeq;

static GStaticMutex lock = G_STATIC_MUTEX_INIT;

typedef struct _ZFreeQueueItem
{
  gpointer f;
  ZFreeFunc freefn;
} ZFreeQueueItem;

typedef struct _ZRealFreeQueue 
{
  ZFreeQueueItem *free_queue;
  gint first_free, free_queue_size;
} ZRealFreeQueue;

static ZFreeQueue *
z_free_queue_new(void)
{
  ZRealFreeQueue *self = g_new0(ZRealFreeQueue, 1);
  
  self->free_queue_size = 64;
  self->free_queue = g_new0(ZFreeQueueItem, self->free_queue_size);
  return (ZFreeQueue *) self;
}

static void
z_free_queue_free(ZFreeQueue *s)
{
  ZRealFreeQueue *self = (ZRealFreeQueue *) s;
  
  g_free(self->free_queue);
  g_free(self);
}

static void
z_free_queue_add_item(ZFreeQueue *s, gpointer f, ZFreeFunc freefn)
{
  ZRealFreeQueue *self = (ZRealFreeQueue *) s;
  
  g_static_mutex_lock(&lock);
  if (self->first_free >= self->free_queue_size)
    {
      /*LOG
        This message indicates that the free queue - used for freeing up
        structures in a synchronized manner - is full. Zorp automatically
        resizes the free queue so this message is harmless. The old size
        of the free queue is reported, the new one will be twice as big.
       */
      z_log(NULL, CORE_DEBUG, 5, "Free queue full resizing free queue; old_freeq_size='%d'", self->free_queue_size);
      self->free_queue = g_renew(ZFreeQueueItem, self->free_queue, self->free_queue_size * 2);
      self->free_queue_size *= 2;
    }

  self->free_queue[self->first_free].f = f;
  self->free_queue[self->first_free].freefn = freefn;
  self->first_free++;
  g_static_mutex_unlock(&lock);
}

static void
z_free_queue_flush_items(ZFreeQueue *s)
{
  ZRealFreeQueue *self = (ZRealFreeQueue *) s;
  ZFreeQueueItem *items;
  gint i, num;

  g_static_mutex_lock(&lock);
  num = self->first_free;
  items = alloca(sizeof(ZFreeQueueItem) * num);
  memcpy(items, self->free_queue, num * sizeof(ZFreeQueueItem));
  self->first_free = 0;
  g_static_mutex_unlock(&lock);

  for (i = 0; i < num; i++)
    {
      if (items[i].freefn)
        items[i].freefn(items[i].f);
      else
        g_free(items[i].f);
    }
}

void
z_free_queue_init(void)
{
  freeq = z_free_queue_new();
}

void
z_free_queue_destroy(void)
{
  z_free_queue_free(freeq);
  freeq = NULL;
}

void
z_free_queue_add(gpointer f, ZFreeFunc freefn)
{
  z_free_queue_add_item(freeq, f, freefn);
}

void
z_free_queue_flush(void)
{
  z_free_queue_flush_items(freeq);
}
