/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: poll.c,v 1.37.2.5 2003/09/16 19:40:28 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/poll.h>
#include <zorp/stream.h>
#include <zorp/log.h>
#include <zorp/source.h>
#include <zorp/error.h>

#include <glib.h>

#include <sys/types.h>
#ifndef G_OS_WIN32
#  include <sys/poll.h>
#endif
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <assert.h>

/*+

  ZRealPoll is an implementation of the ZPoll interface defined in
  poll.h. It's a callback based poll loop, which can be used by proxy
  modules. It holds a collection of ZStream objects and calls its
  callbacks when a requested I/O event occurs. ZPoll objects are
  reference counted and are automatically freed when the number of
  references to a single instance reaches zero.

  +*/
typedef struct _ZRealPoll
{
  guint ref_count;
  GMainContext *context;
  GPollFD *pollfd;
  guint pollfd_num;
  gboolean quit;
  GStaticMutex lock;
  GSource *wakeup;
  GHashTable *streams;
} ZRealPoll;

typedef struct _ZPollSource
{
  GSource super;
  gboolean wakeup;
} ZPollSource;
    
static gboolean
z_poll_source_prepare(GSource *s, gint *timeout)
{
  ZPollSource *self = (ZPollSource *)s;
    
  z_enter();
  
  if (self->wakeup)
    {
      return TRUE;
    }
  
  *timeout = -1;
  z_leave();
  return FALSE;
}

static gboolean
z_poll_source_check (GSource *s)
{
  z_enter();
      
  z_leave();
  return FALSE;
}

static gboolean
z_poll_source_dispatch (GSource *s,
                    GSourceFunc  callback,
                       gpointer  user_data)
{
  ZPollSource *self = (ZPollSource *)s;
  
  z_enter();

  self->wakeup = FALSE;

  z_leave();
  return TRUE;
}

GSourceFuncs z_poll_source_funcs = 
{
  z_poll_source_prepare,
  z_poll_source_check,
  z_poll_source_dispatch,
  NULL
};

static void 
z_poll_stream_unref(gpointer s)
{
  ZStream *stream = (ZStream *)s;
  
  z_enter();
  z_stream_detach_source(stream);
  z_stream_unref(stream);
  z_leave();
}

/*+

  Creates a new ZPoll instance.

  Parameters:
    none

  Returns:
    new instance

  +*/
ZPoll *
z_poll_new(void)
{
  ZRealPoll *self = g_new0(ZRealPoll, 1);
  
  z_enter();
  g_return_val_if_fail( self != NULL, NULL);
  
  self->ref_count = 1;
  self->quit = FALSE;
  self->pollfd_num = 4;
  self->pollfd = g_new(GPollFD, self->pollfd_num);
  self->streams = g_hash_table_new_full(g_direct_hash,
                                        g_direct_equal,
                                        z_poll_stream_unref,
                                        NULL);
  
  self->context = g_main_context_default();
  if (g_main_context_acquire(self->context))
    {
      g_main_context_ref(self->context);
    }
  else
    {
      self->context = g_main_context_new();
      assert(g_main_context_acquire(self->context));
    }
  
  self->wakeup = g_source_new(&z_poll_source_funcs,
                              sizeof(ZPollSource));
  
  g_source_attach(self->wakeup, self->context);
  
  z_leave();
  return (ZPoll *) self;
}

/*+

  Used internally to free up an instance when the reference count
  reaches 0.

  Parameters:
    s         instance pointer

  Returns:
    none

  +*/
static void
z_poll_destroy(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *)s;

  z_enter();
  if(self->wakeup)
    {
      g_source_destroy(self->wakeup);
      g_source_unref(self->wakeup);
      self->wakeup = NULL;
    }
  g_hash_table_destroy(self->streams);
  g_main_context_release(self->context);
  g_main_context_unref(self->context);
  g_free(self->pollfd);
  
  g_free(self);
  z_leave();
}

/*+

  Increment the reference count of the given ZPoll instance.

  Parameters:
    s            ZPoll instance

  Returns: 
    none

  +*/
void 
z_poll_ref(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;
  
  z_enter();
  self->ref_count++;
  z_leave();
}


/*+

  Decrement the reference count of the given ZPoll instance, and free
  it using z_poll_destroy() if it reaches 0.

  Parameters:
    s           ZPoll instance

  Returns:
    none

  +*/
void
z_poll_unref(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;

  z_enter();
  if (self)
    {
      g_assert(self->ref_count > 0);
      self->ref_count--;
      if (self->ref_count == 0)
        z_poll_destroy(s);
    }
  z_leave();
}

/*+

  Register a ZStream to be monitored by a ZPoll instance.

  Parameters:
    s          ZPoll instance
    stream     stream to register

  Returns:
    none

  +*/
void
z_poll_add_stream(ZPoll *s, ZStream *stream)
{
  ZRealPoll *self = (ZRealPoll *) s;

  z_enter();
  
  z_stream_ref(stream);
  g_hash_table_insert(self->streams, stream, NULL);
  z_stream_attach_source(stream, self->context);
  
  z_leave();
}

/*+

  Remove a ZStream from a ZPoll instance.

  Parameters:
    s          ZPoll instance
    stream     stream to remove

  Returns:
    none

  +*/
void
z_poll_remove_stream(ZPoll *s, ZStream *stream)
{
  ZRealPoll *self = (ZRealPoll *) s;
  
  z_enter();
  
  g_hash_table_remove(self->streams, stream);

  z_leave();
}

/*+

  Run an iteration of the poll loop. Monitor filedescriptors of
  registered Streams, and call appropriate callbacks.

  Parameters:
    s          ZPoll instance
    timeout    poll timeout in milliseconds
 
  Returns:
    TRUE if the iteration should be ended.

  +*/
guint
z_poll_iter_timeout(ZPoll *s, gint timeout)
{
  ZRealPoll *self = (ZRealPoll *) s;
  gint max_priority = G_PRIORITY_LOW;
  gint polltimeout;
  gint fdnum = 0;
  GPollFunc pollfunc;
  gint rc;

  z_enter();
  z_errno_set(0);
  if (self->quit)
    {
      z_leave();
      return 0;
    }
  
  g_main_context_prepare (self->context, &max_priority);

  fdnum = g_main_context_query(self->context,
                               max_priority,
                               &polltimeout,
                               self->pollfd,
                               self->pollfd_num);

  while (fdnum > (gint)self->pollfd_num)
    {
      /*LOG
        This message indicate that polling fd-s growing.
       */
      z_log(NULL, CORE_DEBUG, 7, "Polling fd structure growing; old_num='%d'", self->pollfd_num);
      self->pollfd_num *= 2;
      self->pollfd = g_renew(GPollFD, self->pollfd, self->pollfd_num);
      
      fdnum = g_main_context_query(self->context,
                                   max_priority,
                                   &polltimeout,
                                   self->pollfd,
                                   self->pollfd_num);
    }

  if (polltimeout <= -1)
    polltimeout = timeout;
  else if (timeout > -1)
    polltimeout = MIN(polltimeout, timeout);

  pollfunc = g_main_context_get_poll_func(self->context);

  rc = pollfunc(self->pollfd, fdnum, polltimeout);

  g_main_context_check(self->context, max_priority, self->pollfd, fdnum);
  g_main_context_dispatch(self->context);

  if (rc == -1 && !z_errno_is(EINTR))
    {
      z_leave();
      return 0;
    }

  if (rc == 0 && polltimeout == timeout)
    {
      z_errno_set(ETIMEDOUT);
      z_leave();
      return 0;
    }

  z_leave();
  return 1;
}

void
z_poll_wakeup(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;
  ZPollSource *src;
  
  z_enter();
  
  src = (ZPollSource *)self->wakeup;
  src->wakeup = TRUE;
  g_main_context_wakeup(self->context);
  z_leave();
}


gboolean
z_poll_is_running(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;
  
  z_enter();
  z_leave();
  return !self->quit;
}

void
z_poll_quit(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;  
  
  z_enter();
  
  self->quit = TRUE;
  z_poll_wakeup(s);
  
  z_leave();
}

GMainContext *
z_poll_get_context(ZPoll *s)
{
  ZRealPoll *self = (ZRealPoll *) s;
  
  z_enter();
  z_leave();
  return self->context;
}
