/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: sockaddr.c,v 1.19.2.5 2003/07/28 14:14:01 sasa Exp $
 *
 * Author  : Bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/sockaddr.h>
#include <zorp/log.h>
#include <zorp/cap.h>
#include <zorp/socket.h>
#include <zorp/error.h>

#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
  #include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#else
#  include <sys/socket.h>
#  include <netinet/in.h>
#  include <sys/un.h>
#  include <arpa/inet.h>
#endif

static ZSockAddrFuncs inet_range_sockaddr_funcs;

/*+

  Thread friendly version of inet_ntoa(), converts an IP address to
  human readable form.

  Parameters:
    buf        store result in this buffer
    bufsize    the available space in buf
    a          address to convert.

  Returns:
    the address of buf

  +*/
gchar *
z_inet_ntoa(char *buf, size_t bufsize, struct in_addr a)
{
  unsigned int ip = ntohl(a.s_addr);

  g_snprintf(buf, bufsize, "%d.%d.%d.%d", 
	   (ip & 0xff000000) >> 24,
	   (ip & 0x00ff0000) >> 16,
	   (ip & 0x0000ff00) >> 8,
	   (ip & 0x000000ff));
  return buf;
}

gboolean
z_inet_aton(const char *buf, struct in_addr *a)
{
#ifdef HAVE_INET_ATON
  return inet_aton(buf, a);
#else
  a->s_addr = inet_addr(buf);
  return TRUE;
#endif
}


/* general ZSockAddr functions */


/*+

  General function to allocate and initialize a ZSockAddr structure,
  and convert a libc style sockaddr * pointer to our representation.

  Parameters:
    sa         libc sockaddr * pointer to convert
    salen      size of sa

  Returns:
    a ZSockAddr instance

  +*/
ZSockAddr *
z_sockaddr_new(struct sockaddr *sa, int salen)
{
  z_enter();
  switch (sa->sa_family)
    {
    case AF_INET:
      if (salen != sizeof(struct sockaddr_in))
        {
          z_leave();
          return NULL;
        }
      z_leave();
      return z_sockaddr_inet_new2((struct sockaddr_in *) sa);
#ifndef G_OS_WIN32
    case AF_UNIX:
      z_leave();
      return z_sockaddr_unix_new2((struct sockaddr_un *) sa, salen);
#endif
    default:
      /*LOG
        This message indicates an internal error, Zorp tries to use an
        address family it doesn't support.
       */
      z_log(NULL, CORE_ERROR, 3, "Unsupported socket family in z_sockaddr_new(); family='%d'", sa->sa_family);
      z_leave();
      return NULL;
    }
  z_leave();
}

/*+

  Format a ZSockAddr into human readable form, calls the format
  virtual method of ZSockAddr.

  Parameters:
    a        instance pointer of a ZSockAddr
    text     destination buffer
    n        the size of text

  Returns:
    text is filled with a human readable representation of a, and a
    pointer to text is returned.

  +*/
char *
z_sockaddr_format(ZSockAddr *a, gchar *text, gulong n)
{
  return a->sa_funcs->sa_format(a, text, n);
}

/*+

  Increment the reference count of a ZSockAddr instance.

  Parameters:
    a         pointer to ZSockAddr
  
  Returns:
    the same instance

  +*/
ZSockAddr *
z_sockaddr_ref(ZSockAddr *a)
{
  if (a)
    a->refcnt++;
  return a;
}

/*+

  Decrement the reference count of a ZSockAddr instance, and free if
  the refcnt reaches 0.

  Parameters:
    a        ZSockAddr instance

  Returns:
    none

  +*/
void 
z_sockaddr_unref(ZSockAddr *a)
{
  /* FIXME: maybe add a callback to funcs table */
  if (a) 
    {
      g_assert(a->refcnt > 0);
      if (--a->refcnt == 0)
        {
          if (!a->sa_funcs->freefn)
            g_free(a);
          else
            a->sa_funcs->freefn(a);
        }
    }
}

/* AF_INET socket address */
/*+

  ZSockAddrInet is an implementation of the ZSockAddr interface,
  encapsulating an IPv4 host address and port number.

  +*/
typedef struct _ZSockAddrInet
{
  gint refcnt;
  guint32 flags;
  ZSockAddrFuncs *sa_funcs;
  int salen;
  struct sockaddr_in sin;
  gpointer options;
  guint options_length;
} ZSockAddrInet;

/*+ private function to prepare an IPv4 style bind, e.g. set
  SO_REUSEADDR +*/
static GIOStatus
z_sockaddr_inet_bind_prepare(int sock, ZSockAddr *addr)
{
  int tmp = 1;

  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *)&tmp, sizeof(tmp)) == 0)
    return G_IO_STATUS_NORMAL;
  
  return G_IO_STATUS_ERROR;
}

static ZSockAddr *
z_sockaddr_inet_clone(ZSockAddr *addr, gboolean wildcard)
{
  ZSockAddrInet *self = g_new0(ZSockAddrInet, 1);
  
  memcpy(self, addr, sizeof(*self));
  self->refcnt = 1;
  if (wildcard)
    self->sin.sin_port = 0;
  
  return (ZSockAddr *) self;
}

/*+ format an IPv4 address into human readable form */
gchar *
z_sockaddr_inet_format(ZSockAddr *addr, gchar *text, gulong n)
{
  ZSockAddrInet *inet_addr = (ZSockAddrInet *) addr;
  char buf[32];
  
  g_snprintf(text, n, "AF_INET(%s:%d)", 
	     z_inet_ntoa(buf, sizeof(buf), inet_addr->sin.sin_addr), 
             ntohs(inet_addr->sin.sin_port));
  return text;
}

void
z_sockaddr_inet_free(ZSockAddr *addr)
{
  ZSockAddrInet *self = (ZSockAddrInet *) addr;

  if (self->options)
    g_free(self->options);
  g_free(addr);
}

static ZSockAddrFuncs inet_sockaddr_funcs = 
{
  z_sockaddr_inet_bind_prepare,
  NULL,
  z_sockaddr_inet_clone,
  z_sockaddr_inet_format,
  z_sockaddr_inet_free
};

/*+

  Allocate and initialize an IPv4 socket address.

  Parameters:
    ip          text representation of an IP address
    port        port number in host byte order

  Returns:
    the new instance

  +*/
ZSockAddr *
z_sockaddr_inet_new(const gchar *ip, guint16 port)
{
  ZSockAddrInet *addr;
  struct in_addr netaddr;
  
  if (!z_inet_aton(ip, &netaddr))
    {
      return NULL;
    }
  
  addr = g_new0(ZSockAddrInet, 1);
  addr->refcnt = 1;
  addr->flags = 0;
  addr->salen = sizeof(struct sockaddr_in);
  addr->sin.sin_family = AF_INET;
  addr->sin.sin_addr = netaddr;
  addr->sin.sin_port = htons(port);
  addr->sa_funcs = &inet_sockaddr_funcs;
  
  return (ZSockAddr *) addr;
}


/*+

  Allocate and initialize an IPv4 socket address using libc sockaddr *
  structure.

  Parameters:
    sin         the sockaddr_in structure to convert

  Returns:
    the allocated instance.

  +*/
ZSockAddr *
z_sockaddr_inet_new2(struct sockaddr_in *sin)
{
  ZSockAddrInet *addr = g_new0(ZSockAddrInet, 1);
  
  addr->refcnt = 1;
  addr->flags = 0;
  addr->salen = sizeof(struct sockaddr_in);
  addr->sin = *sin;
  addr->sa_funcs = &inet_sockaddr_funcs;
  
  return (ZSockAddr *) addr;
}

guint 
z_sockaddr_inet_get_address(ZSockAddr *s)
{
  ZSockAddrInet *self = (ZSockAddrInet *) s;

  return self->sin.sin_addr.s_addr;
}

guint 
z_sockaddr_inet_get_port(ZSockAddr *s)
{
  ZSockAddrInet *self = (ZSockAddrInet *) s;

  if (self->sa_funcs == &inet_range_sockaddr_funcs)
    return 0;
  return ntohs(self->sin.sin_port);
}

gboolean 
z_sockaddr_inet_check(ZSockAddr *s)
{
  return (s->sa_funcs == &inet_sockaddr_funcs) || (s->sa_funcs == &inet_range_sockaddr_funcs);
}

#if ZORPLIB_ENABLE_IPOPTIONS
void
z_sockaddr_inet_get_options(ZSockAddr *s, gpointer *options, guint *options_length)
{
  ZSockAddrInet *self = (ZSockAddrInet *) s;
  
  *options = self->options;
  *options_length = self->options_length;
}

void
z_sockaddr_inet_set_options(ZSockAddr *s, gpointer options, guint options_length)
{
  ZSockAddrInet *self = (ZSockAddrInet *) s;
  
  if (self->options)
    g_free(self->options);
  
  self->options = g_malloc0(options_length);
  memcpy(self->options, options, options_length);
  self->options_length = options_length;
}
#endif


/*+ Similar to ZSockAddrInet, but binds to a port in the given range +*/

/* 
 * NOTE: it is assumed that it is safe to cast from ZSockAddrInetRange to
 * ZSockAddrInet
 */
typedef struct _ZSockAddrInetRange
{
  gint refcnt;
  guint32 flags;
  ZSockAddrFuncs *sa_funcs;
  int salen;
  struct sockaddr_in sin;
  gpointer options;
  guint options_length;
  guint16 min_port, max_port, last_port;
} ZSockAddrInetRange;

static GIOStatus
z_sockaddr_inet_range_bind(int sock, ZSockAddr *a)
{
  ZSockAddrInetRange *addr = (ZSockAddrInetRange *) a;
  int port;
  
  if (addr->min_port > addr->max_port)
    {
      /*LOG
        This message indicates that SockAddrInetRange was given incorrect
        parameters, the allowed min_port is greater than max_port.
       */
      z_log(NULL, CORE_ERROR, 3, "SockAddrInetRange, invalid range given; min_port='%d', max_port='%d'", addr->min_port, addr->max_port);
      return G_IO_STATUS_ERROR;
    }
  for (port = addr->last_port; port <= addr->max_port; port++)
    {
      /* attempt to bind */
      addr->sin.sin_port = htons(port);
      if (z_ll_bind(sock, (struct sockaddr *) &addr->sin, addr->salen) == 0)
        {
          /*LOG
            This debug message is used by SockAddrInetRange to inform you
            about the allocated dynamic port.
           */
          z_log(NULL, CORE_DEBUG, 6, "SockAddrInetRange, successfully bound; min_port='%d', max_port='%d', port='%d'", addr->min_port, addr->max_port, port);
          addr->last_port = port + 1;
          return G_IO_STATUS_NORMAL;
        }
    }
  for (port = addr->min_port; port <= addr->max_port; port++)
    {
      /* attempt to bind */
      addr->sin.sin_port = htons(port);
      if (z_ll_bind(sock, (struct sockaddr *) &addr->sin, addr->salen) == 0)
        {
          /*NOLOG*/ /* the previous message is the same */
          z_log(NULL, CORE_DEBUG, 6, "SockAddrInetRange, successfully bound; min_port='%d', max_port='%d', port='%d'", addr->min_port, addr->max_port, port);
          addr->last_port = port + 1;
          return G_IO_STATUS_NORMAL;
        }
    }
  addr->last_port = addr->min_port;
  return G_IO_STATUS_ERROR;
}

static ZSockAddr *
z_sockaddr_inet_range_clone(ZSockAddr *addr, gboolean wildcard_clone G_GNUC_UNUSED)
{
  ZSockAddrInetRange *self = g_new0(ZSockAddrInetRange, 1);
  
  memcpy(self, addr, sizeof(*self));
  self->refcnt = 1;
  if (self->max_port > self->min_port)
    {
      self->last_port = (rand() % (self->max_port - self->min_port)) + self->min_port;
    }
  else if (self->max_port == self->min_port)
    {
      self->last_port = self->min_port;
    }
  
  return (ZSockAddr *) self;
}

static ZSockAddrFuncs inet_range_sockaddr_funcs = 
{
  NULL,
  z_sockaddr_inet_range_bind,
  z_sockaddr_inet_range_clone,
  z_sockaddr_inet_format,
  z_sockaddr_inet_free,
};

ZSockAddr *
z_sockaddr_inet_range_new(const gchar *ip, guint16 min_port, guint16 max_port)
{
  ZSockAddrInetRange *addr;
  struct in_addr netaddr;

  if (!z_inet_aton(ip, &netaddr))
    return NULL;
  
  addr = g_new0(ZSockAddrInetRange, 1);
  addr->refcnt = 1;
  addr->flags = 0;
  addr->salen = sizeof(struct sockaddr_in);
  addr->sin.sin_family = AF_INET;
  addr->sin.sin_addr = netaddr;
  addr->sin.sin_port = 0;
  addr->sa_funcs = &inet_range_sockaddr_funcs;
  if (max_port > min_port)
    {
      addr->last_port = (rand() % (max_port - min_port)) + min_port;
    }
  else if (max_port == min_port)
    {
      addr->last_port = min_port;
    }
  addr->min_port = min_port;
  addr->max_port = max_port;
  
  return (ZSockAddr *) addr;
}

#ifndef G_OS_WIN32
/* AF_UNIX socket address */

/*+

  The ZSockAddrUnix class is an implementation of the ZSockAddr
  interface encapsulating AF_UNIX domain socket names.

  +*/
typedef struct _ZSockAddrUnix
{
  gint refcnt;
  guint32 flags;
  ZSockAddrFuncs *sa_funcs;
  int salen;
  struct sockaddr_un saun;
} ZSockAddrUnix;

static GIOStatus z_sockaddr_unix_bind_prepare(int sock, ZSockAddr *addr);
static gchar *z_sockaddr_unix_format(ZSockAddr *addr, gchar *text, gulong n);

static ZSockAddr *
z_sockaddr_unix_clone(ZSockAddr *addr, gboolean wildcard_clone G_GNUC_UNUSED)
{
  ZSockAddrUnix *self = g_new0(ZSockAddrUnix, 1);
  
  memcpy(self, addr, sizeof(*self));
  self->refcnt = 1;

  return (ZSockAddr *) self;
}

static ZSockAddrFuncs unix_sockaddr_funcs = 
{
  z_sockaddr_unix_bind_prepare,
  NULL,
  z_sockaddr_unix_clone,
  z_sockaddr_unix_format
};

/* anonymous if name == NULL */

/*+

  Allocate and initialize a ZSockAddrUnix instance.

  Parameters:
    name              socket filename

  Returns:
    the new instance

  +*/
ZSockAddr *
z_sockaddr_unix_new(const gchar *name)
{
  ZSockAddrUnix *addr = g_new0(ZSockAddrUnix, 1);
  
  addr->refcnt = 1;
  addr->flags = 0;
  addr->sa_funcs = &unix_sockaddr_funcs;
  addr->saun.sun_family = AF_UNIX;
  if (name)
    {
      strncpy(addr->saun.sun_path, name, sizeof(addr->saun.sun_path) - 1);
      addr->saun.sun_path[sizeof(addr->saun.sun_path) - 1] = 0;
      addr->salen = sizeof(addr->saun) - sizeof(addr->saun.sun_path) + strlen(addr->saun.sun_path) + 1;
    }
  else
    {
      addr->saun.sun_path[0] = 0;
      addr->salen = 2;
    }
  return (ZSockAddr *) addr;
}

/*+

  Allocate and initialize a ZSockAddrUnix instance, using libc
  sockaddr_un strucutre.

  Parameters:
    saun        sockaddr_un structure to convert
    sunlen      size of saun

  Returns:
    the new instance

  +*/
ZSockAddr *
z_sockaddr_unix_new2(struct sockaddr_un *saun, int sunlen)
{
  ZSockAddrUnix *addr = g_new0(ZSockAddrUnix, 1);
  
  addr->refcnt = 1;
  addr->flags = 0;
  addr->sa_funcs = &unix_sockaddr_funcs;
  addr->salen = sunlen;
  addr->saun = *saun;
  return (ZSockAddr *) addr;
}

/*+ private function to prepare a bind on AF_UNIX sockets, e.g. unlink
  the socket if it exists and is a socket. +*/
static GIOStatus
z_sockaddr_unix_bind_prepare(int sock, ZSockAddr *addr)
{
  ZSockAddrUnix *unix_addr = (ZSockAddrUnix *) addr;
  struct stat st;
  
  if (unix_addr->saun.sun_path[0] == 0)
    return G_IO_STATUS_ERROR;

  if (stat(unix_addr->saun.sun_path, &st) == -1 ||
      !S_ISSOCK(st.st_mode))
    return G_IO_STATUS_ERROR;
  
  unlink(unix_addr->saun.sun_path);  
  return G_IO_STATUS_NORMAL;
}

/*+ Convert a ZSockAddrUnix into human readable form. +*/
gchar *
z_sockaddr_unix_format(ZSockAddr *addr, gchar *text, gulong n)
{
  ZSockAddrUnix *unix_addr = (ZSockAddrUnix *) addr;
  
  g_snprintf(text, n, "AF_UNIX(%s)", 
             unix_addr->salen > 2 && unix_addr->saun.sun_path[0] ? 
             unix_addr->saun.sun_path : "anonymous");
  return text;
}

#endif
