/***************************************************************************
 *
 * Copyright (c) 2000,2001,2002 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: socketsource.c,v 1.6.2.4 2003/08/18 15:29:30 sasa Exp $
 *
 * Author  : bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/socketsource.h>
#include <zorp/log.h>
#include <zorp/error.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#endif

static gboolean
z_socket_source_prepare(GSource *s, gint *timeout)
{
  ZSocketSource *self = (ZSocketSource *) s;

  if (self->suspended)
    {
      self->poll.events = 0;
      self->poll.revents = 0;
      *timeout = -1;
      return FALSE;
    }
  else
    {
#ifdef G_OS_WIN32
      self->poll.events = G_IO_IN;
      z_trace(NULL, "WinSock: WSAEventSelect(%d,%d,%o) at %s line %d", self->fd, self->poll.fd, self->cond, __FILE__, __LINE__);
      if (WSAEventSelect(self->fd, (HANDLE) self->poll.fd, self->cond) == SOCKET_ERROR)
        {
          /*LOG
            This message attempt when some reason WinSock event
            cannot set up.
           */
          z_log(NULL, CORE_ERROR, 0, "Failed to setup WinSock event; error='%m'");
          *timeout = -1;
          return FALSE;
        }
#else
      self->poll.events = self->cond;
#endif
    }
    
  if (self->timeout_time != -1)
    {
      *timeout = (self->timeout_time - time(NULL)) * 1000;
      if (*timeout < 0)
        *timeout = 0;
    }
  else
    *timeout = -1;

  return FALSE;
}

static gboolean
z_socket_source_check(GSource *s)
{
  ZSocketSource *self = (ZSocketSource *) s;
#ifdef G_OS_WIN32
  WSANETWORKEVENTS evs;
#endif

  if (self->timeout_time > 0 && time(NULL) >= self->timeout_time)
    {
      self->timed_out = TRUE;
      return TRUE;
    }
  else
    self->timed_out = FALSE;

#ifdef G_OS_WIN32
  if (self->acceptevent)
    return TRUE;
  WSAEnumNetworkEvents(self->fd, (HANDLE) self->poll.fd, &evs);
  self->poll.revents = (gushort)evs.lNetworkEvents;
  if(evs.lNetworkEvents != 0)
    self->acceptevent = TRUE;
  z_trace(NULL, "WinSock: Event %d on fd %d at %s line %d", self->poll.revents, self->poll.fd, __FILE__, __LINE__);
#endif

  return !!self->poll.revents;
}

static gboolean
z_socket_source_dispatch(GSource     *s,
                         GSourceFunc  callback,
                         gpointer     user_data)
{
  ZSocketSource *self = (ZSocketSource *) s;

  z_trace(NULL, "Dispatching event for fd %d", self->poll.fd);
  if(!self->suspended)
    {
#ifdef G_OS_WIN32
      self->acceptevent = FALSE;
#endif
      return ((ZSocketSourceFunc) callback)(self->timed_out, user_data);
    }
  return TRUE;
}

void
z_socket_source_finalize(GSource *source)
{
#ifdef G_OS_WIN32
  ZSocketSource *self = (ZSocketSource *) source;
  
  z_trace(NULL, "WinSock: Event #%d destroyed at %s line %d",self->poll.fd, __FILE__, __LINE__);
  WSACloseEvent((HANDLE) self->poll.fd);
#endif
}

GSourceFuncs z_socket_source_funcs = 
{
  z_socket_source_prepare,
  z_socket_source_check,
  z_socket_source_dispatch,
  z_socket_source_finalize
};

void
z_socket_source_suspend(GSource *s)
{
  ZSocketSource *self = (ZSocketSource *) s;

  z_enter();
  self->suspended = TRUE;
  z_leave();
}

void
z_socket_source_resume(GSource *s)
{
  ZSocketSource *self = (ZSocketSource *) s;
  
  z_enter();
  self->suspended = FALSE;
  z_leave();
}


GSource *
z_socket_source_new(gint fd, GIOCondition cond, gint timeout)
{
  ZSocketSource *self;
  
  self = (ZSocketSource *) g_source_new(&z_socket_source_funcs, sizeof(ZSocketSource));

#ifdef G_OS_WIN32
  self->fd = fd;
  self->poll.fd = (int)WSACreateEvent(); //by Abi
#else
  self->poll.fd = fd;
#endif
  self->cond = cond;
  g_source_add_poll(&self->super, &self->poll);
  g_source_set_can_recurse(&self->super, FALSE);
  if (timeout != -1)
    self->timeout_time = time(NULL) + timeout;
  else
    self->timeout_time = -1;
    
  return &self->super;
}
