/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: stackdump.c,v 1.11 2004/05/22 14:04:17 bazsi Exp $
 *
 * Author  : bazsi
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/stackdump.h>
#include <zorp/log.h>

#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#if HAVE_BACKTRACE
#include <execinfo.h>
#endif

#ifdef __i386__


/**
 * z_stackdump_log_stack:
 * @p: signal context
 *
 * Log parts of the current stack in hexadecimal form.
 **/
void
z_stackdump_log_stack(ZSignalContext *p)
{
  unsigned long *esp __attribute__((__may_alias__)) = (unsigned long *) p->esp;
  int i;

  if (!esp)
    {
      /*LOG
        This message indicates that the contents of the pre-signal ESP register is bogus, stackdump will rely on the current stack frame.
       */
      z_log(NULL, CORE_ERROR, 0, "ESP is NULL, stackdump is not available, falling back to current frame;");
      esp = (unsigned long *) &esp;
    }

  for (i = 0; i < 64; i++)
    {
      /*NOLOG*/
      z_log(NULL, CORE_ERROR, 0, "Stack 0x%08lx: %08lx %08lx %08lx %08lx", (unsigned long) esp, esp[0], esp[1], esp[2], esp[3]);
      esp += 4;
    }
}

/**
 * z_stackdump_log_backtrace:
 * @p: signal context
 * 
 * x86 specific stack unrolling function which logs all return addresses and
 * their associated frame pointers.
 **/
void
z_stackdump_log_backtrace(ZSignalContext *p)
{
  /* NOTE: this is i386 specific */
  unsigned long *ebp __attribute__((__may_alias__)) = (unsigned long *) p->ebp;
  
  /*NOLOG*/
  z_log(NULL, CORE_ERROR, 0, "retaddr=0x%lx, ebp=0x%lx", p->eip, (unsigned long) ebp);
  
  while (ebp > (unsigned long *) &ebp && *ebp) 
    {
      /*NOLOG*/
      z_log(NULL, CORE_ERROR, 0, "retaddr=0x%lx, ebp=0x%lx", *(ebp+1), *ebp);
      ebp = (unsigned long *) *ebp;
    }
}

/**
 * z_stackdump_log_context:
 * @p: signal context
 *
 * Log information found directly in the signal context (register contents).
 **/
void
z_stackdump_log_context(ZSignalContext *p)
{
  /*LOG
    This message indicates that the program caught a fatal signal.
    Please report this event to the Zorp QA team.
   */
  z_log(NULL, CORE_ERROR, 0, 
        "Fatal signal occurred, dumping stack; eax='%08lx', ebx='%08lx', ecx='%08lx', edx='%08lx', esi='%08lx', edi='%08lx', ebp='%08lx', esp='%08lx', eip='%08lx'",
        p->eax, p->ebx, p->ecx, p->edx, p->esi, p->edi, p->ebp, p->esp, p->eip);
}

#else

#define z_stackdump_log_stack(p) 
#define z_stackdump_log_backtrace(p)
#define z_stackdump_log_context(p)

#endif

/**
 * z_stackdump_log_maps:
 *
 * This function reads and logs the contents of the /proc/<pid>/maps file
 * which includes memory mapping for mapped shared objects.
 **/
void
z_stackdump_log_maps(void)
{
  int fd;

  fd = open("/proc/self/maps", O_RDONLY);
  if (fd != -1)
    {
      gchar buf[32768];
      int rc;
      gchar *p, *eol;
      gint avail, end = 0;

      while (1)
        {
          avail = sizeof(buf) - end;
          rc = read(fd, buf + end, avail);

          if (rc == -1)
            break;
	  end += rc;
	  if (rc == 0)
	    break;
          p = buf;
          while (*p && p < (buf + end))
            {
              eol = memchr(p, '\n', buf + end - p);
              if (eol)
                {
                  *eol = 0;
                  /*NOLOG*/
                  z_log(NULL, CORE_ERROR, 0, "%s", p);
                  p = eol + 1;
                }
              else
                {
                  end = end - (p - buf);
                  memmove(buf, p, end);
                  break;
                }
            }
        }
      if (end)
	/*NOLOG*/
        z_log(NULL, CORE_ERROR, 0, "%.*s", end, buf);
      close(fd);
    }
  else
    {
      /*LOG
        This message indicates that system was unable to open the /proc/self/maps 
	file to gather information on the previous error and the dump will lack
	this information.
       */
      z_log(NULL, CORE_ERROR, 0, "Error opening /proc/self/maps;");
    }
}

#if HAVE_BACKTRACE
/**
 * z_stackdump_log_symbols:
 * 
 * This function uses the libc backtrace() and backtrace_symbols() functions
 * to display a backtrace with resolved names. As this does not always
 * succeed (as it uses malloc()) the alternative implementation without
 * symbols is also present.
 **/
void
z_stackdump_log_symbols(void)
{
  void *bt[256];
  int count, i;
  
  count = backtrace(bt, 256);
  if (count)
    {
      gchar **symbols;
     
      /*LOG
        This message reports the number of dumped symbols.
       */
      z_log(NULL, CORE_ERROR, 0, "Symbol dump; count='%d'", count);
      symbols = backtrace_symbols(bt, count);
      for (i = 0; i < count; i++)
        {
          /*NOLOG*/
          z_log(NULL, CORE_ERROR, 0, "%p: %s", bt[i], symbols[i]);
        }
    }
}
#else

#define z_stackdump_log_symbols()

#endif

/**
 * z_stackdump_log:
 * @p: signal context
 *
 * This function gathers as much information for post-mortem analysis as
 * possible. It is usually called from a fatal signal handler (like SIGSEGV).
 * The current signal context can be queried with the z_stackdump_get_context() macro.
 * This function is Linux & x86 specific.
 **/
void
z_stackdump_log(ZSignalContext *p)
{
  z_stackdump_log_context(p);
  z_stackdump_log_backtrace(p);
  z_stackdump_log_maps();
  z_stackdump_log_stack(p);
  z_stackdump_log_symbols();
}

