/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: streamline.c,v 1.87 2004/07/29 08:40:18 bazsi Exp $
 *
 * Author  : SaSa
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/stream.h>
#include <zorp/streamline.h>
#include <zorp/log.h>

#include <glib.h>

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <assert.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#else
#  include <sys/socket.h>
#  include <sys/poll.h>
#endif

#define ZRL_SAVED_FLAGS_MASK    0x0000FFFF

#define ZRL_IGNORE_TILL_EOL     0x00010000
#define ZRL_LINE_AVAIL_SET      0x00020000
#define ZRL_LINE_AVAIL          0x00040000
#define ZRL_ERROR		0x00080000

extern ZClass ZStreamLine__class;

typedef struct _ZStreamLine
{
  ZStream super;

  guint flags;
  gchar *buffer;
  gsize bufsize, pos, end, oldpos;
  GIOCondition child_cond;
            
} ZStreamLine;

typedef struct _ZStreamLineExtra
{
  guint flags;
} ZStreamLineExtra;

static inline gboolean
z_stream_line_have_line(ZStreamLine *self)
{
  z_enter();
  if (!(self->flags & ZRL_LINE_AVAIL_SET))
    {
      guint avail = self->end - self->pos;
      char *eol = memchr(self->buffer + self->pos, self->flags & ZRL_EOL_NUL ? '\0' : '\n', avail);

      self->flags |= ZRL_LINE_AVAIL_SET;
      if (eol)
        self->flags |= ZRL_LINE_AVAIL;
      else
        self->flags &= ~ZRL_LINE_AVAIL;
    }
  z_leave();
  return !!(self->flags & ZRL_LINE_AVAIL);
}

static inline gboolean
z_stream_line_buf_empty(ZStreamLine *self)
{
  return self->pos == self->end;
}

static GIOStatus
z_stream_line_get_from_buf(ZStreamLine *self, gchar **line, gsize *length)
{
  guint avail = self->end - self->pos;
  char *eol = memchr(self->buffer + self->pos, self->flags & ZRL_EOL_NUL ? '\0' : '\n', avail);
  char *nul;

  z_enter();

  if (eol)
    {
      *length = eol - (self->buffer + self->pos) + 1;
      *line = self->buffer + self->pos;
      self->oldpos = self->pos;
      self->pos += *length;

      if (!(self->flags & ZRL_EOL_NUL))
        {
          nul = memchr(*line, '\0', *length);
          if (nul)
            {
              if (!(self->flags & ZRL_NUL_NONFATAL))
                {
		  /*LOG
		    This message indicates that an invalid NUL character was found in the line, but the
		    policy prohibits it.
		   */
                  z_log(NULL, CORE_ERROR, 2, "Invalid line, embedded NUL character found; buffer='%.*s'", (gint) *length, *line);
                  z_leave();
                  return G_IO_STATUS_ERROR;
                }
            }
        }

      if ((self->flags & ZRL_EOL_NL) || (self->flags & ZRL_EOL_NUL))
        {
          (*length)--;
        }
      else if (self->flags & ZRL_EOL_CRLF)
        {
          (*length)--;
          if (eol - self->buffer >= 1 && *(eol - 1) == '\r')
            {
              (*length)--;
            }
          else if (self->flags & ZRL_EOL_FATAL)
            {
	      /*LOG
	        This message indicates that the CRLF sequence was invalid, because no LF character was
		found before the CR character, but the policy requires it.
	       */
              z_log(NULL, CORE_ERROR, 2, "Invalid line, bad CRLF sequence; buffer='%.*s'", (gint) *length, *line);
              z_leave();
              return G_IO_STATUS_ERROR;
            }
        }
      z_leave();
      return G_IO_STATUS_NORMAL;
    }
  else if (self->pos)
    {
      *length = 0;
      memmove(self->buffer, self->buffer + self->pos, avail);
      self->end = avail;
      self->pos = 0;
      self->oldpos = 0;
    }
  z_leave();
  return G_IO_STATUS_AGAIN;
}

GIOStatus
z_stream_line_get(ZStream *stream, gchar **line, gsize *length, GError **error)
{
  ZStreamLine *self;
  gsize avail, bytes_read;
  gint rc;

  z_enter();
  
  self = Z_CAST(z_stream_search_stack(stream, Z_STREAM_FLAG_READ, Z_CLASS(ZStreamLine)), ZStreamLine);
  
  if (self->flags & ZRL_ERROR)
    {
      z_leave();
      return G_IO_STATUS_ERROR;
    }

  self->child_cond = 0;
  self->flags &= ~ZRL_LINE_AVAIL_SET;
  if (self->end != self->pos)
    {
      /* we have something, try to return it */
      rc = z_stream_line_get_from_buf(self, line, length);
      if (rc == G_IO_STATUS_NORMAL)
        {
          z_leave();
          return rc;
        }
      else if (rc == G_IO_STATUS_ERROR)
        {
          /* no need to log, we sent the exact error reason in get_from_buf() */
          self->flags |= ZRL_ERROR;
          z_leave();
          return rc;
        }
      /* the available data is now at the beginning of our buffer */
    }
  else
    {
      self->pos = self->end = self->oldpos = 0;
    }

  *length = 0;
  *line = NULL;

  while (1)
    {
      avail = self->bufsize - self->end;
      if (!avail)
        {
          /*
           * this means that there's no space in the buffer, and no eol could
           * be found
           */
          if (self->flags & ZRL_IGNORE_TILL_EOL)
            {
              self->pos = self->end = self->oldpos = 0;
              avail = self->bufsize;
            }
          else if (self->flags & ZRL_TRUNCATE)
            {
              /* FIXME: this hack violates the standard GIOStatus model, as
               * it returns G_IO_STATUS_AGAIN while consuming some of the
               * input data, callers detect this case by checking whether
               * line_len is not 0.
               */
              *line = self->buffer;
              *length = self->bufsize;
              self->pos = self->end = self->oldpos = 0;
              self->flags |= ZRL_IGNORE_TILL_EOL;
              z_leave();
              return G_IO_STATUS_NORMAL;
            }
          else if (self->flags & ZRL_SPLIT)
            {
              /* FIXME: this hack violates the standard GIOStatus model, as
               * it returns G_IO_STATUS_AGAIN while consuming some of the
               * input data, callers detect this case by checking whether
               * line_len is not 0.
               */
              *line = self->buffer;
              *length = self->bufsize;
              self->pos = self->end = self->oldpos = 0;
              z_leave();
              return G_IO_STATUS_AGAIN;
            }
          else
            {
              /*LOG
                This message is sent when the proxy reading too long
                input line. This may cause by a cracking attempt. But if
                not try to increase max_line_length.
               */
              z_log(NULL, CORE_ERROR, 2, "Line too long; buffer='%.*s'", (gint) self->bufsize, self->buffer);
              *line = NULL;
              *length = 0;
              self->flags |= ZRL_ERROR;
              z_leave();
              return G_IO_STATUS_ERROR;
            }
        }

      self->super.child->timeout = self->super.timeout;
      rc = z_stream_read(self->super.child, self->buffer + self->end, avail, &bytes_read, error);
      switch (rc)
        {
        case G_IO_STATUS_NORMAL:
          self->end += bytes_read;
          rc = z_stream_line_get_from_buf(self, line, length);
          switch (rc)
            {
            case G_IO_STATUS_NORMAL:
              if (self->flags & ZRL_IGNORE_TILL_EOL)
                {
                  self->flags &= ~ZRL_IGNORE_TILL_EOL;
                  rc = G_IO_STATUS_AGAIN;
                  /* No break, go to next case */
                }
              else
                {
                  z_leave();
                  return rc;
                }
            case G_IO_STATUS_AGAIN:
              if (self->flags & ZRL_SINGLE_READ)
                {
                  *line = NULL;
                  *length = 0;
                  z_leave();
                  return rc;
                }
              break;
            default:
              *line = NULL;
              *length = 0;
              z_leave();
              return rc;
            }
          break;
        case G_IO_STATUS_EOF:
          z_leave();
          return G_IO_STATUS_EOF;
        case G_IO_STATUS_AGAIN:
          *line = NULL;
          *length = 0;
          z_leave();
          return G_IO_STATUS_AGAIN;
        default:
          /* I/O error is already logged from z_stream_read() */
          self->flags |= ZRL_ERROR;
          z_leave();
          return G_IO_STATUS_ERROR;
        }
    }
}

GIOStatus
z_stream_line_get_copy(ZStream *s, gchar *line, gsize *length, GError **error)
{
  gchar *b;
  gsize len;
  GIOStatus res;
  ZStreamLine *self;

  z_enter();
  
  self = Z_CAST(z_stream_search_stack(s, Z_STREAM_FLAG_READ, Z_CLASS(ZStreamLine)), ZStreamLine);
  
  res = z_stream_line_get(s, &b, &len, error);
  
  if (res == G_IO_STATUS_NORMAL || (res == G_IO_STATUS_AGAIN && len > 0))
    {
      if (len > *length)
        {
          /* FIXME: this uses the non-standard trick to return the part of
           * the line which fits the result buffer, e.g. it returns
           * G_IO_STATUS_AGAIN with line_len set to a non-zero value while
           * consuming the returned amount from the buffer.  this should be
           * cleaned up, but currently it is the less risky solution. */
          if (self->flags & ZRL_SPLIT)
            {
              if (self->end == 0)
                {
                  self->pos = len - *length;
                  self->end = len;
                }
              else
                {
                  self->pos = self->oldpos + *length;
                }
              len = *length;
              res = G_IO_STATUS_AGAIN;
            }
          else
            {
	      /*LOG
	        This message indicates that the line buffer is too small to hold the
		whole line. It is likely caused by some max size limit or by some
		standard default.
	       */
              z_log(NULL, CORE_ERROR, 2, "Line buffer too small; buffer='%.*s'", len, b);
              z_leave();
              return G_IO_STATUS_ERROR;
            }
        }
      *length = len;
      memcpy(line, b, len);
      z_leave();
      return res;
    }
  else
    {
      *length = 0;
    }
  z_leave();
  return res;
}

void
z_stream_line_unget_line(ZStream *stream)
{
  ZStreamLine *self;
  
  z_enter();
  
  self = Z_CAST(z_stream_search_stack(stream, Z_STREAM_FLAG_READ, Z_CLASS(ZStreamLine)), ZStreamLine);
  
  self->pos = self->oldpos;
  
  z_leave();
}

gboolean
z_stream_line_unget(ZStream *stream, const gchar *unget, gsize unget_len)
{
  ZStreamLine *self;
  gsize avail_before, avail_after;
                                                                                                                                          
  z_enter();
                                                                                                                                          
  self = Z_CAST(z_stream_search_stack(stream, Z_STREAM_FLAG_READ, Z_CLASS(ZStreamLine)), ZStreamLine);
  avail_before = self->pos;
  avail_after = self->bufsize - self->end;
  if (avail_before + avail_after > unget_len)
    {
      if (avail_before > unget_len)
        {
          memmove(&self->buffer[self->pos - unget_len], unget, unget_len);
          self->pos -= unget_len;
        }
      else
        {
          memmove(&self->buffer[unget_len], &self->buffer[self->pos], self->end - self->pos);
          memmove(self->buffer, unget, unget_len);
          self->end = self->end - self->pos + unget_len;
          self->pos = 0;
        }
      z_leave();
      return TRUE;
    }
  z_leave();
  return FALSE;
}

/* I/O callbacks for stacked stream */

static gboolean
z_stream_line_read_callback(ZStream *stream G_GNUC_UNUSED, GIOCondition poll_cond G_GNUC_UNUSED, gpointer s)
{
  ZStreamLine *self = (ZStreamLine *) s;

  z_enter();
  self->child_cond |= Z_STREAM_FLAG_READ;
  z_leave();
  return TRUE;
}

static gboolean
z_stream_line_pri_callback(ZStream *stream G_GNUC_UNUSED, GIOCondition poll_cond G_GNUC_UNUSED, gpointer s)
{
  ZStreamLine *self = (ZStreamLine *) s;

  z_enter();
  self->child_cond |= Z_STREAM_FLAG_PRI;
  z_leave();
  return TRUE;
}

static gboolean
z_stream_line_write_callback(ZStream *stream G_GNUC_UNUSED, GIOCondition poll_cond G_GNUC_UNUSED, gpointer s)
{
  ZStreamLine *self = (ZStreamLine *) s;
  gboolean rc;

  z_enter();
  
  rc = (*self->super.write_cb)(s, poll_cond, self->super.user_data_write);
  
  z_leave();
  return rc;
}

/* virtual methods */

static GIOStatus
z_stream_line_read_method(ZStream *stream,
                          gchar   *buf,
                          gsize   count,
                          gsize   *bytes_read,
                          GError  **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;
  guint avail = self->end - self->pos;
  
  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);
  
  if (avail)
    {
      *bytes_read = MIN(count, avail);
      memmove(buf, self->buffer + self->pos, *bytes_read);
      
      self->oldpos = self->pos;
      self->pos += *bytes_read;

      if (self->pos == self->end)
        self->pos = self->end = 0;
        
      self->flags &= ~ZRL_LINE_AVAIL_SET;

      res = G_IO_STATUS_NORMAL;
    }
  else
    {
      self->child_cond = 0;
      self->super.child->timeout = self->super.timeout;
      res = z_stream_read(self->super.child, buf, count, bytes_read, error);
    }
  
  z_leave();
  return res;
}

static GIOStatus
z_stream_line_write_method(ZStream     *stream,
                           const gchar *buf,
                           gsize       count,
                           gsize       *bytes_written,
                           GError      **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;

  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  self->super.child->timeout = self->super.timeout;
  res = z_stream_write(self->super.child, buf, count, bytes_written, error);
  
  z_leave();
  return res;
}

static GIOStatus
z_stream_line_write_pri_method(ZStream     *stream,
                               const gchar *buf,
                               gsize       count,
                               gsize       *bytes_written,
                               GError      **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;
  
  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);
  
  self->super.child->timeout = self->super.timeout;
  res = z_stream_write_pri(self->super.child, buf, count, bytes_written, error);
  
  z_leave();
  return res;
}

static GIOStatus
z_stream_line_shutdown_method(ZStream *stream, int i, GError **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;
  
  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  res = z_stream_shutdown(self->super.child, i, error);
  
  z_leave();
  return res;

}

static GIOStatus
z_stream_line_close_method(ZStream *stream, GError **error)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GIOStatus res;

  z_enter();
  g_return_val_if_fail ((error == NULL) || (*error == NULL), G_IO_STATUS_ERROR);

  res = z_stream_close(self->super.child, error);
  
  z_leave();
  return res;
}

static gboolean
z_stream_line_ctrl_method(ZStream *s, guint function, gpointer value, guint vlen)
{
  ZStreamLine *self = Z_CAST(s, ZStreamLine);
  gboolean ret = FALSE;
  
  z_enter();

  switch (ZST_CTRL_MSG(function))
    {
    case ZST_LINE_SET_TRUNCATE:
      if (vlen == sizeof(gboolean))
        {
          gboolean flag = *((gboolean *)value);
          if (flag)
            self->flags |= ZRL_TRUNCATE;
          else
            self->flags &= ~ZRL_TRUNCATE;
          
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_SET_NUL_NONFATAL:
      if (vlen == sizeof(gboolean))
        {
          gboolean flag = *((gboolean *)value);
          if (flag)
            self->flags |= ZRL_NUL_NONFATAL;
          else
            self->flags &= ~ZRL_NUL_NONFATAL;
          
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_SET_SPLIT:
      if (vlen == sizeof(gboolean))
        {
          gboolean flag = *((gboolean *)value);
          if (flag)
            self->flags |= ZRL_SPLIT;
          else
            self->flags &= ~ZRL_SPLIT;
          
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_SET_SINGLE_READ:
      if (vlen == sizeof(gboolean))
        {
          gboolean flag = *((gboolean *)value);
          if (flag)
            self->flags |= ZRL_SINGLE_READ;
          else
            self->flags &= ~ZRL_SINGLE_READ;
          
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_SET_POLL_PARTIAL:
      if (vlen == sizeof(gboolean))
        {
          gboolean flag = *((gboolean *)value);
          if (flag)
            self->flags |= ZRL_POLL_PARTIAL;
          else
            self->flags &= ~ZRL_POLL_PARTIAL;
          
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_GET_TRUNCATE:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = !!(self->flags & ZRL_TRUNCATE);
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_GET_SPLIT:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = !!(self->flags & ZRL_SPLIT);
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_GET_SINGLE_READ:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = !!(self->flags & ZRL_SINGLE_READ);
          z_leave();
          return TRUE;
        }
      break;
    case ZST_LINE_GET_POLL_PARTIAL:
      if (vlen == sizeof(gboolean))
        {
          *(gboolean *)value = !!(self->flags & ZRL_POLL_PARTIAL);
          z_leave();
          return TRUE;
        }
      break;
    case ZST_CTRL_SET_CALLBACK_READ:
    case ZST_CTRL_SET_CALLBACK_WRITE:
    case ZST_CTRL_SET_CALLBACK_PRI:
      ret = z_stream_ctrl_method(s, function, value, vlen);
      break;
    default:
      ret = z_stream_ctrl_method(s, ZST_CTRL_MSG_FORWARD | function, value, vlen);
      break;
    }
  z_leave();
  return ret;
}

static void
z_stream_line_attach_source_method(ZStream *stream, GMainContext *context)
{
  ZStreamLine *self = (ZStreamLine *)stream;
  ZStream *p;
  gint stacked_count = 0;

  z_enter();

  z_stream_ref(stream);
  p = self->super.child;
  while (p)
    {
      stacked_count++;
      p = p->child;
    }
  z_stream_attach_source(self->super.child, context);
  if (!stream->source)
    {
      stream->source = z_stream_source_new(stream);
      g_source_set_priority(stream->source, G_PRIORITY_DEFAULT - stacked_count);
      g_source_attach(stream->source, context);
    }
  
  z_stream_unref(stream);
  z_leave();
}

static void
z_stream_line_detach_source_method(ZStream *stream)
{
  ZStreamLine *self = (ZStreamLine *) stream;
  GSource *source;
  
  z_enter();
  if (stream->source)
    {
      source = stream->source;
      stream->source = NULL;
      /*
         NOTE Must be in this order because
         g_source_unref may drop the last
         reference to source.
       */
      g_source_destroy(source);
      g_source_unref(source);
    }

  z_stream_detach_source(self->super.child);
    
  z_leave();
}

static gboolean 
z_stream_line_watch_prepare(ZStream *s, GSource *src G_GNUC_UNUSED, gint *timeout)
{
  ZStreamLine *self = Z_CAST(s, ZStreamLine);
  gboolean ret = FALSE;
  gboolean child_enable = FALSE, child_readable;

  z_enter();
  
  *timeout = -1;

  if (s->want_read)
    {
      child_readable = !!(self->child_cond & Z_STREAM_FLAG_READ);
      if (self->flags & ZRL_POLL_PARTIAL)
        {
          if (z_stream_line_buf_empty(self) && !child_readable)
            {
              child_enable = TRUE;
            }
          else
            {
              child_enable = FALSE;
              ret = TRUE;
            }
        }
      else
        {
          if (!z_stream_line_have_line(self) && !child_readable)
            {
              child_enable = TRUE;
            }
          else
            {
              child_enable = FALSE;
              ret = TRUE;
            }
        }
    }
  else
    child_enable = FALSE;
  
  if (s->want_pri && (self->child_cond & Z_STREAM_FLAG_PRI))
    ret = TRUE;

  z_stream_set_cond(s->child, Z_STREAM_FLAG_READ, child_enable);
  
  if (s->want_write)
    z_stream_set_cond(s->child, Z_STREAM_FLAG_WRITE, TRUE);
  else
    z_stream_set_cond(s->child, Z_STREAM_FLAG_WRITE, FALSE);
  
  z_leave();
  return ret;
}

static gboolean 
z_stream_line_watch_check(ZStream *s, GSource *src G_GNUC_UNUSED)
{
  ZStreamLine *self = (ZStreamLine *) s;
  gboolean ret = FALSE, child_readable;

  z_enter();

  if (s->want_read)
    {
      child_readable = !!(self->child_cond & Z_STREAM_FLAG_READ);
      if (self->flags & ZRL_POLL_PARTIAL)
        {
          if (!z_stream_line_buf_empty(self) || child_readable)
            {
              ret = TRUE;
            }
        }
      else
        {
          if (z_stream_line_have_line(self) || child_readable)
            {
              ret = TRUE;
            }
        }
    }
  
  if (s->want_pri && (self->child_cond & Z_STREAM_FLAG_PRI))
    ret = TRUE;

  z_leave();
  return ret;
}

static
gboolean 
z_stream_line_watch_dispatch(ZStream *s, GSource *src G_GNUC_UNUSED)
{
  ZStreamLine *self = (ZStreamLine *) s;
  gboolean rc = TRUE;

  z_enter();
  
  if (s->want_read)
    rc = self->super.read_cb(s, G_IO_IN, self->super.user_data_read);
  else if (s->want_pri)
    rc = self->super.pri_cb(s, G_IO_PRI, self->super.user_data_pri);

  z_leave();
  return rc;
}

static gsize
z_stream_line_extra_get_size_method(ZStream *s)
{
  return Z_SUPER(s, ZStream)->extra_get_size(s) + sizeof(ZStreamLineExtra);
}

static gsize
z_stream_line_extra_save_method(ZStream *s, gpointer extra)
{
  ZStreamLine *self = Z_CAST(s, ZStreamLine);
  ZStreamLineExtra *line_extra;
  gsize ofs;
  
  ofs = Z_SUPER(s, ZStream)->extra_save(s, extra);
  
  line_extra = (ZStreamLineExtra *) (((gchar *) extra) + ofs);
  line_extra->flags = self->flags & ZRL_SAVED_FLAGS_MASK;
  return ofs + sizeof(ZStreamLineExtra);
}

static gsize
z_stream_line_extra_restore_method(ZStream *s, gpointer extra)
{
  ZStreamLine *self = Z_CAST(s, ZStreamLine);
  ZStreamLineExtra *line_extra;
  gsize ofs;
  
  ofs = Z_SUPER(s, ZStream)->extra_restore(s, extra);
  
  line_extra = (ZStreamLineExtra *) (((gchar *) extra) + ofs);
  self->flags = (self->flags & ~ZRL_SAVED_FLAGS_MASK) | (line_extra->flags & ZRL_SAVED_FLAGS_MASK);
  return ofs + sizeof(ZStreamLineExtra);
}

/* destructor */
static void
z_stream_line_free_method(ZObject *s)
{
  ZStreamLine *self = Z_CAST(s, ZStreamLine);

  z_enter();
  
  g_free(self->buffer);
  z_stream_free_method(s);
  z_leave();
}

ZStreamFuncs z_stream_line_funcs =
{
  {
    Z_FUNCS_COUNT(ZStream),
    z_stream_line_free_method,
  },
  z_stream_line_read_method,
  z_stream_line_write_method,
  NULL,
  z_stream_line_write_pri_method,
  z_stream_line_shutdown_method,
  z_stream_line_close_method,
  z_stream_line_ctrl_method,
  
  z_stream_line_attach_source_method,
  z_stream_line_detach_source_method,
  z_stream_line_watch_prepare,
  z_stream_line_watch_check,
  z_stream_line_watch_dispatch,
  NULL,
  z_stream_line_extra_get_size_method,
  z_stream_line_extra_save_method,
  z_stream_line_extra_restore_method
};

ZClass ZStreamLine__class = 
{
  Z_CLASS_HEADER,
  &ZStream__class,
  "ZStreamLine",
  sizeof(ZStreamLine),
  &z_stream_line_funcs.super,
};

/**
 * z_stream_line_new:
 * @child: 
 * @bufsize: 
 * @flags:
 *
 * Constructs a new ZStreamLine instance, reading from @child, using a buffer
 * sized according to @bufsize.
 **/
ZStream *
z_stream_line_new(ZStream *child, gsize bufsize, guint flags)
{
  ZStreamLine *self;

  z_enter();

  self = Z_CAST(z_stream_new(Z_CLASS(ZStreamLine), child->name, child, Z_STREAM_FLAG_READ), ZStreamLine);
  self->flags = flags;
  self->bufsize = bufsize;
  self->buffer = g_new(gchar, bufsize);
  self->super.timeout = self->super.child->timeout;
  
  z_stream_set_callback(self->super.child, Z_STREAM_FLAG_READ, z_stream_line_read_callback, self, NULL);
  z_stream_set_callback(self->super.child, Z_STREAM_FLAG_WRITE, z_stream_line_write_callback, self, NULL);
  z_stream_set_callback(self->super.child, Z_STREAM_FLAG_PRI, z_stream_line_pri_callback, self, NULL);

  z_leave();
  return (ZStream *) self;
}
