/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: log.h,v 1.31 2004/05/18 15:33:40 abi Exp $
 *
 ***************************************************************************/

#ifndef ZORP_LOG_H_INCLUDED
#define ZORP_LOG_H_INCLUDED

#include <zorp/zorplib.h>

#include <string.h>

#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifndef ENABLE_TRACE
#if ZORPLIB_ENABLE_TRACE
  #define ENABLE_TRACE 1
#endif /* ZORPLIB_ENABLE_TRACE */
#endif /* ENABLE_TRACE */

LIBZORPLL_DLLIMPEX extern gchar fake_session_id[256];

typedef gint (*ZLogMapTagFunc)(const gchar *tag, gsize len);

#define ZORP_SYSLOG_FACILITY LOG_LOCAL6

#define ZLF_SYSLOG      0x0001
#define ZLF_TAGS        0x0002
#define ZLF_THREAD      0x0004
#define ZLF_STDERR      0x0008
#define ZLF_WINDEBUG    0x0010
#define ZLF_ESCAPE      0x0020

#ifndef G_OS_WIN32
  #define z_debug(level, format, args...)   z_llog("core.debug", level, format, ##args)
  #define z_warning(level, format, args...) z_llog("core.warning", level, format, ##args)
  #define z_message(level, format, args...) z_llog("core.message", level, format, ##args)
#else
  #define z_debug(level, format, args)   z_llog("core.debug", level, format, ##args)
  #define z_warning(level, format, args) z_llog("core.warning", level, format, ##args)
  #define z_message(level, format, args) z_llog("core.message", level, format, ##args)
#endif

gboolean z_log_init(const gchar *ls, gint verbose_level, const gchar *syslog_name, guint flags);

void z_log_enable_syslog(const gchar *syslog_name);
void z_log_enable_stderr_redirect(gboolean threaded);
void z_log_enable_tag_map_cache(ZLogMapTagFunc map_tags, gint max_tag);


void z_logv(gchar *class, int level, gchar *format, va_list ap);

#ifndef G_OS_WIN32
  void z_llog(gchar *class, int level, gchar *format, ...) __attribute__ ((format(printf, 3, 4)));
  void z_llog2(gchar *class, int level, gchar *format, ...) __attribute__ ((format(printf, 3, 4)));
#else
  void z_llog(gchar *class, int level, gchar *format, ...);
  void z_llog2(gchar *class, int level, gchar *format, ...);
#endif

int z_log_register_class(gchar *class);
gboolean z_log_enabled_len(gchar *class, gsize class_len, int level);

#ifndef G_OS_WIN32
  #define z_log_enabled(class, level) \
    ({								\
      gboolean __res;						\
      if (__builtin_constant_p(class))				\
        __res = z_log_enabled_len(class, __builtin_strlen(class), level);	\
      else							\
        __res = z_log_enabled_len(class, strlen(class), level);	\
      __res;							\
    })
#else
  #define z_log_enabled(class, level) \
    z_log_enabled_len(class, strlen(class), level)
#endif
      
gboolean z_log_change_verbose_level(gint direction, gint value, gint *new_value);
gboolean z_log_change_logspec(const gchar *log_spec, const gchar **new_value);

void z_log_clear_caches(void);
void z_log_destroy(void);

const gchar *z_log_session_id(const gchar *session_id);

#ifdef G_OS_WIN32
  void z_log(gchar* session_id, gchar* class, int level, gchar* format, ...);
#else
  #define z_log(session_id, class, level, format, args...) \
    do \
      { \
        if (z_log_enabled(class, level)) \
	  /*NOLOG*/ \
          z_llog2(class, level, "(%s): " format, z_log_session_id(session_id) , ##args); \
      } \
    while (0)

#endif

#if ENABLE_TRACE
  #ifndef G_OS_WIN32
    #define z_trace(session_id, args...) z_log(session_id , CORE_TRACE, 7, ##args)
  #else
    #define z_trace
  #endif
  #define z_enter() z_log(NULL, CORE_TRACE, 7, "Enter %s (%s:%d)", __FUNCTION__, __FILE__, __LINE__)
  #define z_leave() z_log(NULL, CORE_TRACE, 7, "Leave %s (%s:%d)", __FUNCTION__, __FILE__, __LINE__)
  #define z_cp() z_log(NULL, CORE_TRACE, 7, "Checkpoint %s (%s:%d)", __FUNCTION__, __FILE__, __LINE__)
  #define z_session_enter(s) z_log(s, CORE_TRACE, 7, "Enter %s (%s:%d)", __FUNCTION__, __FILE__, __LINE__)
  #define z_session_leave(s) z_log(s, CORE_TRACE, 7, "Leave %s (%s:%d)", __FUNCTION__, __FILE__, __LINE__)
  #define z_session_cp(s) z_log(s, CORE_TRACE, 7, "Checkpoint %s (%s:%d)", __FUNCTION__, __FILE__, __LINE__)
#else
  #ifndef G_OS_WIN32
    #define z_trace(session_id, args...)
  #else
    #define z_trace
  #endif
  #define z_enter()
  #define z_leave()
  #define z_cp()
  #define z_session_enter(s)
  #define z_session_leave(s)
  #define z_session_cp(s)

#endif

#ifdef __cplusplus
}
#endif

#endif
