# Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Tests for the "prepare" command."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = []

import httplib
from itertools import (
    chain,
    repeat,
    )
import json
from random import randint
from subprocess import check_call
from textwrap import dedent

from apiclient.maas_client import MAASClient
from maastest import (
    kvmfixture,
    maasfixture,
    utils,
    )
from maastest.maas_enums import NODEGROUPINTERFACE_MANAGEMENT
from maastest.maasfixture import NoBootImagesError
from maastest.testing.factory import make_file
import mock
import netaddr
from six import text_type
from testtools import TestCase
from testtools.matchers import FileContains


class FakeResponse(object):
    """A fake http response used in testing."""

    def __init__(self, content, code):
        self.content = content
        self._code = code

    def read(self):
        return self.content

    @property
    def code(self):
        return self._code


class TestMAASFixture(TestCase):

    def make_kvm_fixture(self):
        """Create a `KVMFixture` with arbitrary series/architecture.

        Its `run_command` method will be replaced by a `MagicMock`, which
        returns success by default.  Its `ip_address` will be patched to
        return an arbitrary IP address.
        """
        fixture = kvmfixture.KVMFixture(
            'droopy', 'i860', proxy_url="http://example.com:8080")
        fixture.run_command = mock.MagicMock(
            return_value=(0, 'stdout output', 'stderr output'))
        ip_address = '.'.join('%d' % randint(1, 254) for octets in range(4))
        fixture.ip_address = mock.MagicMock(return_value=ip_address)
        return fixture

    def make_maas_fixture(self, kvm_fixture=None, proxy_url=None):
        """Create a `MAASFixture` with arbitrary series/architecture.

        The maasfixture.MAASClient's post/get/put methods are replaced by a
        `MagicMock`, which returns success (i.e. a FakeResponse with a
        return code of 200) by default.
        """
        if kvm_fixture is None:
            kvm_fixture = self.make_kvm_fixture()
        series = self.getUniqueString()
        arch = self.getUniqueString()
        if proxy_url is None:
            proxy_url = self.getUniqueString()
        filter = self.getUniqueString()
        fixture = maasfixture.MAASFixture(
            kvm_fixture, proxy_url, series, arch, filter)
        ok_response = FakeResponse('', httplib.OK)
        for method in ('get', 'post', 'put'):
            self.patch(
                maasfixture.MAASClient, method,
                mock.MagicMock(return_value=ok_response))
        return fixture

    def make_maas_local_settings(self, default_maas_url):
        """Compose a `maas_local_settings.py`, as installed."""
        return ('DEFAULT_MAAS_URL = "%s"' % default_maas_url).encode('ascii')

    def make_rewritten_maas_local_settings(self, old_url, new_url):
        """Compose a `maas_local_settings.py` as rewritten.

        We rewrite this file to change its `DEFAULT_MAAS_URL` setting.  It's
        done in a shell command, so needs careful testing.
        """
        return dedent("""
            # Replaced by maas-test:
            # DEFAULT_MAAS_URL = "%s"
            DEFAULT_MAAS_URL = "%s"
            """ % (old_url, new_url)).encode('ascii')

    def make_file(self, name=None, content=None):
        return make_file(self, name=name, content=content)

    def patch_MAASClient(self, client, method, content, code=httplib.OK):
        """Patch a MAASClient so that it will return a pre-cooked response."""
        json_content = json.dumps(content)
        response = FakeResponse(json_content, code)
        mock_object = mock.MagicMock(return_value=response)
        self.patch(client, method, mock_object)
        return mock_object

    def test_compose_rewrite_for_default_maas_url_rewrites_url(self):
        old_url = 'http://1.1.1.1/'
        new_url = 'http://2.2.2.2/'
        config_file = make_file(
            self, content=self.make_maas_local_settings(old_url))

        command = maasfixture.compose_rewrite_for_default_maas_url(
            config_file, new_url)
        # Test the generated sed command, by running it locally.  In real use
        # this runs only on the virtual machine.
        # (But make sure it doesn't sudo.  That won't do in tests.)
        self.assertNotIn('sudo', command)
        check_call(command)

        new_content = self.make_rewritten_maas_local_settings(old_url, new_url)
        self.assertThat(config_file, FileContains(new_content.decode('ascii')))

    def test_compose_rewrite_for_default_maas_url_accepts_path(self):
        old_url = 'http://old-host.com/OLDMAAS'
        new_url = 'http://new-host.local/NEWMAAS'
        config_file = make_file(
            self, content=self.make_maas_local_settings(old_url))

        check_call(
            maasfixture.compose_rewrite_for_default_maas_url(
                config_file, new_url))

        new_content = self.make_rewritten_maas_local_settings(old_url, new_url)
        self.assertThat(config_file, FileContains(new_content.decode('ascii')))

    def test_compose_rewrite_for_default_maas_url_preserves_other_text(self):
        top = b"# Text before setting."
        bottom = b"# Text after setting."
        old_url = "http://1.1.1.1/"
        old_content = b'\n'.join(
            [top, self.make_maas_local_settings(old_url), bottom])
        config_file = make_file(self, content=old_content)
        new_url = "http://2.2.2.2/"

        check_call(
            maasfixture.compose_rewrite_for_default_maas_url(
                config_file, new_url))

        new_part = self.make_rewritten_maas_local_settings(old_url, new_url)
        new_content = b'\n'.join([top, new_part, bottom])
        self.assertThat(config_file, FileContains(new_content.decode('ascii')))

    def test_compose_rewrite_for_default_maas_url_ignores_commented_URL(self):
        top = b'#DEFAULT_MAAS_URL = "http://9.9.9.9/"\n'
        old_url = "http://1.1.1.1/"
        old_content = top + self.make_maas_local_settings(old_url)
        config_file = make_file(self, content=old_content)
        new_url = "http://2.2.2.2/"

        check_call(
            maasfixture.compose_rewrite_for_default_maas_url(
                config_file, new_url))

        new_content = top + self.make_rewritten_maas_local_settings(
            old_url, new_url)
        self.assertThat(config_file, FileContains(new_content.decode('ascii')))

    def test_configure_default_maas_url_sets_config_and_restarts(self):
        self.patch(kvmfixture, 'run_command', mock.MagicMock())
        fixture = self.make_kvm_fixture()
        fixture.direct_ip = '127.99.88.77'
        rewrite_command = maasfixture.compose_rewrite_for_default_maas_url(
            '/etc/maas/maas_local_settings.py',
            "http://%s/MAAS" % fixture.direct_ip)

        maas_fixture = self.make_maas_fixture(fixture)
        maas_fixture.configure_default_maas_url()

        self.assertEqual(
            [
                mock.call(['sudo'] + rewrite_command, check_call=True),
                mock.call(
                    ['sudo', 'service', 'apache2', 'restart'],
                    check_call=False),
            ],
            fixture.run_command.mock_calls)

    def test_configure_default_maas_url_retries_if_restart_fails(self):
        self.patch(kvmfixture, 'run_command', mock.MagicMock())
        fixture = self.make_kvm_fixture()
        # Configure fixture.run_command to simulate a transient failure when
        # restarting apache.
        returns = [
            # Simluate response for the 'config rewrite' command.
            (0, 'stdout', 'stderr'),
            # Simulate first response for the 'apache2 restart' command:
            # failure.
            (1, 'stdout', 'an error!'),
            # Simulate second response for the 'apache2 restart' command:
            # success.
            (0, 'stdout', 'stderr'),
        ]

        def side_effect(*args, **kwargs):
            return returns.pop(0)

        fixture.run_command = mock.MagicMock(side_effect=side_effect)
        fixture.direct_ip = '127.99.88.77'

        self.patch(
            utils, 'retries', mock.MagicMock(return_value=[(0, 1), (0, 1)]))
        maas_fixture = self.make_maas_fixture(fixture)

        maas_fixture.configure_default_maas_url()

        # The first call to fixture.run_command is the rewrite
        # configuration command, ignore it since it's tested elsewhere
        # and we're only testing that the 'apache2 restart' command is
        # retried when it fails here.
        run_command_calls = fixture.run_command.mock_calls[1:]
        self.assertEqual(
            [
                # 2 'apache restart' calls.
                mock.call(
                    ['sudo', 'service', 'apache2', 'restart'],
                    check_call=False),
                mock.call(
                    ['sudo', 'service', 'apache2', 'restart'],
                    check_call=False),
            ],
            run_command_calls)

    def test_configure_default_maas_url_fails_if_apache_cannot_restart(self):
        self.patch(kvmfixture, 'run_command', mock.MagicMock())
        fixture = self.make_kvm_fixture()
        error_msg = self.getUniqueString()
        # Configure fixture.run_command to simulate a failure when
        # restarting apache.
        returns = chain([
            # Simluate response for the 'config rewrite' command.
            (0, 'stdout', 'stderr')],
            repeat((1, 'stdout', error_msg)))

        def side_effect(*args, **kwargs):
            return returns.next()

        fixture.run_command = mock.MagicMock(side_effect=side_effect)
        fixture.direct_ip = '127.99.88.77'

        self.patch(
            utils, 'retries', mock.MagicMock(return_value=[(0, 1), (0, 1)]))
        maas_fixture = self.make_maas_fixture(fixture)

        error = self.assertRaises(
            Exception, maas_fixture.configure_default_maas_url)
        self.assertIn(error_msg, text_type(error))

    def test_maas_admin_property_returns_maas_if_old_maas(self):
        # Pre-trusty MAAS servers use 'maas' to control the region controller.
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(fixture, 'run_command', mock.MagicMock(
            return_value=(1, b'stdout output', b'stderr output')))
        admin_cmd = maas_fixture.maas_admin
        self.assertEqual('maas', admin_cmd)
        create_call = fixture.run_command.call_args_list[0]
        args, kwargs = create_call
        command_line = args[0]
        self.assertEqual(
            ['which', 'maas-region-admin'],
            command_line[:2])

    def test_maas_admin_property_returns_maas_admin_region_if_new_maas(self):
        # Trusty and later MAAS servers use 'maas-region-admin' to control the
        # region controller.
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(fixture, 'run_command', mock.MagicMock(
            return_value=(0, b'stdout output', b'stderr output')))
        admin_cmd = maas_fixture.maas_admin
        self.assertEqual('maas-region-admin', admin_cmd)
        create_call = fixture.run_command.call_args_list[0]
        args, kwargs = create_call
        command_line = args[0]
        self.assertEqual(
            ['which', 'maas-region-admin'],
            command_line[:2])

    def test_create_maas_admin_creates_admin(self):
        fixture = self.make_kvm_fixture()

        maas_fixture = self.make_maas_fixture(fixture)
        maas_fixture._maas_admin = 'maas-region-admin'
        username, password = maas_fixture.create_maas_admin()

        create_call = fixture.run_command.call_args_list[0]
        args, kwargs = create_call
        command_line = args[0]
        self.assertEqual(
            ['sudo', 'maas-region-admin', 'createadmin'],
            command_line[:3])
        self.assertIn('--username=%s' % username, command_line)
        self.assertIn('--password=%s' % password, command_line)

    def test_query_api_key_returns_api_key(self):
        username = "maas-user"
        actual_api_key = "API KEY"
        fixture = self.make_kvm_fixture()
        fixture.run_command.return_value = (0, actual_api_key, 'stderr output')

        maas_fixture = self.make_maas_fixture(fixture)
        maas_fixture._maas_admin = 'maas-region-admin'
        api_key = maas_fixture.query_api_key(username)

        args, kwargs = fixture.run_command.call_args_list[0]
        command_line = args[0]
        self.assertEqual(
            ['sudo', 'maas-region-admin', 'apikey',
                '--username=%s' % username], command_line)
        self.assertEqual(actual_api_key, api_key)

    def test_get_maas_api_client_returns_client(self):
        actual_api_key = "actual:maas:key"
        self.patch(kvmfixture, 'run_command', mock.MagicMock(
            return_value=(0, b'stdout output', b'stderr output')))
        fixture = self.make_kvm_fixture()
        fixture.run_command.return_value = (
            0, actual_api_key, b'stderr output')
        maas_fixture = self.make_maas_fixture(fixture)
        client = maas_fixture.get_maas_api_client(actual_api_key)
        self.assertIsInstance(client, MAASClient)

    def test_install_maas_installs_integrates_with_install_packages(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)

        maas_fixture.install_maas()

        fixture.run_command.assert_has_call(
            mock.call(
                maas_fixture.kvm_fixture._make_apt_get_install_command(
                    ['language-pack-en']),
                check_call=True),
            )

    def test_install_maas_logs_maas_version(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        version = self.getUniqueString()
        mock_get_maas_version = mock.MagicMock(return_value=version)
        self.patch(maas_fixture, 'get_maas_version', mock_get_maas_version)
        mock_logging = mock.MagicMock()
        self.patch(maasfixture.logging, 'info', mock_logging)

        maas_fixture.install_maas()

        self.assertEqual([mock.call()], mock_get_maas_version.mock_calls)
        self.assertEqual(
            [
                mock.call("Installing MAAS (version %s)..." % version),
                mock.call("Done installing MAAS."),
            ],
            mock_logging.mock_calls)

    def test_install_maas_installs_only_once(self):
        fixture = self.make_maas_fixture()
        fixture.kvm_fixture.install_packages = mock.MagicMock()

        fixture.install_maas()
        interim_installs = list(
            fixture.kvm_fixture.install_packages.mock_calls)
        interim_runs = list(fixture.kvm_fixture.run_command.mock_calls)
        fixture.install_maas()
        final_installs = list(
            fixture.kvm_fixture.install_packages.mock_calls)
        final_runs = list(fixture.kvm_fixture.run_command.mock_calls)

        self.assertGreater(len(interim_installs), 0)
        self.assertEqual(interim_installs, final_installs)
        self.assertGreater(len(interim_runs), 0)
        self.assertEqual(interim_runs, final_runs)

    def test_get_maas_version_returns_maas_version(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        version = self.getUniqueString()
        policy = "Candidate: %s\n" % version
        self.patch(fixture, 'run_command', mock.MagicMock(
            return_value=(0, policy, b'stderr')))

        extracted_version = maas_fixture.get_maas_version()

        self.assertEqual(version, extracted_version)

    def test_log_connection_details_logs_info(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(
            maas_fixture, 'query_api_key',
            mock.MagicMock(return_value='fake:api:key'))
        self.patch(
            maas_fixture, 'wait_until_boot_images_scanned', mock.MagicMock())

        with maas_fixture:
            pass
        mock_logging = mock.MagicMock()
        self.patch(maasfixture.logging, 'info', mock_logging)

        maas_fixture.log_connection_details()

        self.assertEqual(
            [
                mock.call(
                    "MAAS server URL: http://%s/MAAS/ "
                    "username:%s, password:%s" % (
                        fixture.ip_address(),
                        maas_fixture.admin_user,
                        maas_fixture.admin_password,
                    )
                ),
                mock.call(
                    "SSH login: sudo ssh -i %s %s" % (
                        fixture.ssh_private_key_file,
                        fixture.identity(),
                    )
                ),
            ],
            mock_logging.mock_calls)

    def test_install_maas_installs_language_pack_followed_by_maas(self):
        # We need to do it in this order, or postgres (and other things) may
        # break during installation.
        fixture = self.make_kvm_fixture()
        fake_install = mock.MagicMock()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(fixture, 'install_packages', fake_install)

        maas_fixture.install_maas()

        self.assertEqual(
            [
                mock.call(['language-pack-en']),
                mock.call(['maas', 'maas-dhcp', 'maas-dns']),
            ],
            fake_install.mock_calls)

    def test_get_master_ng_uuid_returns_uuid(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(maas_fixture, 'install_packages', mock.MagicMock())
        self.patch(maas_fixture, 'admin_maas_client', mock.MagicMock())

        uuid = self.getUniqueString()
        response = [{'uuid': uuid}]
        call_mock = self.patch_MAASClient(
            maas_fixture.admin_maas_client, 'get', response)

        self.assertEqual(
            (uuid, [mock.call('/api/1.0/nodegroups/', op='list')]),
            (maas_fixture.get_master_ng_uuid(), call_mock.mock_calls))

    def test_check_cluster_connected_returns_if_cluster_connected(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(maas_fixture, 'install_packages', mock.MagicMock())
        self.patch(maas_fixture, 'admin_maas_client', mock.MagicMock())

        uuid = self.getUniqueString()
        response = [{'uuid': uuid}]
        self.patch_MAASClient(
            maas_fixture.admin_maas_client, 'get', response)

        self.assertEqual(None, maas_fixture.check_cluster_connected())

    def test_check_cluster_connected_fails_if_cluster_not_connected(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(maas_fixture, 'install_packages', mock.MagicMock())
        self.patch(maas_fixture, 'admin_maas_client', mock.MagicMock())
        self.patch(
            utils, 'retries', mock.MagicMock(return_value=[(0, 1)]))

        response = [{'uuid': 'master'}]
        call_mock = self.patch_MAASClient(
            maas_fixture.admin_maas_client, 'get', response)

        self.assertRaises(Exception, maas_fixture.check_cluster_connected)
        self.assertEqual(
            [mock.call('/api/1.0/nodegroups/', op='list')],
            call_mock.mock_calls)

    def test_configure_cluster_updates_dhcp_info(self):
        fixture = self.make_kvm_fixture()
        fixture.direct_ip = '192.168.2.0'
        network = netaddr.IPNetwork('192.168.2.0/24')
        fixture.direct_network = network
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(maas_fixture, 'install_packages', mock.MagicMock())
        self.patch(maas_fixture, 'admin_maas_client', mock.MagicMock())
        self.patch(
            utils, 'retries', mock.MagicMock(return_value=[(0, 1)]))

        uuid = self.getUniqueString()
        response = [{'uuid': uuid}]
        call_mock = self.patch_MAASClient(
            maas_fixture.admin_maas_client, 'get', response)
        put_mock = self.patch_MAASClient(
            maas_fixture.admin_maas_client, 'put', '')

        maas_fixture.configure_cluster()
        interface_url = '/api/1.0/nodegroups/%s/interfaces/%s/' % (
            uuid, 'eth1')
        last_ip = "%s" % netaddr.IPAddress(
            fixture.direct_network.last - 1)
        self.assertEqual(
            (
                [mock.call('/api/1.0/nodegroups/', op='list')],
                [mock.call(
                    interface_url,
                    ip=fixture.direct_ip,
                    interface="eth1",
                    subnet_mask="%s" % network.netmask,
                    broadcast_ip="%s" % network.broadcast,
                    router_ip=fixture.direct_ip,
                    management=(
                        '%s' % NODEGROUPINTERFACE_MANAGEMENT.DHCP_AND_DNS),
                    ip_range_low=fixture.direct_first_available_ip(),
                    ip_range_high=last_ip,
                    )],
            ),
            (call_mock.mock_calls, put_mock.mock_calls))

    def test_collect_logs_collects_logs(self):
        kvm_fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(kvm_fixture)

        maas_fixture.collect_logs()

        expected_calls = [
            mock.call(['sudo', 'cat', filename], check_call=False)
            for filename in maasfixture.LOG_FILES]
        self.assertEqual(
            expected_calls,
            kvm_fixture.run_command.mock_calls)

    def test_setUp_sets_admin_details(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        fixture.direct_ip = None
        user = 'maas-admin-user'
        password = 'maas-admin-password'
        self.patch(
            maas_fixture, 'create_maas_admin',
            mock.MagicMock(return_value=(user, password)))
        key = 'fake:api:key'
        self.patch(
            maas_fixture, 'query_api_key', mock.MagicMock(return_value=key))
        self.patch(
            maas_fixture, 'wait_until_boot_images_scanned', mock.MagicMock())

        with maas_fixture:
            pass

        client_key = "%s:%s:%s" % (
            maas_fixture.admin_maas_client.auth.consumer_token.key,
            maas_fixture.admin_maas_client.auth.resource_token.key,
            maas_fixture.admin_maas_client.auth.resource_token.secret)
        self.assertEqual(
            (user, password, key, key),
            (
                maas_fixture.admin_user,
                maas_fixture.admin_password,
                maas_fixture.admin_api_key,
                client_key,
            ))

    def test_setUp_calls_cleanup_methods(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(
            maas_fixture, 'query_api_key',
            mock.MagicMock(return_value='fake:api:key'))
        self.patch(
            maas_fixture, 'collect_logs', mock.MagicMock())
        self.patch(
            maas_fixture, 'wait_until_boot_images_scanned', mock.MagicMock())
        with maas_fixture:
            pass

        self.assertEqual(
            [
                [mock.call()],
            ],
            [
                maas_fixture.collect_logs.mock_calls,
            ])

    def test_configure_default_series_configures_series(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(maas_fixture, 'admin_maas_client', mock.MagicMock())
        series = self.getUniqueString()
        call_mock = self.patch_MAASClient(
            maas_fixture.admin_maas_client, 'post', '')

        maas_fixture.configure_default_series(series)

        self.assertEqual(
            [mock.call(
                '/api/1.0/maas/', op='set_config',
                name='commissioning_distro_series', value=series)],
            call_mock.mock_calls)

    def test_configure_http_proxy_configure_http_proxy(self):
        fixture = self.make_kvm_fixture()
        proxy_url = self.getUniqueString()
        maas_fixture = self.make_maas_fixture(fixture, proxy_url=proxy_url)
        self.patch(maas_fixture, 'admin_maas_client', mock.MagicMock())
        call_mock = self.patch_MAASClient(
            maas_fixture.admin_maas_client, 'post', '')

        maas_fixture.configure_http_proxy(proxy_url)

        self.assertEqual(
            [mock.call(
                '/api/1.0/maas/', op='set_config',
                name='http_proxy', value=proxy_url)],
            (call_mock.mock_calls))

    def test_setUp_calls_methods_if_direct_ip_set(self):
        fixture = self.make_kvm_fixture()
        fixture.direct_ip = '127.11.24.38'
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(
            maas_fixture, 'configure_cluster', mock.MagicMock())
        self.patch(
            maas_fixture, 'check_cluster_connected', mock.MagicMock())
        self.patch(
            maas_fixture, 'query_api_key',
            mock.MagicMock(return_value='fake:api:key'))
        # Patch methods that we want to check setUp calls.
        self.patch(
            maas_fixture, 'configure_default_maas_url', mock.MagicMock())
        self.patch(
            maas_fixture, 'check_cluster_connected', mock.MagicMock())
        self.patch(
            maas_fixture, 'configure_cluster', mock.MagicMock())
        self.patch(
            maas_fixture, 'configure_http_proxy', mock.MagicMock())
        self.patch(
            maas_fixture, 'import_maas_images', mock.MagicMock())
        self.patch(
            maas_fixture, 'log_connection_details', mock.MagicMock())
        self.patch(
            maas_fixture, 'wait_until_boot_images_scanned', mock.MagicMock())

        with maas_fixture:
            pass

        self.assertEqual(
            [
                [mock.call()],
                [mock.call()],
                [mock.call()],
                [mock.call(maas_fixture.proxy_url)],
                [mock.call(
                    series=maas_fixture.series,
                    architecture=maas_fixture.architecture,
                    simplestreams_filter=maas_fixture.simplestreams_filter)],
                [mock.call()],
            ],
            [
                maas_fixture.configure_default_maas_url.mock_calls,
                maas_fixture.check_cluster_connected.mock_calls,
                maas_fixture.configure_cluster.mock_calls,
                maas_fixture.configure_http_proxy.mock_calls,
                maas_fixture.import_maas_images.mock_calls,
                maas_fixture.log_connection_details.mock_calls,
            ])

    def test_setUp_does_not_configure_proxy_if_no_proxy(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture, proxy_url='')
        self.patch(
            maas_fixture, 'query_api_key',
            mock.MagicMock(return_value='fake:api:key'))
        self.patch(
            maas_fixture, 'configure_http_proxy', mock.MagicMock())
        self.patch(
            maas_fixture, 'wait_until_boot_images_scanned', mock.MagicMock())

        with maas_fixture:
            pass

        self.assertEqual(
            [[]],
            [
                maas_fixture.configure_http_proxy.mock_calls,
            ])

    def test_setUp_does_not_call_methods_if_direct_ip_unset(self):
        fixture = self.make_kvm_fixture()
        maas_fixture = self.make_maas_fixture(fixture)
        self.patch(
            maas_fixture, 'query_api_key',
            mock.MagicMock(return_value='fake:api:key'))
        # Patch methods that we want to check setUp doesn't call.
        self.patch(
            maas_fixture, 'configure_default_maas_url', mock.MagicMock())
        self.patch(
            maas_fixture, 'check_cluster_connected', mock.MagicMock())
        self.patch(
            maas_fixture, 'configure_cluster', mock.MagicMock())
        self.patch(
            maas_fixture, 'wait_until_boot_images_scanned', mock.MagicMock())

        with maas_fixture:
            pass

        self.assertEqual(
            [[], [], []],
            [
                maas_fixture.configure_default_maas_url.mock_calls,
                maas_fixture.check_cluster_connected.mock_calls,
                maas_fixture.configure_cluster.mock_calls,
            ])

    def test_import_maas_images_imports_pxe_images(self):
        kvm_fixture = self.make_kvm_fixture()
        proxy_url = self.getUniqueString()
        maas_fixture = self.make_maas_fixture(kvm_fixture, proxy_url)
        series = self.getUniqueString()
        arch = self.getUniqueString()
        filter = self.getUniqueString()
        returned_arches = ['amd64', 'i386']
        mock_mipf_arch_list = mock.MagicMock(return_value=returned_arches)
        self.patch(
            utils, 'mipf_arch_list', mock_mipf_arch_list)

        maas_fixture.import_maas_images(series, arch, filter)

        self.assertEqual(
            mock.call([
                'sudo',
                'http_proxy=%s' % proxy_url,
                'https_proxy=%s' % proxy_url,
                'maas-import-pxe-files',
            ], check_call=True),
            kvm_fixture.run_command.mock_calls[-1])
        self.assertEqual([mock.call(arch)], mock_mipf_arch_list.mock_calls)

    def test_dump_data_calls_dumpdata(self):
        maas_fixture = self.make_maas_fixture()
        maas_fixture._maas_admin = 'maas-region-admin'
        self.patch(
            maas_fixture.kvm_fixture, 'run_command',
            mock.MagicMock(return_value=(0, '', '')))
        maas_fixture.dump_data()
        self.assertIn(
            mock.call([
                'sudo', 'maas-region-admin', 'dumpdata',
                'metadataserver.NodeCommissionResult']),
            maas_fixture.kvm_fixture.run_command.mock_calls)

    def test_wait_until_boot_images_scanned_times_out_eventually(self):
        maas_fixture = self.make_maas_fixture()
        self.patch(
            maas_fixture, 'list_boot_images',
            mock.MagicMock(return_value=[]))
        self.patch(
            utils, 'retries', mock.MagicMock(return_value=[(0, 1), (0, 1)]))

        exception = self.assertRaises(
            NoBootImagesError, maas_fixture.wait_until_boot_images_scanned)
        message = text_type(exception)
        self.assertIn(
            "Boot image download timed out: cluster reported no images.",
            message)

    def test_wait_until_boot_images_scanned_returns_with_error(self):
        maas_fixture = self.make_maas_fixture()
        self.patch(
            maas_fixture, 'list_boot_images',
            mock.MagicMock(return_value=['non-empty1', 'non-empty2']))

        self.assertEqual(None, maas_fixture.wait_until_boot_images_scanned())

    def test_list_boot_images_lists_images(self):
        maas_fixture = self.make_maas_fixture()
        response = [{'fake-response': self.getUniqueString()}]
        self.patch(maas_fixture, 'admin_maas_client', mock.MagicMock())
        call_mock = self.patch_MAASClient(
            maas_fixture.admin_maas_client, 'get', response)
        uuid = self.getUniqueString()
        self.patch(
            maas_fixture, 'get_master_ng_uuid',
            mock.MagicMock(return_value=uuid))

        self.assertEqual(
            (
                response,
                [mock.call('/api/1.0/nodegroups/%s/boot-images/' % uuid)],
            ),
            (maas_fixture.list_boot_images(), call_mock.mock_calls))
