/*
 * mad2pl:  Converts a madman database into its respective playlists.
 * 
 */

#include <iostream>
 using std::cout;
 using std::cerr;
 using std::endl;

#include <fstream>
 using std::ifstream;
 using std::ofstream;

#include <string>
 using std::string;

#include <map>
 using std::map;
 
#include <stdlib.h>

// Main program
int main( int argc, char* argv[] ) {

   //---------------------------------------------------------------------------------------
   // First, make sure we have the proper parameters.  
   // argv[0] - exe name.
   // argv[1] - the *.mad file we'll be parsing.
   // argv[2] - the directory we'll be putting playlist files in.
   if (argc != 3 && argc != 4) {
      cout << "Mad2pl:" << endl;
      cout << " Parses a Madman database file, writing a single playlist file for every" << endl;
      cout << " playlist Madman knows about." << endl;
      cout << endl; 
      cout << "Usage:" << endl;
      cout << " " << argv[0] << " <.mad file> <playlist directory>" << endl;
      cout << endl;
      cout << ".mad file:           The *.mad file opened up when you start Madman" << endl;
      cout << "playlist directory:  The folder to write the playlists into." << endl;
      cout << endl;
      cout << "Options:" << endl;
      cout << " -v        Verbose output." << endl;
      cout << endl;
      cout << endl;
      exit(1);
   }

   //---------------------------------------------------------------------------------------
   // Store inputs.
   string sMadFileName;
   string sPlaylistDir;
   bool bVerbose;
   if ( argc == 3 ) {
      bVerbose = false;
      sMadFileName = argv[1];
      sPlaylistDir = argv[2];
   } else if ( argc == 4 ) {
      if ( strcmp( argv[1], "-v" ) == 0 ) {
         bVerbose == true;
      } else {
         cerr << endl;
         cerr << " Unrecognized option: " << argv[1] << endl;
         cerr << endl;
         exit(1);
      }
         
      sMadFileName = argv[2];
      sPlaylistDir = argv[3];
   }

   //---------------------------------------------------------------------------------------
   // Now, open up the mad file for input and error check.
   ifstream MadFile;
   MadFile.open( sMadFileName.c_str() , std::ios::in );
   if ( !MadFile.is_open() ) {
      cerr << "Could not open the .mad file: (" << sMadFileName << ") for processing." << endl;
      cerr << "Please check the filename and try again.";
      exit(1);
   }

   bool bDone = false;
   char szLine[1024];
   string sLine;
   
   //---------------------------------------------------------------------------------------
   // Parse through the file until we get the start of the songs, get to the end or hit an error.
   while ( !bDone && !MadFile.eof() && MadFile.good() ) {
      

      // Read a line, error check.
      MadFile.getline(szLine,1024);
      sLine = szLine;
      if ( !MadFile.good() ) {
         cerr << endl;
         cerr << "There was an unidentified error while reading from the .mad file," << endl;
         cerr << "before finding the song list." << endl;
         cerr << endl;
         exit(1);
      }

      // Look for a line saying '<songs>'.
      if ( sLine.find("<songs") != string::npos ) {
         bDone = true;
      }
      
   }

   //---------------------------------------------------------------------------------------
   // If we're to this point and we didn't find the <songs> line, we should exit.
   if ( !bDone ) {
      cerr << "Error:  Couldn't find the <songs> line in the .mad file...";
      cerr << "Is this really a madman database?" << endl;
      cerr << endl;
      exit(1);
   }
   bDone = false;

   //---------------------------------------------------------------------------------------
   // Read all the songs in the database... they're cross-referenced by a number called
   //  "unique_id".
   map<int, string> Songs;
   while ( !bDone && !MadFile.eof() && MadFile.good() ) {
      
      // Read a line, error check.
      MadFile.getline(szLine,1024);
      sLine = szLine;
      if ( !MadFile.good() ) {
         cerr << endl;
         cerr << "There was an unidentified error while reading from the .mad file," << endl;
         cerr << "before finding the song list." << endl;
         cerr << endl;
         exit(1);
      }

      // Look for a line saying '</songs>'.
      if ( sLine.find("</songs") != string::npos ) {
         bDone = true;
      } else {

         // Find the unique ID of this song.
         int iStart = sLine.find("uniqueid=\"") + 10;
         int iEnd = sLine.find("\"",iStart) - 1;
         string sUniqueId = sLine.substr(iStart, iEnd-iStart+1);

         // Find the filename of this song.
         iStart = sLine.find("filename=\"") + 10;
         iEnd = sLine.find("\"",iStart) - 1;
         string sFileName = sLine.substr(iStart, iEnd-iStart+1);

         // If the filename contains an ampersand, HTML-encoded, replace it with a single
         //  ampersand.
         if ( sFileName.find("&amp;") != string::npos ) {
            int iPos = sFileName.find("amp;");
            sFileName = sFileName.substr(0, iPos-1) + "&" + sFileName.substr(iPos+4);
         }
         
         // Convert the string to an integer.
         int iId = atoi( sUniqueId.c_str() );
         
         // Put this song into the map.
         Songs[iId] = sFileName;
      }
      
   }
   
   if (bVerbose) {
      cout << "Found and loaded " << Songs.size() << " songs." << endl;
   }
   
   //---------------------------------------------------------------------------------------
   // If we're to this point and we didn't find the </songs> line, we should exit.
   if ( !bDone ) {
      cerr << "Error:  Couldn't find the </songs> line in the .mad file...";
      cerr << "Is this really a madman database?" << endl;
      cerr << endl;
      exit(1);
   }
   bDone = false;
      
   //---------------------------------------------------------------------------------------
   // Now, look for playlists.  They'll be in the form '<rendering>'.  There might be
   //  lines that say <rendering/>, which are empty playlists.  We'll skip those.
   // Then, we'll read every <song unique_id="" /> in the list, writing out a line for every 
   //  song in there in the proper order.
   while ( !bDone && !MadFile.eof() && MadFile.good() ) {

      // Read a line.
      bool bBegin = false;
      bool bEnd = false;
      ofstream OutFile;
      int iSongCount = 0;
      string sPlName;
      while ( !bBegin && !MadFile.eof() && MadFile.good() ) {

         // Read a line, error check.
         MadFile.getline(szLine,1024);
         sLine = szLine;

         if ( !MadFile.good() ) {
            cerr << endl;
            cerr << "There was an unidentified error while reading from the .mad file," << endl;
            cerr << "before finding the beginning of a playlist... sorry." << endl;
            cerr << "Last line read: " << sLine << endl;
            cerr << endl;
            exit(1);
         }

         // Check to see if this line is declaring a new song_set_node.  This is our playlist
         //  name.
         if ( sLine.find("<song_set_node") != string::npos ) {

            // Parse the name out.
            int iStart = sLine.find("name=\"") + 6;
            int iEnd = sLine.find("\"",iStart) - 1;
            sPlName = sLine.substr(iStart, iEnd-iStart+1);

            // Verbose output.
            if ( bVerbose ) { cout << "Found playlist " << sPlName << endl; } 

         }

         // Look for a line saying '<rendering>'.
         if ( sLine.find("<rendering>") != string::npos ) {
            
            // Make sure the playlist directory ends in a /.
            if ( sPlaylistDir[sPlaylistDir.size()-1] != '/') {
               sPlaylistDir += "/";
            }  

            // Open up a file for writing, error check.
            string sPlFileName = sPlaylistDir + sPlName;
            OutFile.open(sPlFileName.c_str(), std::ios::out );
            if ( !OutFile.good() ) {
               cerr << " Error:  could not open file " << sPlFileName << "." << endl;
               cerr << endl;
               exit(1);
            }
            iSongCount = 0;

            bBegin = true;
         }

         // Look for a line saying '<history>', which means we're done.
         if ( sLine.find("<history>") != string::npos ) {
            bBegin = true;
            bDone = true;
         }
      }
      

      // Read loop... when we find </rendering>, this playlist is done.
      while (!bDone && !bEnd && !MadFile.eof() && MadFile.good()) {
         
         // Read a line, error check.
         MadFile.getline(szLine,1024);
         sLine = szLine;
         if ( !MadFile.good() ) {
            cerr << endl;
            cerr << "There was an unidentified error while reading from the .mad file," << endl;
            cerr << "before finding the end of a playlist... sorry." << endl;
            cerr << "Last line read: " << sLine << endl;
            cerr << endl;
            exit(1);
         }

         // Look for a song.  If we find it, parse out the unique_id, and write out the 
         //  corresponding filename.
         if ( sLine.find("<song unique_id=\"") != string::npos ) {
            
            // Parse the unique_id out.
            int iStart = sLine.find("unique_id=\"") + 11;
            int iEnd = sLine.find("\"",iStart) - 1;
            string sId = sLine.substr(iStart, iEnd-iStart+1);
            
            // Convert to integer.
            int iId = atoi( sId.c_str() );

            // Write the corresponding filename out to the output file.
            iSongCount++;
            OutFile << Songs[iId] << endl;
         }

         // If we find the </rendering> line, we're done.
         if ( sLine.find("</rendering") != string::npos ) {
            bEnd = true;
            if (bVerbose) {
               cout << " +- wrote " << iSongCount << " songs to playlist." << endl;
            }
            OutFile.close();
         }
      }
      
   }
   
   
   //---------------------------------------------------------------------------------------
   // Exit gracefully.
   MadFile.close();
   exit(0);
}
