/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

#ifndef Exceptions_H
#define Exceptions_H

#ifndef machine_H
#include "machine.h"
#endif

#ifndef Log_H
#include "Log.h"
#endif

void Panic(const char*);
void Panic(const char *msg,int line,const char *file,const char *proc);

// General purpose exception
// Other exceptions must be defined in the class that throw them

// Misc. errors

class Exception : public exception {


public:
	virtual const char *what() const THROW_NOTHING() {  return what_.c_str(); }
	Exception(const string&);
	~Exception() THROW_NOTHING();
	virtual bool retryOnServer() const        { return false; }
	virtual bool retryOnClient() const        { return false; }
	virtual bool terminateApplication() const { return false; }

	static bool throwing();
	static void exceptionStack(ostream&);

protected:
	void reason(const string&);
	Exception();
private:
	string what_;
	//SaveStatus save_;
	Exception* next_;
};

class SeriousBug : public Exception {
public:
	SeriousBug(const string& w) : Exception(string("Serious Bug:") + w) {}
	SeriousBug(const string&,const string&);
	SeriousBug(const string&,int);
};

class TooManyRetries : public Exception {
public:
	TooManyRetries(const int);
};

class TimeOut : public Exception {
public:
	TimeOut(const unsigned long);
};

class FailedSystemCall : public Exception {
public:
	FailedSystemCall(const string&);
	FailedSystemCall(const char*,int,const char*,const char*,int);
};

class AssertionFailed : public Exception {
public:
	AssertionFailed(const string&);
	AssertionFailed(const char*,int,const char*,const char*);
};

class BadParameter : public Exception {
public:
	BadParameter(const string& s);
};

class NotImplemented : public Exception {
public:
	NotImplemented(int,const char*,const char*);
};

class Stop : public Exception {
public:
	Stop(const string&);
};

class Abort : public Exception {
public:
	Abort(const string&);
};

class Cancel : public Exception {
public:
	Cancel(const string&);
};

class UserError : public Exception {
public:
	UserError(const string&);
	UserError(const string&,const string&);
	UserError(const string&,int);
};

class OutOfRange : public Exception {
public:
	OutOfRange(unsigned long long, unsigned long long);
};

// File errors

class FileError : public Exception { 
protected:
	FileError(const string&);
	FileError()					{  }
};

class CantOpenFile : public FileError { 
	bool retry_;
	virtual bool retryOnServer() const { return retry_; }
public:
	CantOpenFile(const string&,bool retry = false);
};

class WriteError : public FileError { 
public:
	WriteError(const string&);
};

class ReadError : public FileError { 
public:
	ReadError(const string&);
};

class ShortFile : public ReadError { 
public:
	ShortFile(const string&);
};

// ObjectStore

class Ostore : public Exception {
public:
	Ostore(const string&);
};

// =======================================

inline void SysCall(long long code,const char *msg,int line,const char *file,
	const char *proc)
{
	if(code<0)
		throw FailedSystemCall(msg,line,file,proc,errno);
}

inline void ThrCall(int code,const char *msg,int line,const char *file,
	const char *proc)
{
	if(code != 0) // Threads return errno in return code
		throw FailedSystemCall(msg,line,file,proc,code);
}

inline void Assert(int code,const char *msg,int line,const char *file,
	const char *proc)
{
	if(code != 0)
		throw AssertionFailed(msg,line,file,proc);
}

inline void Panic(int code,const char *msg,int line,const char *file, 
	const char *proc)
{
	if(code != 0) 
		Panic(msg,line,file,proc);
}

//--------------------------------------------------------------
// For compatibility
//--------------------------------------------------------------
class OutOfMemory : public Exception {
	virtual bool terminateApplication() const { return true; }
	virtual const char *what() const THROW_NOTHING() {  return "OutOfMemory"; }
public:
	OutOfMemory();

};

#define THRCALL(a) ThrCall(a,#a,__LINE__,__FILE__,__FUNCTION__)
#define SYSCALL(a) SysCall(a,#a,__LINE__,__FILE__,__FUNCTION__)
#define ASSERT(a)  Assert(!(a),#a,__LINE__,__FILE__,__FUNCTION__)
#define PANIC(a)   Panic((a),#a,__LINE__,__FILE__,__FUNCTION__)
#define NOTIMP     throw NotImplemented(__LINE__,__FILE__,__FUNCTION__)


#endif
