/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file OdaDecoder.cc
    \brief Implementation of the Template class OdaDecoder.
    
    Magics Team - ECMWF 2004
    
    Started: Fri 16-Jan-2004
    
    Changes:
    
*/

#include "oda.h"
extern "C" {
#include "odbcapi.h"
}
#include "Exceptions.h" //for oda


#include "../common/Timer.h"
#include "OdaDecoder.h"

#include "TextVisitor.h"


OdaGeoDecoder::OdaGeoDecoder() 
{
}


OdaGeoDecoder::~OdaGeoDecoder() 
{
	
}

/*!
 Class information are given to the output-stream.
*/		
void OdaGeoDecoder::print(ostream& out)  const
{
	out << "OdaGeoDecoder[";
        OdaGeoDecoderAttributes::print(out);
        out << "]";
}

double degrees(double val) 
{
	return val;
}

double radians(double val) 
{
	return val*180/3.14;
}

typedef double (*Converter)(double);

map<string, Converter> converters_;


void OdaGeoDecoder::decode(const Transformation& transformation)
{    
	if ( !empty() ) 
		return;  

	try
	{
		odb_start();
		odb::Reader oda(path_);	
		odb::Reader::iterator it = oda.begin();
	
		size_t latIndex = it->columns().columnIndex(latitude_);
		size_t lonIndex = it->columns().columnIndex(longitude_);
		size_t valueIndex = it->columns().columnIndex(value_);	
	
		Log::info() <<  "Indices: " << latIndex << " " << lonIndex << " " << valueIndex << endl;
	
		unsigned int row=0;
	
		for(; it != oda.end(); ++it) 
		{			
			double lat = (*it)[latIndex];
			double lon = (*it)[lonIndex];
			position(lat, lon);
	
			if ( transformation.in(lon, lat)  ) 
			{
				statistics_.push_back((*it)[valueIndex]);
				push_back(GeoPoint(lon, lat, (*it)[valueIndex]));
	
				row++;
				if (nb_rows_ != -1 && row >= nb_rows_)
				{
					break;
				}
			}
		}

		Log::info() <<  "Number of rows: " << row << endl;

		computeStats();
	}
	catch(Exception e)
	{
		Log::error() << "Failed to read ODB data: " << e.what();
		return;
	}
}

void OdaGeoDecoder::visit(Transformation& transformation) {

	return;	

	cout << "SET AUTOMATIC AXIS" << endl;
	decode();
	if ( empty() ) return;
	double minx = front().x_;
	double maxx = front().x_;
	double miny = front().y_;
	double maxy = front().y_;

	for (iterator point = begin(); point != end(); ++point) {
		if ( minx > point->x_ ) minx = point->x_;
		if ( miny > point->y_ ) miny = point->y_;
		if ( maxx < point->x_ ) maxx = point->x_;
		if ( maxy < point->y_ ) maxy = point->y_;
	}

	transformation.setMinX(minx);
	transformation.setMinY(miny);
	transformation.setMaxX(maxx);
	transformation.setMaxY(maxy);
}
void OdaGeoDecoder::decode()
{    
	if ( !empty() ) 
		return;  

	odb_start();
    	odb::Reader oda(path_);

	odb::Reader::iterator it = oda.begin();

	size_t latIndex = it->columns().columnIndex(latitude_);
	size_t lonIndex = it->columns().columnIndex(longitude_);
	size_t valueIndex = it->columns().columnIndex(value_);	
  
	Log::info() <<  "Indices: " << latIndex << " " << lonIndex << " " << valueIndex << endl;

	unsigned int row=0;

	for (; it != oda.end(); ++it) 
	{			
		double lat = (*it)[latIndex];
		double lon = (*it)[lonIndex];
		position(lat, lon);
		
		statistics_.push_back((*it)[valueIndex]);
		push_back(GeoPoint(lon, lat, (*it)[valueIndex]));

		row++;
		if (nb_rows_ != -1 && row >= nb_rows_)
		{
				break;
		}
		
	}

	computeStats();

	Log::info() <<  "Number of rows: " << row << endl;
}
void OdaGeoDecoder:: visit(TextVisitor& title)
{
	if ( !title_.empty() )
		title.add(new TextEntry(title_));
	title.add(new TextEntry("OdbDatabase: " + path_));

	if(info("points").empty())
	{
		title.add(new TextEntry(" No points found " ));		
	}
	else
	{
		title.add(new TextEntry("Min: " + info("min") + "  Max: " + info("max") + "    ( " + info("points") + " points) " ));
	}

	/*if ( statistics_.empty() ) {
		title.add(new TextEntry(" No points found " ));
	}
	else {
		double min = *std::min_element(statistics_.begin(), statistics_.end());
		double max = *std::max_element(statistics_.begin(), statistics_.end());
		title.add(new TextEntry("Min: " + tostring(min) + "  Max: " + tostring(max) + "    ( " + tostring(statistics_.size()) + " points) " ));
	}*/
}

static 
double speed(double x, double y)
{
	return sqrt(x*x+ y*y);
}

void OdaGeoDecoder::customisedPoints(const std::set<string>&, CustomisedPointsList& list)
{
     	odb_start();	   
     	odb::Reader oda(path_);

	odb::Reader::iterator it = oda.begin();
		
	size_t latIndex = it.columns().columnIndex(latitude_);
	size_t lonIndex = it.columns().columnIndex(longitude_);
	size_t valueIndex = value_.empty() ? -1 :  it.columns().columnIndex(value_);	
	size_t xIndex = it.columns().columnIndex(x_);	
	size_t yIndex = it.columns().columnIndex(y_);	
			
	Log::info() <<  "Indices: " << latIndex << " " << lonIndex << " " << valueIndex << endl;
	
	unsigned int row=0;
		
	for (; it != oda.end(); ++it, row++) 
	{			
		if(nb_rows_ != -1 && row >= nb_rows_)
		{
			break;
		}

		double lat = (*it)[latIndex];
		double lon = (*it)[lonIndex];
		position(lat, lon);
					
		CustomisedPoint* point = new CustomisedPoint();		
		point->longitude(lon);
		point->latitude(lat);
		(*point)["x_component"] = (*it)[xIndex];
		(*point)["y_component"] = (*it)[yIndex];
		double val =  (valueIndex != -1) ? (*it)[valueIndex] : speed((*it)[xIndex],  (*it)[yIndex]);
		(*point)["colour_component"] = val;
		list.push_back(point);	
		statistics_.push_back(val);	
	}	

	computeStats();	

	Log::info() <<  "Number of rows: " << row << endl;
	
}

void OdaGeoDecoder::position(double& lat, double& lon)
{
	if ( converters_.empty() ) {
		converters_["degrees"] = &degrees;
		converters_["radians"] = &radians;
	}


	map<string, Converter>::iterator converter = converters_.find(lowerCase(unit_));

	if (converter == converters_.end() ) {
		Log::warning() << "odb_coordinates_unit:" << unit_ << " is not valid -->Change back to default:degrees" << endl;
		converter = converters_.find("degrees");
	}
	lat = (*converter->second)(lat);
	lon = (*converter->second)(lon);

}

void OdaGeoDecoder::customisedPoints(const Transformation& transformation, const std::set<string>&, CustomisedPointsList& list)
{

     	odb_start();	   
     	odb::Reader oda(path_);

     	odb::Reader::iterator it = oda.begin();

	size_t latIndex = it.columns().columnIndex(latitude_);
	size_t lonIndex = it.columns().columnIndex(longitude_);
	size_t valueIndex = value_.empty() ? -1 :  it.columns().columnIndex(value_);	
	size_t xIndex = it.columns().columnIndex(x_);	
	size_t yIndex = it.columns().columnIndex(y_);	
		
	Log::info() <<  "Indices: " << latIndex << " " << lonIndex << " " << valueIndex << endl;

	unsigned int row=0;
		
	for (; it != oda.end(); ++it) 
	{
		CustomisedPoint* point = new CustomisedPoint();		
		double lat = (*it)[latIndex];
		double lon = (*it)[lonIndex];
		position(lat, lon);
		if ( transformation.in(lon, lat)  ) 
		{							  
			point->longitude(lon);
			point->latitude(lat);
			double val =  (valueIndex != -1) ? (*it)[valueIndex] : speed((*it)[xIndex],  (*it)[yIndex]);
			(*point)["x_component"] = (*it)[xIndex];
			(*point)["y_component"] = (*it)[yIndex];
			(*point)["colour_component"] =val;
			list.push_back(point);	
			statistics_.push_back(val);
		
			row++;
			if(nb_rows_ != -1 && row >= nb_rows_)
			{
				break;
			}
		}
	}

	computeStats();	 

	Log::info() <<  "Number of rows: " << row << endl;		
}

void OdaGeoDecoder::computeStats()
{
	if(statistics_.size() == 0)
	{
		setInfo("min","");
    		setInfo("max","");
		setInfo("avg","");
    		setInfo("points","");
		return;
	}	

 	double min = *(std::min_element(statistics_.begin(), statistics_.end()));
	double max = *(std::max_element(statistics_.begin(), statistics_.end()));
	
	setInfo("min", tostring(min));
    	setInfo("max", tostring(max));

	if(statistics_.size() > 0)
	{	
		double mean = std::accumulate(statistics_.begin(), statistics_.end(), 0.0) / statistics_.size();
		setInfo("avg", tostring(mean));

		if(statistics_.size() >= 1)
		{					
  		 	double v,m2=0.,m3=0.,m4=0.;
			
			for(vector<double>::const_iterator it=statistics_.begin(); it != statistics_.end(); it++)
			{
				v=(*it)-mean;
				m2+=v*v;
				m3+=v*v*v;
				m4+=v*v*v*v;
			}
			
			m2/=statistics_.size();
			m3/=statistics_.size();
			m4/=statistics_.size();
			
			double std=sqrt(m2);
			setInfo("stdev",tostring(std));

			if(m2 != 0)
			{
				double skew=m3/(std*std*std);
				double kurt=m4/(m2*m2)-3.;
				setInfo("skewness",tostring(skew));
				setInfo("kurtosis",tostring(kurt));
			}
			else
			{
				setInfo("skewness","-");		
				setInfo("kurtosis","-");
			}
		}
		else
		{	
			setInfo("stdev","-");
			setInfo("skewness","-");		
			setInfo("kurtosis","-");
		}
	}

   	setInfo("points", tostring(statistics_.size()));
	
	statistics_.clear();	
}

OdaXYDecoder::OdaXYDecoder() 
{
}


OdaXYDecoder::~OdaXYDecoder() 
{
	
}

/*!
 Class information are given to the output-stream.
*/		
void OdaXYDecoder::print(ostream& out)  const
{
	out << "OdaXYDecoder[";
    OdaXYDecoderAttributes::print(out);
    out << "]";
}

void OdaXYDecoder::customisedPoints(const Transformation& transformation, const std::set<string>&, CustomisedPointsList& list)
{
}

void OdaXYDecoder::customisedPoints(const std::set<string>&, CustomisedPointsList& list)
{
}

void OdaXYDecoder:: visit(TextVisitor& title)
{
	if ( !title_.empty() )
		title.add(new TextEntry(title_));
	title.add(new TextEntry("OdbDatabase: " + path_));
	if ( statistics_.empty() ) {
		title.add(new TextEntry(" No points found " ));
	}
	else {
		double min = *std::min_element(statistics_.begin(), statistics_.end());
		double max = *std::max_element(statistics_.begin(), statistics_.end());
		title.add(new TextEntry("Min: " + tostring(min) + "  Max: " + tostring(max) + "    ( " + tostring(statistics_.size()) + " points) " ));
	}
}
void OdaXYDecoder::decode(const Transformation& transformation)
{
    if ( !empty() ) 
		return;  

	odb_start();
    	odb::Reader oda(path_);

	odb::Reader::iterator it = oda.begin();

	size_t latIndex = it->columns().columnIndex(x_);
	size_t lonIndex = it->columns().columnIndex(y_);
	size_t valueIndex = value_.empty() ? -1 :  it.columns().columnIndex(value_);	
  
	Log::info() <<  "Indices: " << latIndex << " " << lonIndex << " " << valueIndex << endl;

	unsigned int row=0;

	for (; it != oda.end(); ++it) 
	{			
		double lat = (*it)[latIndex];
		double lon = (*it)[lonIndex];
		

		if ( transformation.in(lon, lat)  ) 
		{
			double val =0;
			if (valueIndex!=-1) {
				val = (*it)[valueIndex];
			}
			push_back(UserPoint(lon, lat, val));
			statistics_.push_back(val);

			row++;
			if (nb_rows_ != -1 && row >= nb_rows_)
			{
				break;
			}
		}
	}
 
	Log::info() <<  "Number of rows: " << row << endl;

}

void OdaXYDecoder::decode()
{
    if ( !empty() ) 
		return;  

	odb_start();
    	odb::Reader oda(path_);

	odb::Reader::iterator it = oda.begin();

	size_t latIndex = it->columns().columnIndex(x_);
	size_t lonIndex = it->columns().columnIndex(y_);
	size_t valueIndex = value_.empty() ? -1 :  it.columns().columnIndex(value_);	
  
	Log::info() <<  "Indices: " << latIndex << " " << lonIndex << " " << valueIndex << endl;

	unsigned int row=0;

	for (; it != oda.end(); ++it) 
	{			
		double lat = (*it)[latIndex];
		double lon = (*it)[lonIndex];
		

		
			double val =0;
			if (valueIndex!=-1) {
				val = (*it)[valueIndex];
			}
			statistics_.push_back(val);
			push_back(UserPoint(lon, lat, val));

			row++;
			if (nb_rows_ != -1 && row >= nb_rows_)
			{
				break;
			}
		
	}
 
	Log::info() <<  "Number of rows: " << row << endl;

}
void OdaXYDecoder::visit(Transformation& transformation) {
	cout << "SET AUTOMATIC AXIS" << endl;
	decode();
	if ( empty() ) return;
	double minx = front().x_;
	double maxx = front().x_;
	double miny = front().y_;
	double maxy = front().y_;

	for (iterator point = begin(); point != end(); ++point) {
		if ( minx > point->x_ ) minx = point->x_;
		if ( miny > point->y_ ) miny = point->y_;
		if ( maxx < point->x_ ) maxx = point->x_;
		if ( maxy < point->y_ ) maxy = point->y_;
	}

	transformation.setMinX(minx);
	transformation.setMinY(miny);
	transformation.setMaxX(maxx);
	transformation.setMaxY(maxy);

}
