/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file IsoShading.h
    \brief Definition of the Template class IsoShading.
    
    Magics Team - ECMWF 2004
    
    Started: Tue 9-Mar-2004
    
    Changes:
*/

#ifndef IsoShading_H
#define IsoShading_H

#include "magics.h"
#include "Factory.h"
#include "Translator.h"

#include "IsoShadingAttributes.h"
#include "Colour.h"
#include "LevelSelection.h"
#include "Polyline.h"
#include "MatrixHandler.h"
#include "LegendVisitor.h"
namespace magics {

template <class P>
class NoIsoShading  {

public:
	NoIsoShading() : none_("none"){}
	virtual ~NoIsoShading() {}
	virtual void set(const map<string, string>&) {}
	virtual void set(const XmlNode&) {}
	virtual void toxml(ostream&, int = 0) const {}
	virtual bool accept(const string&) { return true;}
    
	virtual void operator()(MatrixHandler<P>&, BasicGraphicsObjectContainer&) 
			{  }
		virtual void operator()(Data<P>&, BasicGraphicsObjectContainer&) 
				{  }
 
    
    virtual NoIsoShading<P>* clone() {
		NoIsoShading<P>* object = new NoIsoShading<P>();
	    return object;
	}
	
	void close(const MatrixHandler<P>&) const {}
	virtual bool shadingMode() { return false; }
	virtual Colour colour(double ) { return none_; }
	virtual Colour  leftColour(double ) { return none_; }
	virtual Colour rightColour(double ) { return none_; }
	virtual double  leftRange(double) { return 0; }
	virtual double rightRange(double) { return 0; }
	virtual bool operator()(const LevelSelection&) { return false; }
	virtual void visit(LegendVisitor&) {} 
	virtual void operator()(Polyline*) const {}
	
protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream& out) const { out << "No Shading"; } 
     Colour none_;
private:
    //! Copy constructor - No copy allowed
	NoIsoShading(const NoIsoShading<P>&);
    //! Overloaded << operator to copy - No copy allowed
	NoIsoShading& operator=(const NoIsoShading<P>&);
	// -- Friends
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const NoIsoShading<P>& p)
		{ p.print(s); return s; }

};

template <class P>
class IsoShading: public NoIsoShading<P>, public IsoShadingAttributes<P> {

public:
	IsoShading();
	virtual ~IsoShading();
	void set(const map<string, string>& map ) { IsoShadingAttributes<P>::set(map); }
	void set(const XmlNode& node ) { IsoShadingAttributes<P>::set(node); }
	virtual void operator()(Polyline*) const;
	virtual void toxml(ostream& out, int tab = 0) const { IsoShadingAttributes<P>::toxml(out, tab); }
		virtual bool accept(const string&) { return true;}
	virtual NoIsoShading<P>* clone() const {
		IsoShading<P>* object = new IsoShading<P>();
		object->copy(*this);
	    return object;
	}
	
	virtual void operator()(MatrixHandler<P>& data, BasicGraphicsObjectContainer& parent) 
		{ (*this->technique_)(data, parent); }
	virtual void operator()(Data<P>& data, BasicGraphicsObjectContainer& parent) 
			{ (*this->technique_)(data, parent); }
	virtual Colour colour(double value) { return this->colourMethod_->colour(value); }
	virtual Colour  leftColour(double value) { return this->colourMethod_->left(value); }
	virtual Colour rightColour(double value) { return this->colourMethod_->right(value); }
	virtual double  leftRange(double value) { return this->colourMethod_->leftRange(value); }
	virtual double rightRange(double value) { return this->colourMethod_->rightRange(value); }
	virtual bool operator()(const LevelSelection& list)
	{ 
		LevelSelection filter;
		for (LevelSelection::const_iterator level = list.begin(); level != list.end(); ++level) 
			if ( this->min_ <= *level && *level <= this->max_ ) 
			          filter.push_back(*level); 
		(*this->colourMethod_).prepare(filter); 
		return (*this->technique_).prepare(filter, *this->colourMethod_);
	}
	// returns true, if the contouring lines have to be created... False, is the shading is finished...
	virtual void visit(LegendVisitor& legend)  { 
		legend.newLegend(); 
			(*this->technique_).visit(legend, *this->colourMethod_); }
	virtual bool shadingMode() { return (*this->technique_).shadingMode(); }
    
protected:
	//! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream&) const; 
	vector<Colour> colours_;
	vector<Colour>::iterator colour_;

private:
	//! Copy constructor - No copy allowed
	IsoShading(const IsoShading<P>&);
	//! Overloaded << operator to copy - No copy allowed
	IsoShading& operator=(const IsoShading<P>&);


};

template <class P>
class Translator<string, NoIsoShading<P> > { 
public:
	NoIsoShading<P>* operator()(const string& val )
	{
		 return SimpleObjectMaker<NoIsoShading<P> >::create(val);
	}     
	NoIsoShading<P>* magics(const string& param)
	{
		NoIsoShading<P>* object=0;
		ParameterManager::update(param, object);
		return object;
	}
};
} // namespace magics
#include "IsoShading.cc"

#endif
