// RFC822.cc - source file for the mailfilter program
// Copyright (c) 2000 - 2004  Andreas Bauer <baueran@in.tum.de>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.

#include <string>
#include <strstream>
#include <cstring>
#include <vector>
extern "C" {
#include <ctype.h>
}
#include "Header.hh"
#include "RFC822.hh"

using namespace std;


namespace rfc {

  
  RFC822::RFC822() {
  }


  RFC822::~RFC822() {
  }


  //! Extract header information from a RFC822 compliant header, stored entirely in a single string object
  msg::Header RFC822::extract(const string& head, int message, int size, bool normal) {
    msg::Header myHeader;
    msg::Line curLine;
    strstream curDescr;
    strstream curContent;
    
    myHeader.setNumber(message);
    myHeader.setSize(size);
    
    // Skip the first line of the server response to the TOP command, as it is merely '+OK ... \r\n'
    int headerStart = head.find("\n", 0);
    if (headerStart == -1)
      throw MalformedHeader();
    else
      headerStart++;
    
    // Scan e-mail header
    for (int i = headerStart; (i+5 < (int)head.length()) && 
	   !(head[i] == '\r' && head[i+1] == '\n' &&
	     head[i+2] == '.'); i++)
      {
	if (i > 0) {
	  // Reset streams again after initial use
	  curDescr.rdbuf()->freeze(0); curDescr.seekp(0);
	  curContent.rdbuf()->freeze(0); curContent.seekp(0);
	}
	
	// Read field description
	while (head[i] != ':') {
	  curDescr << head[i]; i++;
	  
#ifdef DEBUG
	  cout << head[i-1];
#endif
	  
	} i++;
	
#ifdef DEBUG
	cout << ": ";
#endif
	
	// Ignore possible white spaces in between field-name and field-body
	// (Necessary cause some mailers and/or clients do insert them.)
	while (head[i] == ' ') i++;
	
	// Read field content, e.g. a list of e-mail addresses seperated by "CR LF + LW HT" (See RFC 822, section 3.3 for details)
	while ( i+2 < (int)head.length()  &&  !(head[i] == '\r' && head[i+1] == '\n' && (isalpha(head[i+2]) || head[i+2] == '.')) ) {
	  if (!isspace(head[i]) || head[i] == ' ') {
	    curContent << head[i];
#ifdef DEBUG
	    cout << head[i];
#endif
	  }
	  else if (head[i] == '\n') {
	    curContent << " ";
#ifdef DEBUG
	    cout << " ";
#endif
	  }
	  i++;
	}
	
	// Move pointer forward to beginning of next field-name
	i++;
	
	// Terminate streams, this is _very_ important!
	curDescr << ends; curContent << ends;
	
#ifdef DEBUG
	cout << endl;
#endif
	
	// Store Message-ID and header information for "Deleted..." log-messages and for eventually normalising the subject string later on
	if (strcasecmp(curDescr.str(), "From") == 0)
	  myHeader.setSender(curContent.str());
	else if (strcasecmp(curDescr.str(), "Subject") == 0) {
	  myHeader.setSubject(curContent.str());
	  
	  // Normalise if necessary
	  if (normal)
	    myHeader.setNormalSubject(curDescr.str() + (string)": " + normalise(curContent.str())); // Store the entire header line,
	}
	else if (strcasecmp(curDescr.str(), "Date") == 0)
	  myHeader.setDate(curContent.str());
	else if (strcasecmp(curDescr.str(), "Message-ID") == 0)
	  myHeader.setMessageID(curContent.str());
	
	// Store the current line (descr + content)
	if (strlen(curDescr.str()) > 0 && strlen(curContent.str()) > 0) {
	  curLine.setDescr(curDescr.str());
	  curLine.setContent(curContent.str());
	  myHeader.addLine(curLine);
	}
      }
    
    // A last time: unfreeze strstreams, to release memory
    curDescr.rdbuf()->freeze(0);
    curContent.rdbuf()->freeze(0);
 
    return myHeader;
  }
  
  
  //! Normalise the subject, anySubject does/must contain std::ends already as normalise() is not adding any null-terminator!
  string RFC822::normalise(const string& anySubject) {
    unsigned int i = 0;
    string subject = anySubject;

    try {
      while (i < subject.length()) {
	// Delete multiple spacesx
	while ( isspace(subject[i]) && isspace(subject[i+1]) && (i < subject.length()) )
	  subject.erase(i, 1);
	
	// Delete all non-alphanumeric characters, except spaces and '@'
	if ( !isalpha(subject[i]) && !isspace(subject[i]) && (subject[i] != '@') && !isdigit(subject[i]) )
	  subject.erase(i, 1);
	else
	  i++;
      }

      // Set normalised subject
      return subject;
    }
    catch (...) {
      throw;  // Out Of Range Exception could be thrown
    }
  }


}
