#!/bin/bash
#
# name           : wrapper_mailfilter
#
# syntax         : wrapper_mailfilter <accounts>
#
# description    : This program can be used for automatic poll of your
#                  e-mail accounts (e.g. as a cronjob). Your accounts get
#                  cleaned by mailfilter first and if there are any remaining
#                  mails it runs fetchmail. Any activity is logged and if
#                  desired a spam report is sent to a given email address
#                  (probably simply your loginname).
#
# preparations   : - your .mailfilterrc should be renamed to
#                    .mailfilterrc_<account>
#
#                  - your .fetchmailrc should be renamed to
#                    .fetchmailrc_<account>
#
#                  where <account> can be any string (e.g. name of provider)
#
# author         : Kai Hildebrandt <kai.hildebrandt@web.de>
#
# date           : 2004-02-04
#
# version        : 0.3 (for mailfilter >= 0.5.1)
#
# license        : Licensed under the same terms as Mailfilter, GPL v2 or later
# -----------------------------------------------------------------------------
# BUGS/Limitations
# -----------------------------------------------------------------------------
# o If mailfilter reports an error it returns a negative return value. bash
#   cannot interpret negative return values of programs so this script will
#   run fetchmail even in case of an error.
#   -> workaround: uncomment the server error handling below
# -----------------------------------------------------------------------------
# ChangeLog
# -----------------------------------------------------------------------------
# 04.02.2004  KH    Inserted handling for possible syntax errors in the config
#                   file and send a report
#
# 11.11.2003  KH    Inserted handling for possible server errors
#                   (it is commented out because not everyone wants it)
#
# 12.10.2003  KH    fetchmail messages no longer present in logfile
#                   of this script (substituted "| tee -a" with ">>")
#
# 10.10.2003  KH    initial release
# -----------------------------------------------------------------------------
# set variables below to fit your needs
# -----------------------------------------------------------------------------
# run mailfilter with option --test for the following accounts
# (separated by space, e.g. TESTBOXES="account1 account2")
TESTBOXES=""

# Shall I create logfiles (recommended!)
# values: 0 = disable
#         1 = enable
LOGGING=0

# directory where the logfiles should be stored
# (e.g. LOGDIR=${HOME}/logs)
LOGDIR=${HOME}/logs

# where should the spam report sent to?
# (leave empty if you want to disable this feature)
MAILTO=

# debuglevel of mailfilter
# (number between 0 (nothing) - 6 (debugging))
# default value: 3
MF_DEBUGLEVEL=3

# debuglevel of this script
# (number between 1 (only errors) - 5 (debugging))
# default value: 1
DEBUGLEVEL=1

# options from fetchmail (there must be option -f at the end)
# (e.g FETCHMAILARGS="--verbose -t 60 -f")
FETCHMAILARGS="-n --verbose -t 60 -f"

# your point-to-point interface (internet connection)
#
# this value is for testing for an internet connection via grep.
# you can disable this feature by setting INTERFACE to "."
# (. matches everything)
#
# possible values:
#
# modem/dsl: ppp0
# isdn     : ippp0
#
# probably 0 can also be 1,2,3, etc. depending on your system
INTERFACE=ppp0

# set PATH to fit your environment
# (mailfilter and fetchmail must be found inside these directories)
PATH=/bin:/usr/bin:/usr/local/bin
# -----------------------------------------------------------------------------
# don't edit anything below this line except you know what you're doing!
# -----------------------------------------------------------------------------
function debugm()
{
    if [ ${DEBUGLEVEL} -ge $3 ]; then
	    echo "($2) : $1"
    fi
}


function clean_exit()
{
    debugm "deleting temporary files" "clean_exit" 5

    rm -f ${TMPFILE}
    rm -f ${REPORT}

    debugm "checking for dead tmpfiles" "clean_exit" 5

    MY_PID=$$

    for MAILBOX in ${MAILBOXES}; do

	FILES=$( ls /var/tmp/${MAILBOX}*.[1-9][0-9]*.tmp 2>/dev/null )

        if [ ! -z "${FILES}" ]; then

	    for FILE in ${FILES}; do

		PID=$( echo ${FILE} | sed -e "s/\.tmp$//" -e "s/^.*\.//" )

		if [ ${PID} -ne ${MY_PID} ]; then
		    debugm "deleting old tmpfile: ${FILE}" "clean_exit" 5
		    rm -f ${FILE}
		fi

	    done
	fi
    done    
    
    debugm "reset language environment" "clean_exit" 5
    LANG=${OLDLANG}

    exit 0
}


function usage()
{
    echo "$( basename $0 ) <account>"
}


########
# main #
########

# no parameters
if [ $# -eq 0 ]; then
    usage
    exit 1
fi

# make sure that LOGGING is a number
LOGGING=$(( ${LOGGING} ))

if [ "${LOGGING}" -eq 1 ]; then
    if [ ! -d ${LOGDIR} ]; then
	echo "logdir ${LOGDIR} does not exist."
	echo -n "shall I create it (y/n)? "
	read ANSWER
	if [ "${ANSWER}" = "y" -o "${ANSWER}" = "yes" ]; then
	    mkdir -p ${LOGDIR}
	else
	    echo "logdir not found: logging disabled."
	    LOGGING=0
	fi
    fi
fi

MAILBOXES=$*

for MAILBOX in ${MAILBOXES}; do

    debugm "now checking account ${MAILBOX}" "main" 4

    LOGFILE=${LOGDIR}/mailfilter_${MAILBOX}.log
    FETCHMAILLOG=${LOGDIR}/fetchmail_${MAILBOX}.log
    
    MAILFILTERRC=${HOME}/.mailfilterrc_${MAILBOX}
    FETCHMAILRC=${HOME}/.fetchmailrc_${MAILBOX}

    if [ ! -f ${MAILFILTERRC} ]; then
	echo "your mailfilterrc (${MAILFILTERRC}) could not be found."
	exit 1
    elif [ ! -f ${FETCHMAILRC} ]; then
	echo "your fetchmailrc (${FETCHMAILRC}) could not be found."
	exit 1
    fi
    
    if [ $( /sbin/ifconfig | grep "${INTERFACE}" | wc -l ) -eq 0 ]; then
	debugm "no internet connection available." "main" 1
	exit 1
    fi
    
    TEST=
    
    for BOX in ${TESTBOXES}; do
	if [ ${BOX} = ${MAILBOX} ]; then
	    debugm "account ${MAILBOX} is in test mode" "main" 4
	    TEST="--test"
	    break
	fi
    done
    
    REPORT=/var/tmp/${MAILBOX}_report.$$.tmp
    TMPFILE=/var/tmp/${MAILBOX}.$$.tmp
    
    debugm "unset language environment" "main" 5
    OLDLANG=${LANG}
    unset LANG

    # catch EXIT to clean up tempfiles at exit
    trap "clean_exit" EXIT
    
    echo -n "checking account ${MAILBOX}... "

    MF_OPTIONS="${TEST} --verbose ${MF_DEBUGLEVEL} -r -M ${MAILFILTERRC}"

    if [ "${LOGGING}" -eq 1 ]; then
	MF_OPTIONS="${MF_OPTIONS} -L ${LOGFILE}"
    fi

    mailfilter ${MF_OPTIONS} > ${TMPFILE}
    RETURN=$?
    
    debugm "return value of mailfilter: ${RETURN}" "main" 4

    # 04.02.2004 KH:
    #
    # Check for syntax errors in mailfilterrc and report them
    #
    if [ ${RETURN} -eq 255 ]; then
	if [ ! -z "${MAILTO}" ]; then
	    echo "run mailfilter manually and correct the errors shown by mailfilter." > ${REPORT}
	    SUBJECT="spamreport (${MAILBOX}): error in your mailfilterrc: ${MAILFILTERRC}"
	    mail -e -s "${SUBJECT}" ${MAILTO} < ${REPORT}
	else
	    echo "error in your mailfilterrc: ${MAILFILTERRC}"
	    echo "run mailfilter manually and correct the errors shown by mailfilter."
	fi

    # --- BEGIN --- server error handling
    #
    # 11.11.2003 KH:
    #
    # This is a workaround for frequently appearing errors on my freemail
    # provider. If such an error occurs this script won't run fetchmail.
    #
    # Feel free to enable this function if you know what you are doing!
    #
    # Simply uncomment the 4 lines below.
#    elif [ ${RETURN} -eq 155 ]; then
#	echo "Probably this is only an error: login failure. Not running fetchmail."
#    elif [ ${RETURN} -eq 150 ]; then
#	echo "Probably this is only an error: too many connections. Not running fetchmail."
    elif [ ${RETURN} -gt 0 ]; then

    # ---  END  --- server error handling
	
	echo "OK"
	
	cat ${TMPFILE} | grep "Deleted" > ${REPORT}
	DELETED=$( cat ${REPORT} | wc -l )
	
	if [ ${DELETED} -ne 0 ]; then
	    if [ ! -z "${MAILTO}" ]; then
		debugm "sending spamreport to ${MAILTO}" "main" 4
		SUBJECT="spamreport (${MAILBOX}, ${RETURN} Mails): ${DELETED} messages were deleted by mailfilter"
		mail -e -s "${SUBJECT}" ${MAILTO} < ${REPORT}
	    fi
	fi
	
	if [ ! -z "${TEST}" -o ${RETURN} -gt ${DELETED} ]; then
	    
	    if [ ! -z "${TEST}" ]; then
		echo "receiving ${RETURN} mails..."
		debugm "setting maximum for fetchmail to ${RETURN}" "main" 4

		if [ "${LOGGING}" -eq 1 ]; then
		    fetchmail ${FETCHMAILARGS} ${FETCHMAILRC} --fetchlimit ${RETURN} >> ${FETCHMAILLOG}
		else
		    fetchmail ${FETCHMAILARGS} ${FETCHMAILRC} --fetchlimit ${RETURN}
		fi
	    else
		echo "receiving $(( ${RETURN} - ${DELETED} )) mails..."
		debugm "setting maximum for fetchmail to $(( ${RETURN} - ${DELETED} ))" "main" 4

		if [ "${LOGGING}" -eq 1 ]; then
		    fetchmail ${FETCHMAILARGS} ${FETCHMAILRC} --fetchlimit $(( ${RETURN} - ${DELETED} )) >> ${FETCHMAILLOG}
		else
		    fetchmail ${FETCHMAILARGS} ${FETCHMAILRC} --fetchlimit $(( ${RETURN} - ${DELETED} ))
		fi
	    fi
	    
	else
	    echo "no mail"
	fi
    else
	echo "no mail"
    fi
    
done

# clean_exit
