/*
   Copyright (C) 2007 Omat Holding B.V. <info@omat.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

// Own
#include "headerwidget.h"
#include "headerproxy.h"
#include "headerview.h"
#include "searchline.h"

// Qt
#include <QHeaderView>
#include <QVBoxLayout>

// KDE
#include <KConfigGroup>
#include <KDebug>
#include <akonadi/collection.h>
#include <akonadi/collectionview.h>
#include <akonadi/collectionfilterproxymodel.h>
#include <akonadi/collectionmodel.h>
#include <akonadi/itemfetchjob.h>
#include <akonadi/itemfetchscope.h>
#include <akonadi/collectionmodifyjob.h>
#include <akonadi/monitor.h>
#include <akonadi/kmime/messagemodel.h>

using namespace Mailody;

HeaderWidget::HeaderWidget( QWidget *parent )
        : QWidget( parent )
{
    QVBoxLayout *layout = new QVBoxLayout( this );

    QTime aaaa;
    aaaa.start();

    // Proxy and search
    m_headerProxy = new HeaderProxy( this );
    m_searchLine = new SearchLine( this );
    m_searchLine->setProxy( m_headerProxy );

    // headerview
    m_headerView = new HeaderView( this );
    m_headerView->scrollToBottom();
    m_headerView->header()->setDefaultSectionSize( 200 );
    m_headerView->header()->setMinimumSectionSize( 50 );
    m_headerView->header()->setResizeMode( QHeaderView::Interactive );
    m_headerView->setSelectionMode( QAbstractItemView::ExtendedSelection );
    m_headerView->setSelectionBehavior( QAbstractItemView::SelectRows );

    // messageview
    m_messageModel = new Akonadi::MessageModel( this );
    m_headerProxy->setSourceModel( m_messageModel );
    m_headerView->setModel( m_headerProxy );

    connect( m_headerView, SIGNAL( clicked( QModelIndex ) ), SLOT( itemActivated( QModelIndex ) ) );

    layout->addWidget( m_searchLine );
    layout->addWidget( m_headerView );

    kDebug() << "Finished:" << aaaa.elapsed() << "ms";

    // retrieve columnsizes and set them
    KConfigGroup config = KGlobal::config()->group( "HeaderWidget" );
    m_headerView->setColumnSizes( config.readEntry( "headers", QList<int>() ) );
    int sortColumn = config.readEntry( "sortColumn", 2 );
    Qt::SortOrder sortOrder = ( Qt::SortOrder )config.readEntry( "sortOrder",0 );
    m_headerView->setSortingEnabled( true );
    m_headerView->sortByColumn( sortColumn, sortOrder );
    m_headerView->header()->setSortIndicator( sortColumn, sortOrder );
    m_headerProxy->setHideDeleted( true );

    // start a monitor
    m_monitor = new Akonadi::Monitor( this );
    connect( m_monitor, SIGNAL( itemChanged( const Akonadi::Item&, const QSet< QByteArray >& ) ),
             SLOT( slotItemChanged( const Akonadi::Item& ) ) );
}

HeaderWidget::~HeaderWidget()
{
    KConfigGroup config = KGlobal::config()->group( "HeaderWidget" );
    config.writeEntry( "headers", m_headerView->columnSizes() );
    config.writeEntry( "sortColumn",
                       m_headerView->header()->sortIndicatorSection() );
    config.writeEntry( "sortOrder",
                       ( int )m_headerView->header()->sortIndicatorOrder() );
    config.sync();
}


void HeaderWidget::itemActivated( const QModelIndex &index )
{
    using namespace Akonadi;
    static const HeaderProxy *headerProxy =
        static_cast<const HeaderProxy*>( index.model() );
    static const Akonadi::MessageModel *messageModel =
        static_cast<const Akonadi::MessageModel*>( headerProxy->sourceModel() );

    Akonadi::Item item = messageModel->itemForIndex( headerProxy->mapToSource( index ) );

    // fist monitor the item.
    m_monitor->setItemMonitored( item );

    // fetch the item.
    ItemFetchJob *job = new ItemFetchJob( item, this );
    job->fetchScope().fetchFullPayload();
    connect( job, SIGNAL( result( KJob* ) ), SLOT( itemFetchDone( KJob* ) ) );
}

void HeaderWidget::itemFetchDone( KJob *job )
{
    using namespace Akonadi;
    ItemFetchJob *fetch = static_cast<ItemFetchJob*>( job );
    if ( job->error() ) {
        kWarning() << "Item fetch failed: " << job->errorString();
    } else if ( fetch->items().isEmpty() ) {
        kWarning() << "No item found!";
    } else {
        m_requestedItem = fetch->items().first();
        if ( m_requestedItem.hasPayload() )
            emit selectMessageLeft( m_requestedItem );
    }
}

void HeaderWidget::slotItemChanged( const Akonadi::Item &item )
{
    if ( item == m_requestedItem )
        emit selectMessageLeft( item );
}

void HeaderWidget::setSortOnChild( bool y )
{
    m_headerProxy->setSortOnChild( y );
}

void HeaderWidget::setHideDeleted( bool y )
{
    m_headerProxy->setHideDeleted( y );
}

void HeaderWidget::setCollection( int id )
{
    m_messageModel->setCollection( Akonadi::Collection( id ) );
}

QList< Akonadi::Item > HeaderWidget::currentlySelectedItems()
{
    QList< Akonadi::Item > results;
    QModelIndexList list = m_headerView->selectionModel()->selectedRows( 0 );
    foreach( const QModelIndex& index, list )
    results.append( m_messageModel->itemForIndex( m_headerProxy->mapToSource( index ) ) );

    return results;
}

#include "headerwidget.moc"
