/* This file is part of the KDE project
   Copyright (C) 2006-2007 Omat Holding B.V. <info@omat.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

// Own
#include "messagedata.h"
#include "global.h"

// KDE
#include <KDebug>
#include <KTemporaryFile>
#include <kpimutils/email.h>
#include <kpimutils/linklocator.h>
#include <kmime/kmime_dateformatter.h>
#include <kmime/kmime_message.h>

using KMime::Message;
using KMime::Content;

using namespace Mailody;

MessageData::MessageData( QWidget* parent,  const char* name )
        : QObject( parent ), m_block( false ), m_delete( false ),
        m_xmsp( 0 )
{
    setObjectName( name );
}

MessageData::~MessageData()
{
    //    kDebug() << m_uid << m_mb << endl;
    for ( int i = 0 ; i < m_attachmentPointers.size() ; i++ )
        delete m_attachmentPointers.at( i );
}


bool MessageData::setMsg( const Akonadi::Item& item, const Akonadi::Collection& collection )
{
    m_attachments.clear();
    m_item = item;
    m_collection = collection;
    m_msg = item.payload<MessagePtr>();

    /*
    Akonadi::ItemStoreJob* job = new Akonadi::ItemStoreJob( item );
    job->addFlag( "\\Seen" );
    connect( job, SIGNAL( result( KJob* ) ), SLOT( slotItemChangeFlagDone( KJob* ) ) );
    */

    m_subject = m_msg->subject()->asUnicodeString();
    m_sender_full = m_msg->from()->asUnicodeString();
    KPIMUtils::extractEmailAddressAndName( m_sender_full, m_sender_email, m_sender );
    if ( m_sender.isEmpty() )
        m_sender = m_sender_email;

    m_date = m_msg->date()->dateTime();

    KMime::Headers::Base *organization_raw= m_msg->organization();
    if ( organization_raw )
        m_company = organization_raw->asUnicodeString();

    KMime::Headers::To *to_raw= m_msg->to();
    if ( to_raw ) {
        m_to = to_raw->asUnicodeString().trimmed();

        if ( !to_raw->isEmpty() )
            m_to_addresses = to_raw->prettyAddresses();
    }

    KMime::Headers::Cc *cc_raw= m_msg->cc();
    if ( cc_raw ) {
        m_cc = cc_raw->asUnicodeString().trimmed();

        if ( !cc_raw->isEmpty() )
            m_cc_addresses = cc_raw->prettyAddresses();
    }

    KMime::Headers::Base* ua = m_msg->getHeaderByType( "User-Agent" );
    if ( ua )
        m_userAgent = ua->asUnicodeString();

    KMime::Headers::Base* rt = m_msg->getHeaderByType( "Reply-To" );
    if ( rt )
        m_replyTo = rt->asUnicodeString();

    KMime::Headers::Base* rf = m_msg->getHeaderByType( "Resent-From" );
    if ( rf )
        m_resentFrom = rf->asUnicodeString();

    KMime::Headers::Base* rf2 = m_msg->getHeaderByType( "Resent-To" );
    if ( rf2 )
        m_resentTo = rf2->asUnicodeString();

    KMime::Headers::Base* xms = m_msg->getHeaderByType( "X-Mailody-Signature" );
    if ( xms )
        m_xms = xms->asUnicodeString();

    KMime::Headers::Base* xmsp = m_msg->getHeaderByType( "X-Mailody-SP" );
    if ( xmsp )
        m_xmsp = xmsp->asUnicodeString().toInt();

    foreach( KMime::Content *c, m_msg->attachments() ) {
        // try the content-id
        QString filename;
        KMime::Headers::Base *h = c->getHeaderByType( "Content-ID" );
        if ( h )
            filename = h->asUnicodeString();

        if ( filename.isEmpty() )
            filename = c->contentDisposition()->filename();

        // try the Content-type name parameter
        if ( filename.isEmpty() ) {
            KMime::Headers::ContentType *h = c->contentType( false );
            if ( h )
                filename = h->name();
        }

        // Ok, I give up...
        if ( filename.isEmpty() )
            continue;

        QByteArray result = c->decodedContent();

        // Some stupid mailers...
        if ( filename.startsWith( '<' ) && filename.endsWith( '>' ) )
            filename=filename.mid( 1, filename.length()-2 );

        // Save it
        KTemporaryFile* temp = new KTemporaryFile();
        temp->setPrefix( "Mailody-" );
        temp->open();
        m_attachments[ KUrl::fromPath( temp->fileName() )] = filename;
        temp->write( result );
        temp->close();
        m_attachmentPointers.append( temp );
    }
    return true;
}

void MessageData::slotItemChangeFlagDone( KJob* job )
{
    if ( job->error() ) {
        kWarning() << "Item could not be set to 'read': " << job->errorString();
    } else
        kDebug() << "Item changed to 'read' successfully";
}

void MessageData::requestBody( bodyType type )
{
    m_bodyType = type;

    // Retrieve the plain and html part.
    QString plainPart, htmlPart;
    Content* part = m_msg->mainBodyPart( "text/plain" );
    if ( part )
        plainPart = part->decodedText( true, true );
    part = m_msg->mainBodyPart( "text/html" );
    if ( part )
        htmlPart = part->decodedText( true, true );

    // replace all cid: entries with their filenames.
    QHash<KUrl,QString>::Iterator ita;
    for ( ita = m_attachments.begin(); ita != m_attachments.end(); ita++ ) {
        kDebug() << "cid:"+ita.key().path() << " -> " << ita.value() << endl;
        htmlPart.replace( "cid:"+ita.value(),ita.key().path() );
    }

    // Assign m_body
    using KPIMUtils::LinkLocator;
    int flags = LinkLocator::PreserveSpaces | LinkLocator::HighlightText;

    KConfigGroup config = KGlobal::config()->group( "General" );
    if ( config.readEntry( "ShowSmileys", false ) )
        flags |= LinkLocator::ReplaceSmileys;

    if ( m_bodyType == HTML ) {
        if ( htmlPart.trimmed().isEmpty() ) {
            m_body = LinkLocator::convertToHtml( plainPart, flags );
        } else
            m_body = htmlPart;

        // when replying prefer plain
        !plainPart.isEmpty() ? m_raw = plainPart : m_raw = htmlPart;
    } else if ( m_bodyType == Plain ) {
        if ( plainPart.trimmed().isEmpty() )
            m_body = LinkLocator::convertToHtml( htmlPart, flags );
        else
            m_body = LinkLocator::convertToHtml( plainPart, flags );

        // when replying prefer plain
        !plainPart.isEmpty() ? m_raw = plainPart : m_raw = htmlPart;
    } else if ( m_bodyType == Source ) {
        m_body = LinkLocator::convertToHtml( m_msg->encodedContent() , flags );
        m_raw = m_msg->encodedContent();
    }
    m_realraw = m_msg->encodedContent();

    // make the quotation colors.
    m_body = Global::highlightText( m_body );

    emit messageData( this );
}

QString MessageData::vDate() const
{
    return KGlobal::locale()->formatDateTime( m_date,
            KLocale::FancyShortDate );
}

bool MessageData::isDeleted() const
{
    return m_item.hasFlag( "\\Deleted" );
}

bool MessageData::isNew() const
{
    return !m_item.hasFlag( "\\Seen" );
}

bool MessageData::hasUserTag( int i ) const
{
    return m_item.hasFlag(( "$label" + QString::number( i ) ).toLatin1() );
}

void MessageData::setBlocked( bool block )
{
    // kDebug() << m_uid << m_mb << m_delete << endl;
    m_block=block;
    if ( m_delete )
        deleteLater();
}

void MessageData::done()
{
    // kDebug() << m_uid << m_mb << endl;
    if ( !m_block )
        deleteLater();
    else
        m_delete = true;
}

#include "messagedata.moc"
