/* This file is part of the KDE project
   Copyright (C) 2008 Omat Holding B.V. <info@omat.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "quickreplywidget.h"
#include "messagedata.h"
#include "sendmessage.h"
#include "global.h"

#include <QGridLayout>
#include <QLabel>
#include <QPushButton>

#include <KDebug>
#include <KTextEdit>
#include <KLocale>
#include <KMessageBox>

#include <kpimidentities/identitymanager.h>
#include <kpimidentities/identity.h>
#include <mailtransport/transportmanager.h>
#include <mailtransport/transport.h>

using namespace Mailody;

QuickReplyWidget::QuickReplyWidget( QWidget* parent )
        : QWidget( parent )
{
    QLabel* header = new QLabel( i18n( "Quick Reply:" ) );
    m_editor = new KTextEdit( this );
    m_details = new QLabel();
    QPushButton* send = new QPushButton( i18n( "Send" ) );
    connect( send, SIGNAL( clicked() ), SLOT( slotSend() ) );

    QGridLayout* layout = new QGridLayout( this );
    layout->setSpacing( 5 );
    layout->setMargin( 0 );
    layout->addWidget( header, 0, 0, 1, 2 );
    layout->addWidget( m_editor, 1, 0, 2, 1 );
    layout->addWidget( m_details, 1, 1 );
    layout->addWidget( send, 2, 1 );
}

void QuickReplyWidget::setMessage( MessageData* data )
{
    m_message = data;
    m_editor->clear();

    // Make the text for the details.
    QString text = data->subject();
    m_details->setText( text );
}

void QuickReplyWidget::slotSend()
{
    kDebug();

    // Get the send object.
    SendMessage* toSend = new SendMessage( this );
    connect( toSend, SIGNAL( finished() ), SLOT( slotDone() ) );
    connect( toSend, SIGNAL( sendError( const QString& ) ),
             SLOT( slotError( const QString& ) ) );

    // Get the default transport
    const MailTransport::TransportManager* transportManager = MailTransport::TransportManager::self();
    const int defaultTransportID = transportManager->defaultTransportId();

    // Get the default Identity
    KPIMIdentities::Identity identity = Global::identityManager()->defaultIdentity();

    // Add the to recipient.
    if ( !m_message->replyTo().isEmpty() ) {  // there is a reply to header!
        if ( !Global::myEmail( m_message->replyTo() ) ) // only add if it is not my address.
            toSend->addTo( m_message->replyTo() );
    } else {    // no reply to header.
        if ( !Global::myEmail( m_message->sender() ) )  // only add if it is not my address.
            toSend->addTo( m_message->sender_full( false /* no company*/ ) );
    }

    // Add cc's, first the people that were in the to address originally.
    QStringList list = m_message->to_list();
    if ( !list.isEmpty() ) {
        QStringList::Iterator it=list.begin();
        for ( ; it!=list.end(); ++it )
            if ( !Global::myEmail(( *it ) ) ) // only add if it is not my address.
                toSend->addCc( *it );
            else // if it is my address, reply with the identitity.
                identity = Global::getIdentityForEmail( *it );
    }

    // Add cc's, add the people that were in the cc address originally.
    list = m_message->cc_list();
    if ( !list.isEmpty() ) {
        QStringList::Iterator it=list.begin();
        for ( ; it!=list.end(); ++it )
            if ( !Global::myEmail(( *it ) ) )
                toSend->addCc( *it );
            else
                identity = Global::getIdentityForEmail( *it );
    }

    // Set the subject.
    if ( m_message->subject().startsWith( "Re:", Qt::CaseInsensitive ) )
        toSend->setSubject( m_message->subject() );
    else
        toSend->setSubject( "Re: " + m_message->subject() );

    // Set the in-reply-to.
    toSend->setIRT( m_message->messageID() );

    // Prepare the text and Append the signature to it.
    QString body = m_editor->toPlainText();
    KPIMIdentities::Signature sig = identity.signature();
    body += "\n\n" + sig.withSeparator();

    // Get the original text of the message and quote it;
    QString text = i18n( "At %1, you wrote:\n",
                         KGlobal::locale()->formatDateTime( m_message->date(), KLocale::LongDate ) )
                   + Global::quote( m_message->raw() ) + '\n';

    // Merge it to a message, based on the top or bottom setting.
    int pos = identity.property( s_sigpos ).toInt(); //2 = top, 1 = bottom
    ( pos == 1 ) ? text.append( "\n\n" + body ) : text.prepend( body + "\n\n" );

    kDebug() << "Final text: " << text;

    toSend->send( m_message->collection(), identity.uoid(), defaultTransportID, text,
                  QString() /* html part will be generated if needed */ );
}

void QuickReplyWidget::slotDone()
{
    m_editor->clear();
    emit finished();
}

void QuickReplyWidget::slotError( const QString& error )
{
    KMessageBox::information( this, error );
}

#include "quickreplywidget.moc"
