/* This file is part of the KDE project
   Copyright (C) 2006-2007 Omat Holding B.V. <info@omat.nl>
   Copyright (C) 2007-2008 Frode M. Døving <frode@lnix.net>

   Original copied from showfoto:
    Copyright 2005 by Gilles Caulier <caulier.gilles@free.fr>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

// Own
#include "setupidentities.h"
#include "global.h"

// Qt
#include <QLabel>
#include <QRadioButton>
#include <QPushButton>
#include <QBoxLayout>
#include <QGroupBox>
#include <QButtonGroup>

// KDE
#include <KDialog>
#include <KLineEdit>
#include <KTabWidget>

#include <kpimidentities/identitymanager.h>
#include <kpimidentities/identity.h>
#include <kpimidentities/signatureconfigurator.h>

using namespace Mailody;

SetupIdentities::SetupIdentities( QWidget *parent )
        : QWidget( parent )
{
    QGridLayout *mainGrid = new QGridLayout( this );

    m_identybox = new KListWidget( this );
    m_identybox->setAcceptDrops( false );
    connect( m_identybox,
             SIGNAL( currentItemChanged( QListWidgetItem*, QListWidgetItem* ) ),
             SLOT( slotSelectionChanged( QListWidgetItem*, QListWidgetItem* ) ) );

    mainGrid->addWidget( m_identybox, 0, 0, 1, 2, Qt::AlignLeft );

    QPushButton *add = new QPushButton( i18n( "Add" ), this );
    add->setWhatsThis( i18n( "Add new identity" ) );
    connect( add, SIGNAL( clicked() ), SLOT( slotAdd() ) );
    mainGrid->addWidget( add, 1, 0, Qt::AlignLeft );

    m_del = new QPushButton( i18n( "Remove" ), this );
    m_del->setWhatsThis( i18n( "Remove the selected identity" ) );
    connect( m_del, SIGNAL( clicked() ), SLOT( slotDel() ) );
    mainGrid->addWidget( m_del, 1, 1, Qt::AlignLeft );

    m_tabWidget = new KTabWidget( this );

    mainGrid->addWidget( m_tabWidget, 0, 2, 2, 4 );
    mainGrid->setRowStretch( 0, 0 );
    mainGrid->setColumnStretch( 1, 0 );
    mainGrid->setColumnStretch( 2, 0 );
    mainGrid->setColumnStretch( 3, 2 );
    mainGrid->setColumnStretch( 4, 2 );
    mainGrid->setColumnStretch( 5, 2 );
    mainGrid->setColumnMinimumWidth( 3, 150 );
    mainGrid->setColumnMinimumWidth( 4, 50 );
    mainGrid->setColumnMinimumWidth( 5, 100 );

    // -------------- page 1 ---------------------- //

    QWidget *page1 = new QWidget( this );
    QGridLayout *m1 = new QGridLayout( page1 );

    QLabel *l0 = new QLabel( i18n( "Identity name:" ) + ' ', page1 );
    l0->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    m1->addWidget( l0, 0, 0 );
    l0->setWhatsThis( i18n( "The label of this identiy" ) );
    m_identityName = new KLineEdit( page1 );
    connect( m_identityName, SIGNAL( textChanged( const QString& ) ),
             SLOT( slotIdentityNameEdit( const QString& ) ) );
    m1->addWidget( m_identityName, 0, 1 );
    l0->setBuddy( m_identityName );

    QLabel *l1 = new QLabel( i18n( "Full name:" ) + ' ', page1 );
    l1->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    l1->setWhatsThis( i18n( "Your full name, shown in messages you send using this identity." ) );
    m1->addWidget( l1, 1, 0 );
    m_fullName = new KLineEdit( page1 );
    m1->addWidget( m_fullName, 1, 1 );
    l1->setBuddy( m_fullName );

    QLabel *l2 = new QLabel( i18n( "E-Mail Address:" ) + ' ', page1 );
    l2->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    l2->setWhatsThis( i18n( "Your E-Mail Address" ) );
    m1->addWidget( l2, 2, 0 );
    m_emailAddress = new KLineEdit( page1 );
    m1->addWidget( m_emailAddress, 2, 1 );
    l2->setBuddy( m_emailAddress );

//X     QLabel *htmlp = new QLabel( i18n( "Include HTML part:" ) + ' ', page1 );
//X     m1->addWidget( htmlp, 6, 0 );
//X     htmlp->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    m_htmlPart = new QGroupBox( i18n( "Include Auto HTML Part:" ), page1 );
    m_htmlPart->setWhatsThis( i18n( "Mailody is able to convert your plain "
                                    "message to a html message and include that "
                                    "in the outgoing message. This means the "
                                    "receiver will also have clickable links and "
                                    "colored quote levels." ) );
    QHBoxLayout *htmlPartLay = new QHBoxLayout( m_htmlPart );
    htmlPartLay->setMargin( 0 );
    htmlPartLay->setSpacing( KDialog::spacingHint() );
    QRadioButton *yes = new QRadioButton( i18n( "Yes" ), m_htmlPart );
    QRadioButton *no = new QRadioButton( i18n( "No" ), m_htmlPart );
    htmlPartLay->addWidget( yes );
    htmlPartLay->addWidget( no );
    m1->addWidget( m_htmlPart, 6, 0, 1, 2 );
//X     htmlp->setBuddy( m_htmlPart );

    m_htmlPart_group = new QButtonGroup( page1 );
    m_htmlPart_group->addButton( no, 1 );
    m_htmlPart_group->addButton( yes, 2 );

    m_htmlPrefer = new QGroupBox( i18n( "Prefer composer HTML mode:" ), page1 );
    m_htmlPrefer->setWhatsThis( i18n( "This sets the composer to HTML mode by default" ) );
    QHBoxLayout *htmlPreferLay = new QHBoxLayout( m_htmlPrefer );
    htmlPreferLay->setMargin( 0 );
    htmlPreferLay->setSpacing( KDialog::spacingHint() );
    QRadioButton *yes2 = new QRadioButton( i18n( "Yes" ), m_htmlPrefer );
    QRadioButton *no2 = new QRadioButton( i18n( "No" ), m_htmlPrefer );
    htmlPreferLay->addWidget( yes2 );
    htmlPreferLay->addWidget( no2 );
    m1->addWidget( m_htmlPrefer, 7, 0, 1, 2 );

    m_htmlPrefer_group = new QButtonGroup( page1 );
    m_htmlPrefer_group->addButton( no2, 1 );
    m_htmlPrefer_group->addButton( yes2, 2 );

    m_sigPos = new QGroupBox( i18n( "Signature position:" ), page1 );
    m_sigPos->setWhatsThis( i18n( "Location of the signature when replying to messages" ) );
    QHBoxLayout *sigPosLay = new QHBoxLayout( m_sigPos );
    sigPosLay->setMargin( 0 );
    sigPosLay->setSpacing( KDialog::spacingHint() );
    QRadioButton *yes3 = new QRadioButton( i18n( "Top" ), m_sigPos );
    QRadioButton *no3 = new QRadioButton( i18n( "Bottom" ), m_sigPos );
    sigPosLay->addWidget( yes3 );
    sigPosLay->addWidget( no3 );
    m1->addWidget( m_sigPos, 8, 0, 1, 2 );

    m_sigPos_group = new QButtonGroup( page1 );
    m_sigPos_group->addButton( no3, 1 );
    m_sigPos_group->addButton( yes3, 2 );

    QSpacerItem *spacer1 = new QSpacerItem( 50, 50, QSizePolicy::Minimum,
                                            QSizePolicy::Expanding );
    m1->addItem( spacer1, 9, 1 );

    m_tabWidget->addTab( page1,i18n( "General" ) );

    // ------------------- page 2 -----------------------//
    QWidget *page2 = new QWidget( this );
    QBoxLayout *m2 = new QBoxLayout( QBoxLayout::TopToBottom, page2 );

    m_signatureWidget = new KPIMIdentities::SignatureConfigurator( page2 );
    m2->addWidget( m_signatureWidget );

    m_tabWidget->addTab( page2,i18n( "Signature" ) );
    //---------------------------------------------------//

    readSettings();
    m_tabWidget->setEnabled( false );

    updateButtons();
}

SetupIdentities::~SetupIdentities()
{
}

void SetupIdentities::updateButtons()
{
    if ( m_identybox->count() < 2 )
        m_del->setEnabled( false );
    else
        m_del->setEnabled( true );
}

void SetupIdentities::slotAdd()
{
    // kDebug() << endl;

    int i = 0;
    QString name;

    while ( true ) {
        ++i;
        name = i18n( "Identity #%1", i );
        QList<QListWidgetItem *> find = m_identybox->findItems( name,
                                        Qt::MatchExactly );
        if ( find.count() == 0 )
            break;
    }

    KPIMIdentities::IdentityManager *im = Global::identityManager();
    KPIMIdentities::Identity& id = im->newFromControlCenter( name );
    if ( id.emailAddr().isEmpty() )
        id.setEmailAddr( im->allEmails().first() );

    kDebug() << "New List: " << im->shadowIdentities() << endl;

    m_identybox->setCurrentItem( new QListWidgetItem( name, m_identybox ) );
    m_tabWidget->setCurrentIndex( 0 );
    m_tabWidget->setEnabled( true );
    m_identityName->setFocus();
    m_identityName->selectAll();

    updateButtons();
}

void SetupIdentities::slotDel()
{
    QListWidgetItem* item = m_identybox->currentItem();
    if ( !item || m_identybox->count() == 1 || m_currentIdentity != item->text() )
        return;

    kDebug() << "Deleting: " << item->text() << " - " << m_currentIdentity << endl;
    KPIMIdentities::IdentityManager *im = Global::identityManager();
    im->removeIdentity( item->text() );

    kDebug() << "New List: " << im->shadowIdentities() << endl;

    m_currentIdentity.clear();
    delete item;

    updateButtons();
}

void SetupIdentities::slotSelectionChanged( QListWidgetItem*, QListWidgetItem* )
{
    kDebug() << endl;

    // Save current settings, but only when widget is enabled, deleting
    // an identity disables the widget, but does not clear them...
    if ( m_tabWidget->isEnabled() )
        saveCurrent();

    m_tabWidget->setEnabled( true );

    // Now handle what the user wants. find the identity clicked on and
    // display the values.
    if ( !m_identybox->currentItem() ||
            m_identybox->currentItem()->text().isEmpty() )
        return;

    KPIMIdentities::IdentityManager * im = Global::identityManager();
    kDebug() << im->shadowIdentities() << endl;

    KPIMIdentities::Identity& ident =
        im->modifyIdentityForName( m_identybox->currentItem()->text() );

    if ( ident == KPIMIdentities::Identity::null() ) {
        kWarning() << "Identitity not found" << endl;
        return;
    }

    // we found it. Now set all the fields.
    m_currentIdentity=m_identybox->currentItem()->text();
    m_identityName->setText( ident.identityName() );
    m_fullName->setText( ident.fullName() );
    m_emailAddress->setText( ident.emailAddr() );
    int checked = ident.property( s_usehtml ).toInt();
    if ( checked == 0 )
        checked = 1;
    m_htmlPart_group->button( checked )->setChecked( true );

    checked = ident.property( s_preferhtml ).toInt();
    if ( checked == 0 )
        checked = 1;
    m_htmlPrefer_group->button( checked )->setChecked( true );

    checked = ident.property( s_sigpos ).toInt();
    if ( checked == 0 )
        checked = 1;
    m_sigPos_group->button( checked )->setChecked( true );

    m_signatureWidget->setSignature( ident.signature() );
}

void SetupIdentities::saveCurrent()
{
    // Delete the current identity object.
    if ( !m_currentIdentity.isEmpty() ) {
        KPIMIdentities::IdentityManager *im = Global::identityManager();
        KPIMIdentities::Identity &ident =
            im->modifyIdentityForName( m_currentIdentity );

        ident.setIdentityName( m_identityName->text() );
        ident.setFullName( m_fullName->text() );
        ident.setEmailAddr( m_emailAddress->text() );
        ident.setProperty( s_usehtml, m_htmlPart_group->checkedId() );
        ident.setProperty( s_preferhtml, m_htmlPrefer_group->checkedId() );
        ident.setProperty( s_sigpos, m_sigPos_group->checkedId() );
        ident.setSignature( m_signatureWidget->signature() );
    }
}

void SetupIdentities::slotIdentityNameEdit( const QString &newName )
{
    kDebug() << newName << endl;
    QListWidgetItem *item = m_identybox->currentItem();
    if ( !item )
        return;

    // don't emit a selectionchanged!
    m_identybox->blockSignals( true );
    item->setText( newName );
    m_identybox->blockSignals( false );

    // let the identitymanager know of the change of name.
    KPIMIdentities::IdentityManager *im = Global::identityManager();
    kDebug() << "Old: " << im->shadowIdentities() << endl;

    KPIMIdentities::Identity &ident =
        im->modifyIdentityForName( m_currentIdentity );

    ident.setIdentityName( newName );
    m_currentIdentity = newName;

    kDebug() << "New: " << im->shadowIdentities() << endl;
}

void SetupIdentities::applySettings()
{
    // kDebug() << endl;

    // Save current settings, but only when widget is enabled, deleting
    // an identity disables the widget, but does not clear them...
    if ( m_tabWidget->isEnabled() )
        saveCurrent();

    KPIMIdentities::IdentityManager *im = Global::identityManager();
    im->commit();
}

void SetupIdentities::readSettings()
{
    // kDebug() << endl;
    KPIMIdentities::IdentityManager *im = Global::identityManager();
    im->sort();
    for ( KPIMIdentities::IdentityManager::ConstIterator it = im->modifyBegin();
            it != im->modifyEnd(); ++it ) {
        const QString name = ( *it ).identityName();
        new QListWidgetItem( name, m_identybox );
    }
}

#include "setupidentities.moc"
