/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module defines the data type "value_t", and many
 * operations to build, modify, and print such values.
 * The first cell of a value, its type cell, is used to store the value's type
 * together with some type dependent information, which is an unsigned number
 * less than INFO_MAX.
 * Use the macro TYPE to get the type of a value, and INFO to get the type 
 * dependent information. Use TYPE_CELL to create a type-cell.
 * There are five different types of values:
 * symbol, string, list, record and number. */

/* includes =================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "basic.h"
#include "pools.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL

#include "values.h"

/* constants ================================================================*/

#define CELL_BITS BITS_PER_BYTE * sizeof (cell_t)
#define TYPE_BITS 3
#define INFO_BITS (CELL_BITS - TYPE_BITS)
#define TYPE_MAX ((cell_t) 1 << TYPE_BITS)
#define INFO_MAX ((cell_t) 1 << INFO_BITS)
#define TYPE_MASK ((TYPE_MAX - 1) << INFO_BITS)
#define INFO_MASK (INFO_MAX - 1)

#define SYMBOL_TYPE ((cell_t) 0 << INFO_BITS) 
/* A value of type SYMBOL_TYPE consists only of a type cell. 
 * Its INFO-value is the code for the symbol. */

#define STRING_TYPE ((cell_t) 1 << INFO_BITS) 
/* A value of type STRING_TYPE consists of its type cell,
 * followed by the actual string. Its INFO-value is the
 * number of characters in the string. The actual string
 * is stored in the subsequent cells, two characters
 * paired in a cell, and terminated by one or two
 * NUL-characters, so that the total number of chars is
 * even. The NUL-chars are NOT part of the string. */

#define LIST_TYPE ((cell_t) 2 << INFO_BITS)
/* A value of type LIST_TYPE consists of its type cell and a subsequent cell,
 * which hold the type and the length of the value, followed by 0 or more
 * values of any type, namely the values that form the list. 
 * The length of a list <value>, that is the number of cells needed to store 
 * all the list's values, is (INFO(value[0]) << CELL_BITS) + value[1]. */

#define RECORD_TYPE ((cell_t) 3 << INFO_BITS)
/* A value of type RECORD_TYPE consists of its type cell and a subsequent cell,
 * which hold the type and the length of the value, followed by 0 or more 
 * pairs of values. 
 * In a pair of values, the first value must be a symbol and is considered as 
 * an attribute of that record. The second value is the value of that
 * attribute, and it can be of any type.
 * The length of a record <value>, that is the number of cells
 * needed to store all the record's value pairs, is computed as
 * (INFO(value) << CELL_BITS) + value[1]. */

#define NUMBER_TYPE ((cell_t) 4 << INFO_BITS)
/* A value of type NUMBER_TYPE consists of its type cell,
 * followed by a implementation-dependent number of cells
 * that contain a C "double" value.
 * Its INFO-value is 0. */

/* macros ===================================================================*/

#define TYPE(value) ((*(value)) & TYPE_MASK)
#define INFO(value) ((*(value)) & INFO_MASK)
#define TYPE_CELL(type,info) ((type)|(info))

/* Use one of the following predicates to test a value
 * against a special type. */
#define IS_SYMBOL(value) (TYPE (value) == SYMBOL_TYPE)
#define IS_STRING(value) (TYPE (value) == STRING_TYPE)
#define IS_RECORD(value) (TYPE (value) == RECORD_TYPE)
#define IS_LIST(value) (TYPE (value) == LIST_TYPE)
#define IS_NUMBER(value) (TYPE (value) == NUMBER_TYPE)

#define NEXT_VALUE(value) ((value) + length_of_value (value))
/* Return end of <value>.
 * This may also be the beginning of the next value in a list. */

#define NEXT_ATTRIB(attrib) ((attrib) + 1 + length_of_value ((attrib) + 1))
/* Return the next attribute in a record. */

#define CELLS_PER_NUMBER (sizeof (double) / sizeof (cell_t))
/* The number of cells needed to contain a number value
 * sizeof (double) MUST BE A MULTIPLE OF sizeof (cell_t). */

/* types ====================================================================*/

/* an element in a linked list of hidden attributes */
typedef struct ATTRIBUTE_T 
{
  struct ATTRIBUTE_T *next;
  symbol_t symbol;
} attribute_t;

/* variables ================================================================*/

/* two constant values */
LOCAL cell_t empty_list[] = {TYPE_CELL (LIST_TYPE, 0), 0};
LOCAL cell_t empty_record[] = {TYPE_CELL (RECORD_TYPE, 0), 0};

LOCAL cell_t *value_heap; /* the actual heap */
LOCAL cell_t *value_heap_end; /* pointer to first free cell in heap */
LOCAL int_t value_heap_size; /* size of the value heap in cells */

LOCAL int_t value_stack_size; /* size of the value stack */

LOCAL attribute_t *hidden_attributes; /* the list of hidden attributes */
LOCAL text_t text; /* buffer for conversion of values to text */

/* forward declarations =====================================================*/

LOCAL void value_to_text (text_t text, value_t value, bool_t full_value);

/* support functions ========================================================*/

LOCAL void copy_cells (value_t destination, value_t source, int_t n)
/* Copy <n> cells of value <source> to <destination>. */
{
  value_t source_end = source + n;

  while (source < source_end)
    *destination++ = *source++;
}

/*---------------------------------------------------------------------------*/

LOCAL void copy_value (value_t destination, value_t source)
/* Copy all cells of value <source> to <destination>. */
{
  value_t source_end = NEXT_VALUE (source);

  while (source < source_end)
    *destination++ = *source++;
}

/*---------------------------------------------------------------------------*/

LOCAL int compare_value_pointers (const void *key1, const void *key2)
/* Return -1/0/1 when the value <value_pointer1> points to is stored on a
 * lower/same/higher address than the value <value_pointer2> points to. */
{
  value_t *value_pointer1;
  value_t *value_pointer2;

  value_pointer1 = * (value_t **) key1;
  value_pointer2 = * (value_t **) key2;

  if (*value_pointer1 < *value_pointer2)
    return -1; 
  else if (*value_pointer1 > *value_pointer2)
    return 1;
  else
    return 0;
}

/*---------------------------------------------------------------------------*/

LOCAL void collect_garbage (void)
/* Make sure the value heap only contains values that are on the value stack.
 * Compactify the heap, i.e. move all values on the heap to the beginning. */
{
  int_t i;
  value_t new_value_end;
  value_t old_value_start;
  value_t new_value_start;
  value_t old_value_end;
  value_t **value_pointer;

  new_value_end = value_heap;

  /* Copy values if there is at least one value to save. */
  if (top > 0) 
  {
    /* Create a table of pointers to the values. */
    value_pointer = new_vector (sizeof (value_t *), top);
    
    for (i = 0; i < top; i++)
      value_pointer[i] = value_stack + i;
    
    /* Sort pointers according to the address of the value they point to. */
    qsort (value_pointer, top, sizeof (value_t *), compare_value_pointers);
    
    /* Find the first index <i> whose value is on the heap. */
    for (i = 0; i < top; i++)
    {
      if (*value_pointer[i] >= value_heap)
	break;
    }
    
    /* Work on all values on the heap. */
    while (i < top && *value_pointer[i] < value_heap_end) 
    {
      /* Copy the value. */
      old_value_start = *value_pointer[i];
      old_value_end = NEXT_VALUE (old_value_start);
      new_value_start = new_value_end;
      new_value_end = new_value_start + (old_value_end - old_value_start);
      copy_value (new_value_start, old_value_start);
      
      /* Adjust the value address and the addresses of all values
       * that are part of that value. */
      while (i < top && *value_pointer[i] < old_value_end) 
      {
	*value_pointer[i] = (new_value_start 
			     + (*value_pointer[i] - old_value_start));
	i++;
      }
    }
    
    free_mem (&value_pointer);
  }
  
  value_heap_end = new_value_end;
}

/*---------------------------------------------------------------------------*/

LOCAL value_t space_for_value (int_t size)
/* Get <size> adjacent free cells on the value heap. */
{
  value_t pointer;

  if ((value_heap_end - value_heap) + size > value_heap_size)
  {
    collect_garbage ();
    if ((value_heap_end - value_heap) + size > value_heap_size)
    {
      int_t i;
      value_t old_heap = value_heap;
      value_t old_heap_end = value_heap_end;
      
      /* Enlarge the value heap. */
      value_heap_size = renew_vector (&value_heap, sizeof (cell_t),
				      2 * (size + (old_heap_end - old_heap)));
      value_heap_end = value_heap + (old_heap_end - old_heap);
      
      /* Adapt the value stack pointers. */
      for (i = 0; i < top; i++)
      {
	if (value_stack[i] >= old_heap && value_stack[i] < old_heap_end)
	  value_stack[i] = value_heap + (value_stack[i] - old_heap);
      }
    }
  }

  pointer = value_heap_end;
  value_heap_end += size;
  return pointer;
}

/*---------------------------------------------------------------------------*/

LOCAL value_t space_for_composed_value (int_t type, int_t length)
/* Allocate <length> cells for a composed value of <type>, set its type cell
 * and return the value. */
{
  value_t new_value = space_for_value (length);
  int_t content_size = length - 2;

  if (content_size >= 1L << (INFO_BITS + CELL_BITS))
    error ("value too big");

  new_value[0] = TYPE_CELL (type, content_size >> CELL_BITS);
  new_value[1] = content_size & ((1L << CELL_BITS) - 1);

  return new_value;
}

/* module initialisation ====================================================*/

GLOBAL void init_values (void)
/* Initialise this module. */
{
  value_heap_size = 1000;
  value_heap = new_vector (sizeof (cell_t), value_heap_size);
  value_heap_end = value_heap;
  value_stack_size = 100;
  value_stack = new_vector (sizeof (value_t), value_stack_size);
  top = 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL void terminate_values (void)
/* Terminate this module. */
{
  free_mem (&value_heap);
  free_mem (&value_stack);
  free_text (&text);
  clear_hidden_attributes ();
}

/* value operations =========================================================*/
 
GLOBAL value_t new_value (value_t value)
/* Allocate space for <value> and copy it.
 * Use "free" to free the value space. */
{
  value_t new_value;

  new_value = new_vector (sizeof (cell_t), length_of_value (value));
  copy_value (new_value, value);
  return new_value;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t copy_value_to_pool (pool_t value_pool, 
				   value_t value, 
				   int_t *index)
/* Copy <value> to the pool <value_pool> and store its index in *<index>. */
{
  value_t new_value;

  new_value = (value_t) get_pool_space (value_pool, length_of_value (value), 
					index);
  copy_value (new_value, value);
  return new_value;
}

/*---------------------------------------------------------------------------*/

GLOBAL int_t length_of_value (value_t value)
/* Return the length of <value> in cells. */
{
  switch (TYPE (value)) 
  {
  case SYMBOL_TYPE:
    return 1;
    
  case STRING_TYPE:
    return 2 + INFO (value) / sizeof (cell_t);
    
  case LIST_TYPE:
  case RECORD_TYPE:
    return 2 + ((int_t) INFO (value) << CELL_BITS) + value[1];

  case NUMBER_TYPE:
    return 1 + CELLS_PER_NUMBER;
    
  default:
    error ("internal (value has bad type)");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL symbol_t get_value_type (value_t value)
/* Return the type of <value>. Depending of the type, the result value may be
 * SYMBOL_SYMBOL, STRING_SYMBOL, NUMBER_SYMBOL, 
 * LIST_SYMBOL or RECORD_SYMBOL. */
{
  switch (TYPE (value))
  {
  case SYMBOL_TYPE: return SYMBOL_SYMBOL;
  case STRING_TYPE: return STRING_SYMBOL;
  case NUMBER_TYPE: return NUMBER_SYMBOL;
  case LIST_TYPE: return LIST_SYMBOL;
  case RECORD_TYPE: return RECORD_SYMBOL;
  default:
    error ("illegal type");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void push_value (value_t value)
/* STACK EFFECTS: (nothing) -> <value>. */
{
  if (top + 1 > value_stack_size)
    value_stack_size = renew_vector (&value_stack, sizeof (value_t), 
				     2 * (top + 1));
  value_stack[top++] = value;
}

/*---------------------------------------------------------------------------*/

GLOBAL void insert_value (int_t n, value_t value)
/* STACK EFFECTS: <value_1>...<value_n> -> <value> <value_1>...<value_n>. */
{
  int_t i;

  push_value (NULL);
  for (i = 0; i < n; i++)
    value_stack[top - i - 1] = value_stack[top - i - 2];
  value_stack[top - n - 1] = value;
}

/* symbol operations ========================================================*/

GLOBAL symbol_t value_to_symbol (value_t value)
/* Return <value> as a symbol. It is an error if <value> is no symbol. */
{
  if (! IS_SYMBOL (value))
    error ("value is no symbol");

  return *value;
}

/*---------------------------------------------------------------------------*/

GLOBAL void push_symbol_value (symbol_t symbol)
/* STACK EFFECTS: (nothing) -> <new_symbol>.
 * <new_symbol> is <symbol> converted to a Malaga value. */
{
  value_t value = space_for_value (1);
 
  *value = TYPE_CELL (SYMBOL_TYPE, symbol);
  push_value (value);
}

/* string operations ========================================================*/

GLOBAL string_t value_to_string (value_t string)
/* Return the value of <string> as a C style string. */
{
  if (! IS_STRING (string))
    error ("value is no string");

  return (string_t) (string + 1);
}

/*---------------------------------------------------------------------------*/

GLOBAL void push_string_value (string_t string_start, string_t string_end)
/* STACK EFFECTS: (nothing) -> <new_string>.
 * <new_string> is the string starting at <string_start> as a Malaga value.
 * If <string_end> != NULL, it marks the end of the string. */
{
  value_t value, value_end;
  int_t length;
  string_t target_ptr, source_ptr;

  if (string_end == NULL)
    string_end = string_start + strlen (string_start);

  length = string_end - string_start;
  if (length > INFO_MAX - 1)
    error ("string too long to be a value");

  value = space_for_value (2 + length / sizeof (cell_t));
  *value = TYPE_CELL (STRING_TYPE, length);
   
  source_ptr = string_start;
  target_ptr = (string_t) (value + 1);
  value_end = NEXT_VALUE (value);

  while (source_ptr < string_end)
    *target_ptr++ = *source_ptr++;

  while (target_ptr < (string_t) value_end)
    *target_ptr++ = '\0';

  push_value (value);
}

/*---------------------------------------------------------------------------*/

GLOBAL void concat_string_values (void)
/* STACK EFFECTS: <string_1> <string_2> -> <new_string>.
 * <new_string> is the concatenation of <string_1> and <string_2>. */
{
  int_t new_length;
  string_t new_string, old_string, string_end;
  value_t new_string_value;

  if (! IS_STRING (value_stack[top-2]) || ! IS_STRING (value_stack[top-1]))
    error ("concatenation operands must be strings");

  new_length = ((int_t) INFO (value_stack[top-2]) 
		+ (int_t) INFO (value_stack[top-1]));

  if (new_length > INFO_MAX - 1)
    error ("strings too long for concatenation");
    
  new_string_value = space_for_value (2 + new_length / sizeof (cell_t));
  *new_string_value = TYPE_CELL (STRING_TYPE, new_length);
    
  /* Join the strings. We do it by hand so it's easier to align. */
  new_string = (string_t) (new_string_value + 1);
    
  old_string = (string_t) (value_stack[top-2] + 1);
  while (*old_string != '\0')
    *new_string++ = *old_string++;
  
  old_string = (string_t) (value_stack[top-1] + 1);
  while (*old_string != '\0')
    *new_string++ = *old_string++;
    
  string_end = (string_t) NEXT_VALUE (new_string_value);
  while (new_string < string_end)
    *new_string++ = '\0';
    
  top--;
  value_stack[top-1] = new_string_value;
}

/* record operations ========================================================*/

GLOBAL value_t get_attribute (value_t record, symbol_t attribute)
/* Return the value of <attribute> in the record <record> 
 * or NULL if it doesn't exist. */
{
  value_t record_end = NEXT_VALUE (record);
  value_t v;
 
  /* No error when getting an attribute from "nil". */
  if (*record == NIL_SYMBOL)
    return NULL;

  if (! IS_RECORD (record)) 
    error ("can get an attribute value of a record only");
  
  for (v = record + 2; v < record_end; v = NEXT_ATTRIB (v)) 
  {
    if (*v == attribute)
      return v + 1;
  }
  
  return NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL void build_record (int_t n)
/* STACK EFFECTS: <attr_1> <value_1> ... <attr_n> <value_n> -> <new_record>.
 * <new_record> looks like [<attr_1>: <value_1>, ..., <attr_n>: <value_n>]. */
{
  value_t new_record, v;
  int_t i, j, new_record_length;
  value_t *values = value_stack + top - 2*n;

  /* Check that all attributes are different. */
  for (i = 0; i < n; i++) 
  {
    if (! IS_SYMBOL (values[2*i]))
      error ("attribute must be symbol");
    
    for (j = 0; j < i; j++) 
    {
      if (*values[2*i] == *values[2*j])
	error ("attribute twice in record");
    }
  }

  new_record_length = 2;
  for (i = 0; i < n; i++)
    new_record_length += 1 + length_of_value (values[2*i + 1]);

  new_record = space_for_composed_value (RECORD_TYPE, new_record_length);
  v = new_record + 2;
  for (i = 0; i < n; i++) 
  {
    *v++ = *values[2*i];
    copy_value (v, values[2*i + 1]);
    v = NEXT_VALUE (v);
  }
  
  top -= 2*n;
  push_value (new_record);
}

/*---------------------------------------------------------------------------*/

GLOBAL void join_records (void)
/* STACK EFFECTS: <record_1> <record_2> -> <new_record>.
 * <new_record> contains all attributes of <record_1> and <record_2>, and 
 * their associated values. If an attribute has different values in <record_1>
 * and <record_2>, the value in <record_2> will be taken. */
{
  value_t record1, record2, record1_end, record2_end, new_record, v, v1, v2;
  int_t new_record_length;

  record1 = value_stack[top-2];
  record2 = value_stack[top-1];
  record1_end = NEXT_VALUE (record1);
  record2_end = NEXT_VALUE (record2);
  if (! IS_RECORD (record1) || ! IS_RECORD (record2))
    error ("join operands must be records");

  /* Calculate the space needed. This is the length of the
   * first record plus the length of the second record minus the
   * sum of the length of all attribute-value-pairs in <record1> whose
   * attributes are also in <record2>. */
  new_record_length
    = length_of_value (record1) + length_of_value (record2) - 2;
  for (v1 = record1 + 2; v1 < record1_end; v1 = NEXT_ATTRIB (v1)) 
  {
    for (v2 = record2 + 2; v2 < record2_end; v2 = NEXT_ATTRIB (v2)) 
    {
      if (*v1 == *v2) 
	/* We've discovered two identical attributes */
      { 
	new_record_length -= (1 + length_of_value (v1 + 1));
	break;
      }
    }
  }
  
  /* Allocate a new record value. */
  new_record = space_for_composed_value (RECORD_TYPE, new_record_length);
  
  record1 = value_stack[top-2];
  record2 = value_stack[top-1];
  record1_end = NEXT_VALUE (record1);
  record2_end = NEXT_VALUE (record2);

  /* Copy the attributes of the first record. If an attribut
   * belongs to both <value1> and <value2>, don't copy its value. */
  v = new_record + 2;
  for (v1 = record1 + 2; v1 < record1_end; v1 = NEXT_ATTRIB (v1)) 
  {
    /* Go through <record2> until we reach end or find same attribute. */
    for (v2 = record2 + 2; v2 < record2_end; v2 = NEXT_ATTRIB (v2)) 
    {
      if (*v1 == *v2)
	break;
    }
    
    if (v2 >= record2_end) 
      /* If did not find the attribute in <record2>,
       * copy the attribute value of <record1>. */
    {
      *v = *v1;
      copy_value (v + 1, v1 + 1);
      v = NEXT_ATTRIB (v);
    }
  }
  
  /* Append the attributes of the second record. */
  copy_cells (v, record2 + 2, length_of_value (record2) - 2);
  
  top--;
  value_stack[top-1] = new_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL void select_attribute (symbol_t attribute)
/* STACK EFFECTS: <record> -> <new_record>.
 * <new_record> contains <attribute> and its value in <record>. */
{
  value_t record, record_end, v, new_record;

  record = value_stack[top-1];
  record_end = NEXT_VALUE (record);
  if (! IS_RECORD (record))
    error ("can select attributes from record only");

  for (v = record + 2; v < record_end; v = NEXT_ATTRIB (v))
  {
    if (*v == attribute)
      break;
  }
  if (v == record_end)
    new_record = empty_record;
  else
  {
    int_t new_record_length = 3 + length_of_value (v + 1);

    new_record = space_for_composed_value (RECORD_TYPE, new_record_length);

    record = value_stack[top-1];
    record_end = NEXT_VALUE (record);

    /* Find <attribute> in record. */
    for (v = record + 2; *v != attribute; v = NEXT_ATTRIB (v))
    {
    }
    new_record[2] = attribute;
    copy_value (new_record + 3, v + 1);
  }

  value_stack[top-1] = new_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL void select_attributes (void)
/* STACK EFFECTS: <record> <list> -> <new_record>.
 * <new_record> contains all attribute-value pairs of <record> whose attributes
 * are in <list>. */
{
  value_t record, list, record_end, list_end, v, v1, v2, new_record;
  int_t new_record_length;

  record = value_stack[top-2];
  list = value_stack[top-1];
  record_end = NEXT_VALUE (record);
  list_end = NEXT_VALUE (list);

  if (! IS_RECORD (record))
    error ("can select attributes from record only");
  if (! IS_LIST (list))
    error ("attribute selection list must be a list");

  /* Check that <value2> is a list with symbols only. */
  for (v2 = list + 2; v2 < list_end; v2 = NEXT_VALUE (v2))
  {
    if (! IS_SYMBOL (v2))
      error ("attribute selection list must contain symbols only");
  }
      
  /* Calculate size of new value */
  new_record_length = 2;
  for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1))
  {
    for (v2 = list + 2; v2 < list_end; v2++)
    {
      if (*v1 == *v2)
      {
	new_record_length += 1 + length_of_value (v1 + 1);
	break;
      }
    }
  }

  /* We don't create a new record if no attributes are deleted. */
  if (new_record_length == length_of_value (record))
    new_record = record;
  else
  {
    /* Allocate and copy new value. */
    new_record = space_for_composed_value (RECORD_TYPE, new_record_length);

    record = value_stack[top-2];
    list = value_stack[top-1];
    record_end = NEXT_VALUE (record);
    list_end = NEXT_VALUE (list);

    v = new_record + 2;
    for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1))
    {
      for (v2 = list + 2; v2 < list_end; v2++)
      {
	if (*v1 == *v2)
	{
	  *v++ = *v1;
	  copy_value (v, v1 + 1);
	  v = NEXT_VALUE (v);
	  break;
	}
      }
    }
  }
  
  top--;
  value_stack[top-1] = new_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL void remove_attribute (symbol_t attribute)
/* STACK EFFECTS: <record> -> <new_record>.
 * <new_record> contains all attribute-value pairs of <record> but the one with
 * attribute <attribute>. */
{
  value_t record, new_record, record_end, v, v1;
  int_t new_record_length;

  record = value_stack[top-1];
  record_end = NEXT_VALUE (record);

  if (! IS_RECORD (record))
    error ("can remove attributes from record only");

  /* Find the attribute that is to be deleted. */
  for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1)) 
  {
    if (*v1 == attribute)
      break;
  }

  if (v1 == record_end) /* No attribute to delete - return original record. */
    new_record = record;
  else
  {
    /* Compute its length and get space for the new record. */
    new_record_length = (length_of_value (record)
			 - (length_of_value (v1 + 1) + 1));
    new_record = space_for_composed_value (RECORD_TYPE, new_record_length);
	  
    record = value_stack[top-1];
    record_end = NEXT_VALUE (record);

    /* Copy the record. */
    v = new_record + 2;
    for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1)) 
    {
      if (*v1 != attribute) 
      {
	*v = *v1;
	copy_value (v + 1, v1 + 1);
	v = NEXT_ATTRIB (v);
      }
    }
  }
  
  value_stack[top-1] = new_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL void remove_attributes (void)
/* STACK EFFECTS: <record> <list> -> <new_record>.
 * <new_record> contains all attribute-value pairs of <record> but the ones
 * whose attributes are in <list>. */
{
  value_t v, v1, v2, record, list, record_end, list_end, new_record;
  int_t new_record_length;

  record = value_stack[top-2];
  list = value_stack[top-1];
  record_end = NEXT_VALUE (record);
  list_end = NEXT_VALUE (list);

  if (! IS_RECORD (record))
    error ("can remove attributes from record only");
  if (! IS_LIST (list))
    error ("attribute list must be a list");
  
  /* Check if the list consists of symbols only. */
  for (v2 = list + 2; v2 < list_end; v2++)
  {
    if (! IS_SYMBOL (v2))
      error ("attribute list must contain symbols only");
  }
      
  /* Compute the length of the new record. */
  new_record_length = 2;
  for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1))
  {
    for (v2 = list + 2; v2 < list_end; v2++)
    {
      if (*v1 == *v2)
	break;
    }
    if (v2 == list_end)
      new_record_length += 1 + length_of_value (v1 + 1);
  }
      
  /* We don't create a new record if no attributes will be deleted. */
  if (new_record_length == length_of_value (record))
    new_record = record;
  else
  {
    /* Allocate the new record and copy it. */
    new_record = space_for_composed_value (RECORD_TYPE, new_record_length);

    record = value_stack[top-2];
    list = value_stack[top-1];
    record_end = NEXT_VALUE (record);
    list_end = NEXT_VALUE (list);

    v = new_record + 2;
    for (v1 = record + 2; v1 < record_end; v1 = NEXT_ATTRIB (v1)) 
    {
      for (v2 = list + 2; v2 < list_end; v2++)
      {
	if (*v1 == *v2)
	  break;
      }
      
      if (v2 == list_end)
      {
	*v = *v1;
	copy_value (v + 1, v1 + 1);
	v = NEXT_ATTRIB (v);
      }
    }
  }

  top--;
  value_stack[top-1] = new_record;
}

/*---------------------------------------------------------------------------*/

GLOBAL void replace_attribute (symbol_t attribute)
/* STACK EFFECTS: <record> <value> -> <new_record>.
 * <new_record> is equal to <record>, only the value of <attribute> is replaced
 * by <value>. <record> must contain <attribute>. */
{
  value_t record, value, record_end, new_record, v, nv;
  int_t new_record_length;

  record = value_stack[top-2];
  value = value_stack[top-1];
  record_end = NEXT_VALUE (record);

  if (! IS_RECORD (record))
    error ("value must be a record");

  /* Find the attribute to replace. */
  for (v = record + 2; v < record_end; v = NEXT_ATTRIB (v))
  {
    if (*v == attribute)
      break;
  }

  if (v == record_end)
    error ("missing attribute to replace");

  new_record_length = (length_of_value (record) +
		       length_of_value (value) - length_of_value (v + 1));
  new_record = space_for_composed_value (RECORD_TYPE, new_record_length);

  record = value_stack[top-2];
  value = value_stack[top-1];
  record_end = NEXT_VALUE (record);

  nv = new_record + 2;
  for (v = record + 2; v < record_end; v = NEXT_ATTRIB (v))
  {
    *nv = *v;
    if (*v == attribute)
      copy_value (nv + 1, value);
    else
      copy_value (nv + 1, v + 1);
    nv = NEXT_ATTRIB (nv);
  }

  top--;
  value_stack[top-1] = new_record;
}

/* list operations ==========================================================*/

GLOBAL int_t get_list_length (value_t list)
/* Return the number of elements in <list>. 
 * <list> must be a list. */
{
  int_t elements;
  value_t list_end = NEXT_VALUE (list);
  value_t v;

  if (! IS_LIST (list))
    error ("can get length of a list only");

  elements = 0;
  for (v = list + 2; v < list_end; v = NEXT_VALUE (v))
    elements++;

  return elements;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_element (value_t list, int_t n)
/* Return the <n>-th element of the list <list>,
 * or NULL, if that element doesn't exist.
 * If <n> is positive, elements will be counted from the left border.
 * If it's negative, elements will be counted from the right border. */
{
  value_t list_end = NEXT_VALUE (list);
  value_t v;
  
  /* No error when getting an element from "nil". */
  if (*list == NIL_SYMBOL)
    return NULL;

  if (! IS_LIST (list)) 
    error ("can get an element of a list only");

  if (n < 0)
    n = get_list_length (list) + n + 1;

  if (n <= 0)
    return NULL;

  for (v = list + 2; v < list_end; v = NEXT_VALUE (v)) 
  { 
    n--;
    if (n == 0)
      return v;
  }

  return NULL;
}

/*---------------------------------------------------------------------------*/

GLOBAL void build_list (int_t n)
/* STACK EFFECTS: <value_1> ... <value_n> -> <new_list>.
 * <new_list> looks like < <value_1>, ..., <value_n> >. */
{
  value_t new_list, v;
  int_t i, new_list_length;
  value_t *elements = value_stack + top - n;

  new_list_length = 2;
  for (i = 0; i < n; i++)
    new_list_length += length_of_value (elements[i]);

  new_list = space_for_composed_value (LIST_TYPE, new_list_length);
  v = new_list + 2;
  for (i = 0; i < n; i++) 
  {
    copy_value (v, elements[i]);
    v = NEXT_VALUE (v);
  }
  
  top -= n;
  push_value (new_list);
}

/*---------------------------------------------------------------------------*/

GLOBAL void concat_lists (void)
/* STACK EFFECTS: <list_1> <list_2> -> <new_list>.
 * <new_list> is the concatenation of <list_1> and <list_2>. */
{
  int_t list1_length, list2_length, new_list_length;
  value_t list1, list2, new_list;

  list1 = value_stack[top-2];
  list2 = value_stack[top-1];

  if (! IS_LIST (list1) || ! IS_LIST (list2))
    error ("concatenation operands must be lists");

  list1_length = length_of_value (list1);
  list2_length = length_of_value (list2);
  new_list_length = list1_length + list2_length - 2;
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);

  list1 = value_stack[top-2];
  list2 = value_stack[top-1];
    
  /* Copy all elements of the first and the second list. */
  copy_cells (new_list + 2, list1 + 2, list1_length - 2);
  copy_cells (new_list + list1_length, list2 + 2, list2_length - 2);

  top--;
  value_stack[top-1] = new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL void get_list_difference (void)
/* STACK EFFECTS: <list_1> <list_2> -> <new_list>.
 * <new_list> contains the list difference of <list_1> and <list_2>:
 * An element that appears <m> times in <list_1> and <n> times in <list_2> 
 * appears <m-n> times in <new_list>. */
{
  value_t list1, list2, list1_end, list2_end, new_list, v, v1, v2;
  int_t new_list_length, appearances;

  list1 = value_stack[top-2];
  list2 = value_stack[top-1];
  list1_end = NEXT_VALUE (list1);
  list2_end = NEXT_VALUE (list2);

  if (! IS_LIST (list1) || ! IS_LIST (list2))
    error ("list difference operands must be lists");

  /* Calculate the size of the new value. */
  new_list_length = 2;
  for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
  {
    /* Check whether <v1> will be included in the list.
     * It will be included if the ordinal number of its appearence is 
     * higher than the number of appearences in <list2>. */
    
    /* Count appearences in <list1> up to (including) <v1>. */
    appearances = 1;
    for (v2 = list1 + 2; v2 < v1; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances++;
    }
    
    /* Subtract appearences in <value2>. */
    for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances--;
    }
    
    if (appearances > 0)
      new_list_length += length_of_value (v1);
  }
  
  /* We don't create a new list if no elements will be deleted. */
  if (new_list_length == length_of_value (list1))
    new_list = list1;
  else /* Allocate and copy new value. */
  {
    new_list = space_for_composed_value (LIST_TYPE, new_list_length);

    list1 = value_stack[top-2];
    list2 = value_stack[top-1];
    list1_end = NEXT_VALUE (list1);
    list2_end = NEXT_VALUE (list2);
  
    v = new_list + 2;
    for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
    {
      /* Check whether <v1> will be included in the list.
       * It will be included if the ordinal number of its appearence is 
       * higher than the number of appearences in <value2>. */
      
      /* Count appearences in <value1> up to (including) <v1>. */
      appearances = 1;
      for (v2 = list1 + 2; v2 < v1; v2 = NEXT_VALUE (v2))
      {
	if (values_equal (v1, v2))
	  appearances++;
      }
      
      /* Subtract appearences in <value2>. */
      for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
      {
	if (values_equal (v1, v2))
	  appearances--;
      }
      
      if (appearances > 0)
      {
	copy_value (v, v1);
	v = NEXT_VALUE (v);
      }
    }
  }
  top--;
  value_stack[top-1] = new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL void get_set_difference (void)
/* STACK EFFECTS: <list_1> <list_2> -> <new_list>.
 * <new_list> contains the set difference of <list_1> and <list_2>.
 * Each element of <list_1> is in <new_list> if it is not in <list2>. */
{
  value_t list1, list2, list1_end, list2_end, new_list, v, v1, v2;
  int_t new_list_length;

  list1 = value_stack[top-2];
  list2 = value_stack[top-1];
  list1_end = NEXT_VALUE (list1);
  list2_end = NEXT_VALUE (list2);

  if (! IS_LIST (list1) || ! IS_LIST (list2))
    error ("set difference operands must be lists");

  /* Compute the length of the new list. */
  new_list_length = 2;
  for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
  {
    for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	break;
    }
    
    if (v2 == list2_end)
      new_list_length += length_of_value (v1);
  }
  
  /* No need to create a new list if no elements will be deleted. */
  if (new_list_length == length_of_value (list1))
    new_list = list1;
  else /* Allocate the new list and copy it. */
  {
    new_list = space_for_composed_value (LIST_TYPE, new_list_length);

    list1 = value_stack[top-2];
    list2 = value_stack[top-1];
    list1_end = NEXT_VALUE (list1);
    list2_end = NEXT_VALUE (list2);

    v = new_list + 2;
    for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1)) 
    {
      for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
      {
	if (values_equal (v1, v2))
	  break;
      }
      
      if (v2 == list2_end)
      {
	copy_value (v, v1);
	v = NEXT_VALUE (v);
      }
    }
  }
  
  top--;
  value_stack[top-1] = new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL void intersect_lists (void)
/* STACK EFFECTS: <list_1> <list_2> -> <new_list>.
 * <new_list contains the list intersection of <list_1> and <list_2>.
 * Each element that appears <m> times in <list_1> and <n> times in <list_2>
 * appears min(<m>, <n>) times in <new_list>. */
{
  value_t new_list, list1, list2, list1_end, list2_end, v1, v2, v;
  int_t new_list_length, appearances;

  list1 = value_stack[top-2];
  list2 = value_stack[top-1];
  list1_end = NEXT_VALUE (list1);
  list2_end = NEXT_VALUE (list2);

  if (! IS_LIST (list1) || ! IS_LIST (list2))
    error ("operands for intersection must be lists");

  /* Calculate the size of the new list. */
  new_list_length = 2;
  for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
  {
    /* Check whether <v1> will be included in the list.
     * It will be included if the ordinal number of its appearence is 
     * not higher than the number of appearences in <list2>. */
      
    /* Count appearences in <list1> up to (including) <v1>. */
    appearances = 1;
    for (v2 = list1 + 2; v2 < v1; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances++;
    }
      
    /* Subtract appearences in <list2>. */
    for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	appearances--;
    }
    
    if (appearances <= 0)
      new_list_length += length_of_value (v1);
  }
  
  /* We don't create a new list if no elements will be deleted. */
  if (new_list_length == length_of_value (list1))
    new_list = list1;
  else /* Allocate and copy new value. */
  {
    new_list = space_for_composed_value (LIST_TYPE, new_list_length);

    list1 = value_stack[top-2];
    list2 = value_stack[top-1];
    list1_end = NEXT_VALUE (list1);
    list2_end = NEXT_VALUE (list2);

    v = new_list + 2;
    for (v1 = list1 + 2; v1 < list1_end; v1 = NEXT_VALUE (v1))
    {
      /* Check whether <v1> will be included in the list.
       * It will be included if the ordinal number of its appearence is 
       * not higher than the number of appearences in <value2>. */
      
      /* Count appearences in <value1> up to (including) <v1>. */
      appearances = 1;
      for (v2 = list1 + 2; v2 < v1; v2 = NEXT_VALUE (v2))
      {
	if (values_equal (v1, v2))
	  appearances++;
      }
      
      /* Subtract appearences in <value2>. */
      for (v2 = list2 + 2; v2 < list2_end; v2 = NEXT_VALUE (v2))
      {
	if (values_equal (v1, v2))
	  appearances--;
      }
      
      if (appearances <= 0)
      {
	copy_value (v, v1);
	v = NEXT_VALUE (v);
      }
    }
  }
  
  top--;
  value_stack[top-1] = new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL void remove_element (int_t n)
/* STACK EFFECTS: <list> -> <new_list>.
 * <new_list> is <list> without element at index <n>.
 * If <n> is positive, the elements will be counted from the left border;
 * if <n> is negative, they will be counted from the right border.
 * If <list> contains less than abs(<n>) elements, then <new_list> = <list>. */
{
  value_t list, list_end, new_list, element, v;
  int_t new_list_length;

  list = value_stack[top-1];

  if (! IS_LIST (list))
    error ("can remove an element in a list only");

  /* Find the first/last value in the list that will/won't be copied. */
  element = get_element (list, n);
  if (element == NULL)
    new_list = list;
  else
  {
    new_list_length = length_of_value (list) - length_of_value (element);
    new_list = space_for_composed_value (LIST_TYPE, new_list_length);

    list = value_stack[top-1];
    list_end = NEXT_VALUE (list);
    element = get_element (list, n);

    v = new_list + 2;
    copy_cells (v, list + 2, element - (list + 2));
    v += element - (list + 2);
    copy_cells (v, NEXT_VALUE (element), list_end - NEXT_VALUE (element));
  }

  value_stack[top-1] = new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL void remove_elements (int_t n)
/* STACK EFFECTS: <list> -> <new_list>.
 * <new_list> is <list> without abs(<n>) elements.
 * If <n> is positive, the elements will be cut from the left border,
 * if <n> is negative, they will be cut from the list's right border.
 * If <list> contains less than abs(<n>) elements, then <new_list> = <>. */
{
  value_t new_list, list, border_value;
  int_t new_list_length;

  list = value_stack[top-1];
  if (! IS_LIST (list))
    error ("can delete an element in a list only");

  /* Find the first/last value in the list that will/won't be copied. */
  border_value = get_element (list, n);
  if (border_value == NULL)
    new_list = empty_list;
  else if (n > 0) 
  {
    value_t list_end;

    /* Copy all elements behind <border_value> to a new list. */
    new_list_length = NEXT_VALUE (list) - NEXT_VALUE (border_value) + 2;
    new_list = space_for_composed_value (LIST_TYPE, new_list_length);

    list = value_stack[top-1];
    list_end = NEXT_VALUE (list);

    copy_cells (new_list + 2, list_end - (new_list_length - 2),
		new_list_length - 2);
  }
  else 
  {
    /* Copy all elements in front of <border_value> to a new list. */
    new_list_length = border_value - list;
    new_list = space_for_composed_value (LIST_TYPE, new_list_length);

    list = value_stack[top-1];

    copy_cells (new_list + 2, list + 2, new_list_length - 2);
  }

  value_stack[top-1] = new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL void replace_element (int_t n)
/* STACK EFFECTS: <list> <value> -> <new_list>.
 * <new_list> is <list>, but its <n>-th element is replaced by <value>.
 * If <n> is negative, count from the right end.
 * <list> must contain at least <n> elements. */
{
  value_t list, value, new_list, element, nv;
  int_t new_list_length;

  list = value_stack[top-2];
  value = value_stack[top-1];

  if (! IS_LIST (list))
    error ("can only replace an element in a list");

  element = get_element (list, n);
  if (element == NULL)
    error ("missing element to replace");

  new_list_length = (length_of_value (list) +
		     length_of_value (value) - length_of_value (element));
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);

  list = value_stack[top-2];
  value = value_stack[top-1];
  element = get_element (list, n);

  /* Copy left part */
  nv = new_list + 2;
  copy_cells (nv, list + 2, element - (list + 2));

  /* Copy changed element. */
  nv += element - (list + 2);
  copy_value (nv, value);
  
  /* Copy right part. */
  nv = NEXT_VALUE (nv);
  copy_cells (nv, NEXT_VALUE (element), 
	      NEXT_VALUE (list) - NEXT_VALUE (element));
  
  top--;
  value_stack[top-1] = new_list;
}

/*---------------------------------------------------------------------------*/

GLOBAL void convert_list_to_set (void)
/* STACK EFFECTS: <list> -> <new_list>.
 * <new_list> contains all elements of <list>, but multiple appearances
 * of one value are reduced to a single appearance.
 * That means, <new_list> is <list> converted to a set. */
{
  value_t v1, v2, v, new_list, list, list_end;
  int_t new_list_length;

  list = value_stack[top-1];
  list_end = NEXT_VALUE (list);

  if (! IS_LIST (list))
    error ("can only convert a list to a set");
  
  /* Compute the length of the new list. */
  new_list_length = 2;
  for (v1 = list + 2; v1 < list_end; v1 = NEXT_VALUE (v1))
  {
    /* Check if <v1> already occurred in the list. */
    for (v2 = list + 2; v2 < v1; v2 = NEXT_VALUE (v2))
    {
      if (values_equal (v1, v2))
	break;
    }
    
    if (v2 == v1)
      new_list_length += length_of_value (v1);
  }
  
  /* No need to create a new list if no elements will be deleted. */
  if (new_list_length == length_of_value (list))
    new_list = list;
  else /* Allocate the new list and copy it. */
  {
    new_list = space_for_composed_value (LIST_TYPE, new_list_length);

    list = value_stack[top-1];
    list_end = NEXT_VALUE (list);

    v = new_list + 2;
    for (v1 = list + 2; v1 < list_end; v1 = NEXT_VALUE (v1))
    {
      /* Check if <v1> already occurred in the list. */
      for (v2 = list + 2; v2 < v1; v2 = NEXT_VALUE (v2))
      {
	if (values_equal (v1, v2))
	  break;
      }
      
      if (v2 == v1)
      {
	copy_value (v, v1);
	v = NEXT_VALUE (v);
      }
    }
  }

  value_stack[top-1] = new_list;
}

/* number operations ========================================================*/

GLOBAL double value_to_double (value_t value)
/* Return the value of <value> which must be a number value. */
{
  double number;
  value_t number_ptr;
  int_t i;

  if (! IS_NUMBER (value))
    error ("value is no number");

  number_ptr = (value_t) &number;
  for (i = 0; i < CELLS_PER_NUMBER; i++)
    number_ptr[i] = value[i+1];

  return number;
}

/*---------------------------------------------------------------------------*/

GLOBAL int_t value_to_int (value_t value)
/* Return the value of <value> which must be an integral number value. */
{
  double number = value_to_double (value);
  int_t result;

  result = (int_t) number;
  if (result != number)
    error ("number too big or not integral");

  return result;
}

/*---------------------------------------------------------------------------*/

GLOBAL void push_number_value (double number)
/* STACK EFFECTS: (nothing) -> <new_number>.
 * <new_number> is <number> as a Malaga value. */
{
  int_t i;
  value_t value, number_ptr;

  number_ptr = (value_t) &number;
  value = space_for_value (1 + CELLS_PER_NUMBER);
  *value = TYPE_CELL (NUMBER_TYPE, 0);
  for (i = 0; i < CELLS_PER_NUMBER; i++)
    value[i+1] = number_ptr[i];

  push_value (value);
}

/* type dependent Malaga operations =========================================*/

GLOBAL void dot_operation (void)
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "." <value_2>. 
 * The actual operation depends on the type of the values. */
{
  value_t value1 = value_stack[top-2];
  value_t value2 = value_stack[top-1];

  switch (TYPE (value2))
  {
  case SYMBOL_TYPE:
    top--;
    value_stack[top-1] = get_attribute (value1, value_to_symbol (value2));
    break;
  case NUMBER_TYPE:
    top--;
    value_stack[top-1] = get_element (value1, value_to_int (value2));
    break;
  case LIST_TYPE:
    top--;
    value_stack[top-1] = get_value_part (value1, value2);
    break;
  default:
    error ("in \"<v1> . <v2>\", <v2> must be symbol, number or list");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void plus_operation (void)
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "+" <value_2>. 
 * The actual operation depends on the type of the values. */
{
  value_t value1 = value_stack[top-2];
  value_t value2 = value_stack[top-1];

  switch (TYPE (value1))
  {
  case STRING_TYPE:
    concat_string_values ();
    break;
  case LIST_TYPE:
    concat_lists ();
    break;
  case RECORD_TYPE:
    join_records ();
    break;
  case NUMBER_TYPE:
    top -= 2;
    push_number_value (value_to_double (value1) + value_to_double (value2));
    break;
  default:
    error ("\"+\"-operands must be strings, lists, records, or numbers");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void minus_operation (void)
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "-" <value_2>. 
 * The actual operation depends on the type of the values. */
{
  value_t value1 = value_stack[top-2];
  value_t value2 = value_stack[top-1];

  switch (TYPE (value1))
  {
  case LIST_TYPE:
    switch (TYPE (value2))
    {
    case NUMBER_TYPE:
      top--;
      remove_element (value_to_int (value2)); 
      break;
    case LIST_TYPE:
      get_list_difference ();
      break;
    default:
      error ("in \"<list> - <value>\", <value> must be number or list");
    }
    break;

  case RECORD_TYPE:
    switch (TYPE (value2))
    {
    case SYMBOL_TYPE:
      top--;
      remove_attribute (value_to_symbol (value2));
      break;
    case LIST_TYPE:
      remove_attributes ();
      break;
    default: 
      error ("in \"<record> - <value>\", <value> must be symbol or list");
    }
    break;

  case NUMBER_TYPE:
    top -= 2;
    push_number_value (value_to_double (value1) - value_to_double (value2));
    break;

  default:
    error ("in \"<v1> - <v2>\", <v1> must be list, record, or number");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void asterisk_operation (void)
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "*" <value_2>. 
 * The actual operation depends on the type of the values. */
{
  value_t value1 = value_stack[top-2];
  value_t value2 = value_stack[top-1];

  switch (TYPE (value1))
  {
  case LIST_TYPE:
    intersect_lists ();
    break;

  case RECORD_TYPE:
    switch (TYPE (value2))
    {
    case SYMBOL_TYPE:
      top--;
      select_attribute (value_to_symbol (value2));
      break;
    case LIST_TYPE:
      select_attributes ();
      break;
    case RECORD_TYPE: /* join records, but exchange arguments. */
      top--;
      insert_value (1, value_stack[top]);
      join_records ();
      break;
    default:
      error ("in \"<record> * <value>\", "
	     "<value> must be symbol, list, or record");
    }
    break;

  case NUMBER_TYPE:
    top -= 2;
    push_number_value (value_to_double (value1) * value_to_double (value2));
    break;

  default:
    error ("in \"<v1> *<v2>\", <v1> must be list, record, or number");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void slash_operation (void)
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "/" <value_2>. 
 * The actual operation depends on the type of the values. */
{
  value_t value1 = value_stack[top-2];
  value_t value2 = value_stack[top-1];

  switch (TYPE (value1))
  {
  case LIST_TYPE:
    switch (TYPE (value2))
    {
    case NUMBER_TYPE:
      top--;
      remove_elements (value_to_int (value2));
      break;
    case LIST_TYPE:
      get_set_difference ();
      break;
    default:
      error ("in \"<list> / <value>\", <value> must be number or list");
    }
    break;

  case NUMBER_TYPE:
  {
    double divisor;
    
    divisor = value_to_double (value2);
    if (divisor == 0.0)
      error ("division by 0.0");

    top -= 2;
    push_number_value (value_to_double (value1) / divisor);
    break;
  }

  default:
    error ("\"/\"-operands must be lists or numbers");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void unary_minus_operation (void)
/* STACK EFFECTS: <value> -> <new_value>.
 * <new_value> is "-" <value>.
 * The actual operation depends on the type of the value. */
{
  push_number_value (-value_to_double (value_stack[--top]));
}

/* attribute path functions =================================================*/

GLOBAL value_t get_value_part (value_t value, value_t path)
/* Return the value part of <value> that is specified by the path <path>. 
 * If that value part doesn't exist, return NULL. */
{
  value_t component;
  value_t path_end = NEXT_VALUE (path);
  
  if (! IS_LIST (path))
    error ("path must be a list");

  for (component = path + 2; component < path_end; 
       component = NEXT_VALUE (component))
  {
    if (IS_SYMBOL (component))
      value = get_attribute (value, *component);
    else if (IS_NUMBER (component))
      value = get_element (value, value_to_int (component));
    else
      error ("path must contain symbols and numbers only");

    if (value == NULL)
      return NULL;
  }
  
  return value;
}

/*---------------------------------------------------------------------------*/

GLOBAL void build_path (int_t n)
/* STACK EFFECTS: <value_1> ... <value_n> -> <new_list>.
 * <new_list> is a path which contains <value_1>, ..., <value_n>. 
 * <value_1>, ..., <value_n> must be numbers, symbols or lists of numbers and 
 * symbols. If a value is a list, the elements of this list are inserted into
 * <new_list> instead of the value itself. */
{
  value_t new_list, v;
  int_t i, new_list_length;
  value_t *elements = value_stack + top - n;

  new_list_length = 2;
  for (i = 0; i < n; i++)
  {
    switch (TYPE (elements[i]))
    {
    case LIST_TYPE:
    {
      value_t element_end = NEXT_VALUE (elements[i]);
      
      for (v = elements[i] + 2; v < element_end; v = NEXT_VALUE (v))
      {
	if (! IS_SYMBOL (v) && ! IS_NUMBER (v))
	  error ("sublist in path may contain symbols and numbers only");
      }
      new_list_length += length_of_value (elements[i]) - 2;
      break;
    }
 
    case SYMBOL_TYPE:
    case NUMBER_TYPE:
      new_list_length += length_of_value (elements[i]);
      break;
      
    default:
      error ("value path may contain symbols, numbers and lists only");
    }
  }
    
  new_list = space_for_composed_value (LIST_TYPE, new_list_length);
  v = new_list + 2;
  for (i = 0; i < n; i++) 
  {
    if (IS_LIST (elements[i]))
    {
      copy_cells (v, elements[i] + 2, length_of_value (elements[i]) - 2);
      v += length_of_value (elements[i]) - 2;
    }
    else
    {
      copy_value (v, elements[i]);
      v = NEXT_VALUE (v);
    }
  }
  
  top -= n;
  push_value (new_list);
}

/*---------------------------------------------------------------------------*/

LOCAL void local_modify_value_part (void (*modifier) (void), 
				    int_t value_index, 
				    int_t path_index)
/* STACK EFFECTS: <value> <path> <mod_value> -> <value> <path> <new_value>.
 * <new_value> is <value>, but the part that is described by <path> is 
 * modified. <path> must be a list of symbols and numbers <e1, e2, .. , en>.
 * They will be used as nested attributes and indexes, so the part of <value>
 * that is actually modified is <old_value> := <value>.<e1>.<e2>..<en>. 
 * If this part does not exist, an error will be reported. Else the function 
 * <modifier> will be called on <old_value> and <mod_value>. 
 * The value returned by <modifier> will be entered in <value> in place of
 * <old_value>. */
{
  value_t value, selector;

  value = value_stack[top-3] + value_index;
  selector = get_element (value_stack[top-2], path_index);

  if (selector == NULL) /* No more selectors. */
  {
    insert_value (1, value);
    modifier ();
  }
  else
  {
    value_t subvalue;
    int_t subvalue_index;

    /* Find attribute in <value>. */
    if (IS_SYMBOL (selector))
      subvalue = get_attribute (value, value_to_symbol (selector));
    else if (IS_NUMBER (selector))
      subvalue = get_element (value, value_to_int (selector));
    else
      error ("path must consist of symbols and numbers");

    if (subvalue == NULL)
      error ("can't find element or attribute in value");

    subvalue_index = subvalue - value_stack[top-3];

    /* Go down recursively */
    local_modify_value_part (modifier, subvalue_index, path_index + 1);

    subvalue = value_stack[top-3] + subvalue_index;
    value = value_stack[top-3] + value_index;
    selector = get_element (value_stack[top-2], path_index);

    if (value_stack[top-1] == subvalue)
      value_stack[top-1] = value;
    else if (IS_SYMBOL (selector))
    {
      insert_value (1, value);
      replace_attribute (value_to_symbol (selector));
    }
    else /* IS_NUMBER (selector) */
    {
      insert_value (1, value);
      replace_element (value_to_int (selector));
    }
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void modify_value_part (void (*modifier) (void))
/* STACK EFFECTS: <value> <path> <mod_value> -> <new_value>.
 * <new_value> is <value>, but the part that is described by <path> is 
 * modified. <path> must be a list of symbols and numbers <e1, e2, .. , en>.
 * They will be used as nested attributes and indexes, so the part of <value>
 * that is actually modified is <old_value> := <value>.<e1>.<e2>..<en>. 
 * If this part does not exist, an error will be reported. Else the function 
 * <modifier> will be called on <old_value> and <mod_value>. 
 * The value returned by <modifier> will be entered in <value> in place of
 * <old_value>. */
{
  local_modify_value_part (modifier, 0, 1);
  value_stack[top-3] = value_stack[top-1];
  top -= 2;
}

/*---------------------------------------------------------------------------*/

GLOBAL void right_value (void)
/* STACK EFFECTS: <left_value> <right_value> -> <right_value>.
 * A modifier for "modify_value_part". */
{
  top--;
  value_stack[top-1] = value_stack[top];
}

/* functions for list/record iteration ======================================*/

GLOBAL void get_first_element (void)
/* STACK EFFECTS: <value> -> <new_value>.
 * If <value> is a list, then <new_value> is its first element (or NULL).
 * If <value> is a record, then <new_value> is its first attribute (or NULL).
 * If <value> is a number, then <new_value> is NULL (if <value> == 0),
 * 1 (if <value> > 0) or -1 (if <value> < 0). */
{
  value_t value = value_stack[top-1];

  top--;
  if (*value == NIL_SYMBOL)
    push_value (NULL);
  else
  {
    switch (TYPE (value))
    {
    case RECORD_TYPE:
    case LIST_TYPE:
      if (length_of_value (value) == 2)
	push_value (NULL); /* Return NULL if list or record is empty. */
      else
	push_value (value + 2);
      break;

    case NUMBER_TYPE:
    {
      int_t number = value_to_int (value);
      
      if (number > 0)
	push_number_value (1.0);
      else if (number < 0)
	push_number_value (-1.0);
      else
	push_value (NULL);
      break;
    }
    
    default:
      error ("can iterate on lists, records and numbers only");
    }
  }
}

/*---------------------------------------------------------------------------*/

extern void get_next_element (int_t index)
/* STACK EFFECTS: (nothing) -> (nothing).
 * <value> is <value_stack>[<index>-1], <element> is <value_stack>[<index>].
 * <value_stack>[<index>] will be set to <new_element>.
 * <element> must be the result of an application of "get_first_element" or 
 * "get_next_element" on <value>.
 * If <value> is a list, and <element> one of its elements,
 * then <new_element> is the successor of <element> (or NULL).
 * If <value> is a record, and <element> one of its attributes,
 * then <new_element> is the next attribute in <value> (or NULL).
 * If <value> is a positive number, and <element> a number smaller than
 * <value>, then <new_element> is <element> + 1.
 * If <value> is a negative number, and <element> a number greater than
 * <value>, then <new_element> is <element> - 1. */
{
  value_t value = value_stack[index-1];
  value_t element = value_stack[index];

  if (element == NULL)
    return;

  switch (TYPE (value))
  {
  case RECORD_TYPE:
    element = NEXT_ATTRIB (element);
    if (element >= NEXT_VALUE (value))
      element = NULL;
    break;
    
  case LIST_TYPE:
    element = NEXT_VALUE (element);
    if (element >= NEXT_VALUE (value))
      element = NULL;
    break;
    
  case NUMBER_TYPE:
  {
    int_t max_number = value_to_int (value);
    int_t number = value_to_int (element);
    
    if (max_number > 0 && number < max_number)
    {
      push_number_value (number + 1);
      element = value_stack[--top];
    }
    else if (max_number < 0 && number > max_number)
    {
      push_number_value (number - 1);
      element = value_stack[--top];
    }
    else
      element = NULL;
    break;
  }
  
  default:
    error ("can iterate on lists, records and numbers only");
  }
  value_stack[index] = element;
}

/* functions to compare values ==============================================*/

LOCAL void check_atom_list (value_t atoms)
/* Check if <atoms> is a list that contains of atoms only. */
{
  value_t v;
  value_t atoms_end = NEXT_VALUE (atoms);
  
  if (! IS_LIST (atoms))
    error ("atom list must be a list");
  
  for (v = atoms + 2; v < atoms_end; v++)
  {
    if (! IS_SYMBOL (v))
      error ("atom list must consist of symbols");
  }
}

/*---------------------------------------------------------------------------*/

LOCAL symbol_t next_symbol (value_t atoms, int_t lower_limit)
/* Return the smallest symbol in <atoms> that is greater than <lower_limit>.
 * Return SYMBOL_MAX if no such symbol exists. */
{
  symbol_t symbol;
  value_t v;
  value_t atoms_end = NEXT_VALUE (atoms);
  
  symbol = SYMBOL_MAX;
  for (v = atoms + 2; v < atoms_end; v++)
  {
    if (*v > lower_limit && *v < symbol)
      symbol = *v;
  }
  return symbol;
}

/*---------------------------------------------------------------------------*/

GLOBAL int_t compare_atom_lists (value_t atoms1, value_t atoms2)
/* Compare atom lists <atoms1> and <atoms2>.
 * Return -1 if <atoms1> < <atoms2>.
 * Return 0 if <atoms1> == <atoms2>.
 * Return 1 if <atoms1> > <atoms2>. */
{
  int_t limit1, limit2;

  check_atom_list (atoms1);
  check_atom_list (atoms2);

  limit1 = limit2 = -1;
  while (TRUE)
  {
    limit1 = next_symbol (atoms1, limit1);
    limit2 = next_symbol (atoms2, limit2);
    if (limit1 < limit2)
      return -1;
    else if (limit1 > limit2)
      return 1;
    else if (limit1 == SYMBOL_MAX)
      return 0;
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t values_equal (value_t value1, value_t value2)
/* Return a truth value indicating whether <value1> and <value2> are equal.
 * <value1> an <value2> must be of same type.
 * Refer to documentation to see what "equal" in Malaga really means. */
{
  if (TYPE (value1) != TYPE (value2))
  {
    if (*value1 != NIL_SYMBOL && *value2 != NIL_SYMBOL)
      error ("can compare values of same type only");
    
    return FALSE;
  }

  switch (TYPE (value1)) 
  {
  case SYMBOL_TYPE:
    return (*value1 == *value2);
    
  case STRING_TYPE:
    return (strcmp_no_case ((string_t) (value1 + 1), 
			    (string_t) (value2 + 1)) == 0);

  case LIST_TYPE:
  {
    value_t value1_end = NEXT_VALUE (value1);
    value_t value2_end = NEXT_VALUE (value2);
    
    /* Look for each value pair if they are equal. */ 
    value1 += 2;
    value2 += 2;
    while (value1 < value1_end && value2 < value2_end) 
    {
      if (! values_equal (value1, value2))
	return FALSE;
      
      value1 = NEXT_VALUE (value1);
      value2 = NEXT_VALUE (value2);
    }

    if (value1 == value1_end && value2 == value2_end)
      return TRUE;
    else
      return FALSE;
  }
  
  case RECORD_TYPE:
  {
    value_t value1_end = NEXT_VALUE (value1);
    value_t value2_end = NEXT_VALUE (value2);
    value_t v1, v2;
    
    /* Do the records have the same length? */
    if (length_of_value (value1) != length_of_value (value2))
      return FALSE;
    
    /* Check whether for every attribute in <value1>, there is one
     * in <value2> and that their values are equal. */
    for (v1 = value1 + 2; v1 < value1_end; v1 = NEXT_ATTRIB (v1))
    {
      /* Look for the same attribute in <value2>. */
      for (v2 = value2 + 2; v2 < value2_end; v2 = NEXT_ATTRIB (v2)) 
      {
	if (*v1 == *v2)
	  break;
      }
      
      if (v2 == value2_end || ! values_equal (v1 + 1, v2 + 1)) 
	/* We looked 'till end of value2 and didn't find attribute.
	 * Or they don't have the same values. */
	return FALSE;
    }
    
    return TRUE;
  }
  
  case NUMBER_TYPE:
    return (value_to_double (value1) == value_to_double (value2));

  default:
    error ("internal (value has bad type)");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t values_congruent (value_t value1, value_t value2)
/* Return a truth value indicating whether <value1> and <value2> have
 * at least one element in common.
 * <value1> and <value2> must both be symbols or lists. */
{
  value_t value1_end, value2_end, v1, v2;

  if (TYPE (value1) != TYPE (value2))
    error ("for congruency test, values must be of same type");

  if (IS_SYMBOL (value1))
  {
    value1 = get_atoms (value_to_symbol (value1));
    value2 = get_atoms (value_to_symbol (value2));
  }
  else if (! IS_LIST (value1))
    error ("for congruency test, values must be lists or symbols");

  value1_end = NEXT_VALUE (value1);
  value2_end = NEXT_VALUE (value2);
    
  /* Look for a common element. */
  for (v1 = value1 + 2; v1 < value1_end; v1 = NEXT_VALUE (v1)) 
  {
    for (v2 = value2 + 2; v2 < value2_end; v2 = NEXT_VALUE (v2)) 
    {
      if (values_equal (v1, v2))
	return TRUE;
    }
  }
  
  /* No common symbol found. */
  return FALSE;
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t value_in_value (value_t value1, value_t value2)
/* Return bool value saying if <value1> is element or attribute of <value2>.
 * <value2> must be a list or a record.
 * If <value2> is a record, then <value1> must be a symbol. */
{
  value_t value2_end = NEXT_VALUE (value2);

  if (IS_LIST (value2)) 
  {
    for (value2 += 2; value2 < value2_end; value2 = NEXT_VALUE (value2)) 
    {
      if (values_equal (value1, value2))
	return TRUE;
    }
  }
  else if (IS_RECORD (value2)) 
  {
    if (! IS_SYMBOL (value1))
      error ("only attributes can be found in a record using \"in\"");
    
    for (value2 += 2; value2 < value2_end; value2 = NEXT_ATTRIB (value2)) 
    {
      if (*value1 == *value2)
	return TRUE;
    }
  }
  else 
    error ("can use \"in\" with records and lists only");

  return FALSE;
}

/* functions to print values ================================================*/

LOCAL attribute_t **find_hidden_attribute (symbol_t symbol)
/* Find a hidden attribute in the attribute list and return it.
 * Return NULL if there is none. */
{
  attribute_t **attr_ptr;

  attr_ptr = &hidden_attributes;
  for (attr_ptr = &hidden_attributes; 
       *attr_ptr != NULL; 
       attr_ptr = &(*attr_ptr)->next)
  {
    if ((*attr_ptr)->symbol == symbol)
      break;
  }
  return attr_ptr;
}

/*---------------------------------------------------------------------------*/

GLOBAL symbol_t *get_hidden_attributes (void)
/* Get a SYMBOL_MAX-terminated vector of the currently hidden attributes. 
 * The vector must be freed after use. */
{
  int_t i;
  attribute_t *attr;
  symbol_t *vector;

  /* Count the attributes. */
  i = 0;
  for (attr = hidden_attributes; attr != NULL; attr = attr->next)
    i++;
  
  /* Create the new vector. */
  vector = new_vector (sizeof (symbol_t), i + 1);
  i = 0;
  for (attr = hidden_attributes; attr != NULL; attr = attr->next)
    vector[i++] = attr->symbol;
  vector[i] = SYMBOL_MAX;
    
  return vector;
}

/*---------------------------------------------------------------------------*/

GLOBAL void add_hidden_attribute (symbol_t attribute)
/* Add <attribute> to the list of currently hidden attributes. */
{
  attribute_t **attr_ptr = find_hidden_attribute (attribute);

  if (*attr_ptr == NULL)
  {
    *attr_ptr = new_mem (sizeof (attribute_t));
    (*attr_ptr)->next = NULL;
    (*attr_ptr)->symbol = attribute;
  } 
}

/*---------------------------------------------------------------------------*/

GLOBAL void remove_hidden_attribute (symbol_t attribute)
/* Remove <attribute> from the list of currently hidden attributes. */
{
  attribute_t **attr_ptr = find_hidden_attribute (attribute);
  
  if (*attr_ptr != NULL)
  {
    attribute_t *next_attr = (*attr_ptr)->next;

    free_mem (attr_ptr);
    *attr_ptr = next_attr;
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void clear_hidden_attributes (void)
/* Clear the list of currently hidden attributes. */
{
  while (hidden_attributes != NULL)
  {
    attribute_t *next_attr = hidden_attributes->next;

    free_mem (&hidden_attributes);
    hidden_attributes = next_attr;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void attribute_to_text (text_t text, 
			      value_t attribute, 
			      bool_t full_value)
/* Print <attribute> and its value (behind attribute). */
{
  bool_t attribute_hidden;
  attribute_t *attr;

  /* Is <attribute> a hidden attribute? */
  attribute_hidden = FALSE;
  if (! full_value)
  {
    for (attr = hidden_attributes; attr != NULL; attr = attr->next)
    {
      if (attr->symbol == *attribute)
      {
	attribute_hidden = TRUE;
	break;
      }
    }
  }

  if (attribute_hidden) 
  {
    add_to_text (text, "(");
    add_to_text (text, get_symbol_name (*attribute));
    add_to_text (text, ")");
  } 
  else 
  {
    add_to_text (text, get_symbol_name (*attribute));
    add_to_text (text, ": ");
    value_to_text (text, attribute + 1, full_value);
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void value_to_text (text_t text, value_t value, bool_t full_value)
/* Convert <value> to a format readable for humans and add it to <text>.
 * which extends to <output_end> (this is a pointer to the first byte after
 * <output>. The pointer returned points to the EOS of the built string.
 * If <full_value> == TRUE, show all attributes, even those that are hidden. */
{
  value_t value_end;

  if (value == NULL)
    return;

  value_end = NEXT_VALUE (value);

  switch (TYPE (value)) 
  {
  case SYMBOL_TYPE:
    add_to_text (text, get_symbol_name (*value));
    break;

  case STRING_TYPE:
  {
    string_t string = new_string_readable ((string_t) (value + 1), NULL);

#ifdef HANGUL
    decode_hangul (&string);
#endif
    add_to_text (text, string);
    free_mem (&string);
    break;
  }

  case LIST_TYPE:
  {
    value_t element;

    add_to_text (text, "<");
    
    /* Print elements. */
    for (element = value + 2; 
	 element < value_end; 
	 element = NEXT_VALUE (element))
    {
      if (element > value + 2)
	add_to_text (text, ", ");

      value_to_text (text, element, full_value);
    }
    
    add_to_text (text, ">");
    break; 
  }
    
  case RECORD_TYPE: 
    add_to_text (text, "[");

    switch (attribute_order)
    {
      
    case INTERNAL_ORDER:
    {
      value_t attribute;
      
      for (attribute = value + 2; 
	   attribute < value_end;
	   attribute = NEXT_ATTRIB (attribute))
      {
	if (attribute > value + 2)
	  add_to_text (text, ", ");

	attribute_to_text (text, attribute, full_value);
      } 
      break;
    }

    case ALPHABETIC_ORDER:
    {
      string_t last_name; /* last attribute name that has been printed */
      
      last_name = NULL;
      while (TRUE) 
      {
	value_t attrib, next_attrib;
	string_t name, next_name; /* names of the above attributes */

	/* Find the next attribute to be printed. */
	next_attrib = NULL;
	next_name = NULL;
	for (attrib = value + 2; 
	     attrib < value_end; 
	     attrib = NEXT_ATTRIB (attrib)) 
	{
	  name = get_symbol_name (*attrib);
	  if ((last_name == NULL || strcmp_no_case (name, last_name) > 0)
	      && (next_name == NULL || strcmp_no_case (name, next_name) < 0))
	  {
	    next_attrib = attrib;
	    next_name = name;
	  }
	}
	
	if (next_attrib == NULL)
	  break;
	
	if (last_name != NULL)
	  add_to_text (text, ", ");
	
	attribute_to_text (text, next_attrib, full_value);	
	last_name = next_name; 
      }
      break;
    }

    case DEFINITION_ORDER:
    {
      symbol_t last_symbol; /* last attribute symbol that has been printed */
      
      last_symbol = INFO_MAX;
      while (TRUE) 
      {
	value_t attrib, next_attrib;
	
	/* Find the next attribute to be printed. */
	next_attrib = NULL;
	for (attrib = value + 2; 
	     attrib < value_end; 
	     attrib = NEXT_ATTRIB (attrib)) 
	{
	  if ((last_symbol == INFO_MAX || *attrib > last_symbol)
	      && (next_attrib == NULL || *attrib < *next_attrib))
	    next_attrib = attrib;
	}
	
	if (next_attrib == NULL)
	  break;
	
	if (last_symbol != INFO_MAX)
	  add_to_text (text, ", ");
	
	attribute_to_text (text, next_attrib, full_value);
	last_symbol = *next_attrib;
      }
      break;
    }
    }

    add_to_text (text, "]");
    break;
    
  case NUMBER_TYPE:
  {
    string_t number_string = double_to_string (value_to_double (value));
    
    add_to_text (text, number_string);
    free_mem (&number_string);
    break;
  }

  default:
    error ("internal (value has bad type)");
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t value_to_readable (value_t value, bool_t full_value)
/* Return <value> in a format readable for humans. 
 * If <full_value> == TRUE, show all attributes, even those that are hidden.
 * The result must be freed with "free" after use. */
{
  clear_text (&text);
  value_to_text (text, value, full_value);
  return new_string (text_string (text), NULL);
}

/* end of file ==============================================================*/
