// Maria temporal binary operator class -*- c++ -*-

#ifdef __GNUC__
# pragma implementation
#endif // __GNUC__
#include "TemporalBinop.h"
#include "BoolType.h"
#include "Net.h"
#include "Property.h"
#include "Printer.h"

/** @file TemporalBinop.C
 * Temporal binary operators
 */

/* Copyright  1998-2002 Marko Mkel (msmakela@tcs.hut.fi).

   This file is part of MARIA, a reachability analyzer and model checker
   for high-level Petri nets.

   MARIA is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   MARIA is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

TemporalBinop::TemporalBinop (enum Op op,
			      class Expression& left,
			      class Expression& right) :
  Expression (),
  myOp (op), myLeft (&left), myRight (&right)
{
  assert (myLeft->getType ()->getKind () == Type::tBool);
  assert (myRight->getType ()->getKind () == Type::tBool);
  setType (Net::getBoolType ());
}

TemporalBinop::~TemporalBinop ()
{
  myLeft->destroy ();
  myRight->destroy ();
}

class Expression*
TemporalBinop::ground (const class Valuation& valuation,
		       class Transition* transition,
		       bool declare)
{
  class Expression* left = myLeft->ground (valuation, transition, declare);
  if (!left) return NULL;
  class Expression* right = myRight->ground (valuation, transition, declare);
  if (!right) { left->destroy (); return NULL; }

  assert (valuation.isOK ());

  if (left == myLeft && right == myRight) {
    left->destroy ();
    right->destroy ();
    return copy ();
  }
  else
    return (new class TemporalBinop (myOp, *left, *right))->cse ();
}

class Expression*
TemporalBinop::substitute (class Substitution& substitution)
{
  class Expression* left = myLeft->substitute (substitution);
  class Expression* right = myRight->substitute (substitution);

  if (!left || !right) {
    left->destroy ();
    right->destroy ();
    return NULL;
  }

  if (left == myLeft && right == myRight) {
    left->destroy ();
    right->destroy ();
    return copy ();
  }
  else
    return (new class TemporalBinop (myOp, *left, *right))->cse ();
}

bool
TemporalBinop::depends (const class VariableSet& vars,
			bool complement) const
{
  return
    myLeft->depends (vars, complement) ||
    myRight->depends (vars, complement);
}

bool
TemporalBinop::forVariables (bool (*operation)
			(const class Expression&,void*),
			void* data) const
{
  return
    myLeft->forVariables (operation, data) &&
    myRight->forVariables (operation, data);
}

class Ltl*
TemporalBinop::toFormula (class Property& property)
{
  switch (myOp) {
  case Until:
    return property.addBinop (Property::opUntil, *myLeft, *myRight);
  case Release:
    return property.addBinop (Property::opRelease, *myLeft, *myRight);
  }
  assert (false);
  return 0;
}

/** Convert an operator to a string
 * @param op	the operator to convert
 * @return	a string corresponding to the operator
 */
static const char*
getOpString (enum TemporalBinop::Op op)
{
  switch (op) {
  case TemporalBinop::Until:
    return "until";
  case TemporalBinop::Release:
    return "release";
  }

  return "???";
}

void
TemporalBinop::display (const class Printer& printer) const
{
  printer.delimiter ('(')++;
  myLeft->display (printer);
  --printer.delimiter (')');

  printer.printRaw (::getOpString (myOp));
  printer.delimiter ('(')++;
  myRight->display (printer);
  --printer.delimiter (')');
}
