/*
 * Copyright (C) 2007 Intel Corporation
 *
 * Author:  Xu Bo <bo.b.xu@intel.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gtk/gtk.h>
#include <glib-object.h>
#include <libhildondesktop/libhildondesktop.h>
#include <time.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <string.h>
#include "clock-plugin.h"

#include <errno.h>
#include <sys/resource.h>

#define THEME_DIR "/usr/share/themes/mobilebasic"
#define BTN_HEIGHT  52
#define BTN_WIDTH_FOR_800 256
#define BTN_WIDTH_FOR_1024 320 
#define MAX_READ_SIZE 50
#define CONFIG_FILE "/etc/clock-plugin.conf"
#define DEFAULT_TIME_FORMAT "\%a \%b \%d, \%r\%p"
#define CLOCK_APPLET_CMD  "moblin-applets time"
//#define CLOCK_APPLET_CMD  "mousepad /etc/apt/sources.list"

HD_DEFINE_PLUGIN (ClockPlugin, clock_plugin, TASKNAVIGATOR_TYPE_ITEM);

static void clock_finalize (GObject *object);

static gboolean clock_update_time (gpointer data)
{
	ClockPlugin *clock_plugin = (ClockPlugin *)data;
	struct timeval  time_val;
  	struct tm      *tm_time;
  	gchar          *tz_str;
	gchar		time_str[60];
	
	//reset timezone
	tz_str = g_strdup (g_getenv ("TZ"));
  	g_unsetenv ("TZ");
	if (tz_str)
    	{
     		g_setenv ("TZ", tz_str, FALSE);
      		g_free (tz_str);
    	}
  	gettimeofday (&time_val, NULL);
	tm_time = localtime (&time_val.tv_sec);
	strftime (time_str, G_N_ELEMENTS (time_str), clock_plugin->time_format, tm_time);

	gtk_label_set_text((GtkLabel*)clock_plugin->time_lb, time_str);

	if (clock_plugin->timeout_id)
    		g_source_remove (clock_plugin->timeout_id);
	//update per second
 	clock_plugin->timeout_id = g_timeout_add (1000,
                                              clock_update_time,
                                              clock_plugin);
	return FALSE;
}

// Start the time/date edit dialog
static void on_clicked (GtkWidget *button, const gchar *app)
{
	gchar *program = NULL;
	GError *error = NULL;
	gint argc;
	gchar **argv;
	GPid child_pid;
	gchar *space;

	space = strchr(app, ' ');
	if (space) {
		gchar *cmd = g_strdup (app);
		cmd[space - app] = 0;
		gchar *exc = g_find_program_in_path (cmd);
		program = g_strconcat (exc, space, NULL);
		g_free (exc);
		g_free (cmd);
	} else {
		program = g_find_program_in_path (app);
	}
	if (!program) {
		g_warning("Attempt to exec invalid entry: %s", app);
		return;
	}
	if (g_shell_parse_argv (program, &argc, &argv, &error)) {
		g_spawn_async ( NULL,              //Child's current working directory or NULL to use parent's
				argv, //Child's argument vector. [0] is the path of the program to execute
				NULL,             //Child's environment or NULL to inherit parent's
				(GSpawnFlags)0,   //flags from GSpawnFlags
				NULL,             //Function to run in the child just before exec
				NULL,             //User data for child_setup
				&child_pid,       //Return location for child process ID or NULL
				&error);          //Return location for error
	}
	if (error) {
		g_warning ("Date/time dialog could not be started. (Error: %s)",
			   error->message);
		g_clear_error (&error);
	} else {
		setpriority (PRIO_PROCESS, child_pid, 0);
	}
}

static void clock_plugin_init (ClockPlugin *clock_plugin)
{
   	GtkWidget *btn;
	GtkWidget *time_lb;
//	GdkColor  ft_color;
	//for reading the configure file
	struct stat config_stat;
	FILE *fd_config;
	gchar readbuf[MAX_READ_SIZE];
	gchar *param;

	gboolean format_inited = 0;
   	btn = clock_plugin->btn = gtk_button_new();

	//size of clock depending on size of window (1024= 320, 800 = 256)
	int scn_width = 800;
	GdkScreen *screen = gtk_widget_get_screen ((GtkWidget*)clock_plugin);
	if (screen != NULL) {
	   scn_width = gdk_screen_get_width(screen);
	}
	gtk_widget_set_size_request ((GtkWidget*)clock_plugin, (scn_width == 1024 ? BTN_WIDTH_FOR_1024 : BTN_WIDTH_FOR_800), BTN_HEIGHT);

	time_lb = clock_plugin->time_lb = gtk_label_new("");

	//set the font size smaller
	PangoAttribute *pa_size = pango_attr_size_new (14000);
	pa_size->start_index = 0;
	pa_size->end_index = -1;
	PangoAttribute *pa_font = pango_attr_family_new ("Sans");
	pa_font->start_index = 0;
	pa_font->end_index = -1;
	PangoAttribute *pa_color = pango_attr_foreground_new (0xE2E1,0xE2E1,0xE2E1);
	pa_color->start_index = 0;
	pa_color->end_index = -1;
	PangoAttrList *pl = pango_attr_list_new();
	pango_attr_list_insert(pl,pa_size);
	pango_attr_list_insert(pl,pa_font);
	pango_attr_list_insert(pl,pa_color);

	gtk_label_set_attributes ((GtkLabel*)time_lb, pl);
	pango_attr_list_unref (pl);

	//set color (note: this causes the color to change when you hovered over it)
	//gdk_color_parse("white", &ft_color);
	//gtk_widget_modify_fg(clock_plugin->time_lb, GTK_STATE_NORMAL, &ft_color);

	//add time_lb
	gtk_container_add (GTK_CONTAINER (btn), time_lb);
   	gtk_widget_show_all (btn);
   	gtk_container_add (GTK_CONTAINER (clock_plugin), btn);

	//init time format
	if (stat(CONFIG_FILE, &config_stat) < 0) {
		//no config file, use default format
		strcpy(clock_plugin->time_format, DEFAULT_TIME_FORMAT);
	}
	else {
		fd_config = fopen(CONFIG_FILE,"r");
		while (fgets(readbuf, MAX_READ_SIZE, fd_config) != NULL) {
			//should have a "Format" field
	                if (strncmp(readbuf, "Format:", 7) == 0) {
				//sscanf(readbuf + 7, "%s", parambuf);
  			        param = readbuf+7;
				g_print ("*******\n\n\nClock param: %s\n\n\n******", param);
				if(strncmp(param, "Simple", 6) == 0) {
					//use simple format
					strcpy(clock_plugin->time_format,"\%T \%p");
				} else if(strncmp(param, "Full", 4) == 0) {
					//use full format, default format
					strcpy(clock_plugin->time_format,"\%a \%b \%d, \%R \%p");
				} else {
					//assume correct format was given (e.g. %A %B %D )
					//http://www.opengroup.org/onlinepubs/007908799/xsh/strftime.html
					strncpy(clock_plugin->time_format, param, strlen(param)-1);
					//strip extra "0" and lower am/pm
				}
				format_inited = 1;
				break;  //found format, that's all we need
	                }
		}
		//if no "Format" field
		if (format_inited == 0) {
			//use a default format
			strcpy(clock_plugin->time_format, DEFAULT_TIME_FORMAT);
		}
		fclose(fd_config);
	}
	clock_update_time(clock_plugin);

	//when clicked, launch time/date setting
	g_signal_connect (btn, "clicked", 
			  G_CALLBACK (on_clicked), CLOCK_APPLET_CMD);
}



static void clock_plugin_class_init (ClockPluginClass *class)
{
        GObjectClass   *object_class = G_OBJECT_CLASS (class);
        object_class->finalize    = clock_finalize;
}

static void clock_finalize (GObject *object)
{
   	ClockPlugin *clock_plugin = CLOCK_PLUGIN (object);	
	if (clock_plugin->timeout_id)
    		g_source_remove (clock_plugin->timeout_id);
   	G_OBJECT_CLASS (clock_plugin_parent_class)->finalize (object);
}
