/* massXpert - the true massist's program.

   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Qt includes
#include <QChar>
#include <QString>


/////////////////////// Local includes
#include "calcOptions.hpp"


namespace massXpert
{

  //! Constructs a calculation options object.
  CalcOptions::CalcOptions()
  {
    m_deepCalculation = false;
    m_massType = MXT_MASS_BOTH;
    m_capping = MXT_CAP_BOTH;
    m_monomerEntities = MXT_MONOMER_CHEMENT_NONE;
    m_polymerEntities = MXT_POLYMER_CHEMENT_NONE;
  }


  //! Constructs a calculation options object.
  CalcOptions::CalcOptions(bool deepCalculation,
			    int massType,
			    int capping,
			    int monomerEntities,
			    int polymerEntities)
    : m_deepCalculation(deepCalculation),
      m_massType(massType),
      m_capping(capping),
      m_monomerEntities(monomerEntities),
      m_polymerEntities(polymerEntities)
  {
  }


  //! Constructs a calculation options object.
  CalcOptions::CalcOptions(const CalcOptions &other)
    : m_deepCalculation(other.m_deepCalculation),
      m_massType(other.m_massType),
      m_capping(other.m_capping),
      m_monomerEntities(other.m_monomerEntities),
      m_polymerEntities(other.m_polymerEntities),
      m_selectionType(other.m_selectionType)
  {
  }


  //! Destroys the calculation options object.
  CalcOptions::~CalcOptions()
  {
    // Free all the coordinates from the list.
    while(!m_coordinateList.isEmpty())
      delete(m_coordinateList.takeFirst());
  }
  

  //! Modifies \p other to be identical to \p this.
  /*!  \param other calculation options instance.
   */
  void 
  CalcOptions::clone(CalcOptions *other) const
  {
    Q_ASSERT(other != 0);
  
    other->m_deepCalculation = m_deepCalculation;
    other->m_massType = m_massType;
    other->m_capping = m_capping;
    other->m_monomerEntities = m_monomerEntities;
    other->m_polymerEntities = m_polymerEntities;
    
    other->m_coordinateList.empty();
    other->setCoordinateList(m_coordinateList);

    other->setSelectionType (m_selectionType);
  }


  //! Modifies \p this  to be identical to \p other.
  /*!  \param other calculation options instance to be used as a mold.
   */
  void 
  CalcOptions::mold(const CalcOptions &other)
  {
    if (&other == this)
      return;
  
    m_deepCalculation = other.m_deepCalculation;
    m_massType = other.m_massType;
    m_capping = other.m_capping;
    m_monomerEntities = other.m_monomerEntities;
    m_polymerEntities = other.m_polymerEntities;  

    m_coordinateList.empty();
    setCoordinateList(other.m_coordinateList);

    setSelectionType(other.m_selectionType);
  }

  

  //! Assigns other to \p this calculation options instance.
  /*! \param other calculation options instance used as the mold to set
    values to \p this instance.
  
    \return a reference to \p this calculation options instance.
  */
  CalcOptions & 
  CalcOptions::operator =(const CalcOptions &other)
  {
    if (&other != this)
      mold(other);
  
    return *this;
  }


  //! Sets if the calculation is deep.
  /*!  A deep calculation is a calculation that involves the
    recalculation of the mass of the monomers.

    \param deep true if the calculation must be deep, false otherwise.
  */
  void 
  CalcOptions::setDeepCalculation(bool deep)
  {
    m_deepCalculation = deep;
  }


  //! Returns if the calculation is deep.
  /*!  A deep calculation is a calculation that involves the
    recalculation of the mass of the monomers.

    \return true if the calculation is deep, false otherwise.
  */
  bool 
  CalcOptions::isDeepCalculation() const
  {
    return m_deepCalculation;
  }


  void 
  CalcOptions::setCoordinateList(const Coordinates &coord)
  {
    m_coordinateList.setCoordinates(coord);
  }
  
  
  void 
  CalcOptions::setCoordinateList(const CoordinateList &list)
  {
    m_coordinateList.setCoordinates(list);
  }
  
  
  const CoordinateList & 
  CalcOptions::coordinateList() const
  {
    return m_coordinateList;
  }
  

  //! Sets the mass type.
  /*! 
    \param value New mass type.
  */
  void 
  CalcOptions::setMassType(int value)
  {
    m_massType = value;
  }


  //! Returns the mass type.
  /*! 
    \return the mass type.
  */
  int 
  CalcOptions::massType() const
  {
    return m_massType;
  }

  void 
  CalcOptions::setSelectionType(SelectionType type)
  {
    m_selectionType = type;
  }
  
  
  SelectionType
  CalcOptions::selectionType() const
  {
    return m_selectionType;
  }
  
  


  //! Sets the capping.
  /*! 
    \param value New capping.
  */
  void 
  CalcOptions::setCapping(int value)
  {
    m_capping = value;
  }


  //! Returns the capping.
  /*! 
    \return the capping.
  */
  int 
  CalcOptions::capping() const
  {
    return m_capping;
  }


  //! Sets the monomer entities.
  /*! 
    \param value New monomer entities.
  */
  void 
  CalcOptions::setMonomerEntities(int value)
  {
    m_monomerEntities = value;
  }


  //! Returns the monomer entities.
  /*! 
    \return the monomer entities.
  */
  int 
  CalcOptions::monomerEntities() const
  {
    return m_monomerEntities;
  }


  //! Sets the polymer entities.
  /*! 
    \param value New polymer entities.
  */
  void 
  CalcOptions::setPolymerEntities(int value)
  {
    m_polymerEntities = value;
  }


  //! Returns the polymer entities.
  /*! 
    \return the polymer entities.
  */
  int 
  CalcOptions::polymerEntities() const
  {
    return m_polymerEntities;
  }

} // namespace massXpert
