/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Local includes
#include "cleaveOligomerTreeViewModel.hpp"
#include "application.hpp"


namespace massXpert
{

  CleaveOligomerTreeViewModel::CleaveOligomerTreeViewModel 
 (OligomerList *dataList, QObject *parent) 
    : QAbstractItemModel(parent)
  {
    QList<QVariant> rootData;

    Q_ASSERT(dataList);
    Q_ASSERT(parent);
  
    mp_parentDlg = static_cast<CleavageDlg *>(parent);
  
    rootData << tr("Partial") << tr("Name") << tr("Coordinates")
	     << tr("Mono") << tr("Avg") 
	     << tr("Charge") << tr("Modif");
  
    mpa_rootItem = new CleaveOligomerTreeViewItem(rootData);
  
    mp_list = dataList;
    setupModelData(mpa_rootItem);
  }


  CleaveOligomerTreeViewModel::~CleaveOligomerTreeViewModel()
  {
    delete mpa_rootItem;
  }


  CleavageDlg *
  CleaveOligomerTreeViewModel::getParentDlg()
  {
    return mp_parentDlg;
  }


  void 
  CleaveOligomerTreeViewModel::setTreeView(QTreeView *treeView)
  {
    Q_ASSERT(treeView);
  
    mp_treeView = treeView;
  }


  QTreeView *
  CleaveOligomerTreeViewModel::treeView()
  {
    return mp_treeView;
  }


  QVariant 
  CleaveOligomerTreeViewModel::data(const QModelIndex &index, 
				     int role) const
  {
    if (!index.isValid())
      return QVariant();
  
    if (role != Qt::DisplayRole)
      return QVariant();

    CleaveOligomerTreeViewItem *item = 
      static_cast<CleaveOligomerTreeViewItem *>(index.internalPointer());

    return item->data(index.column());
  }


  // bool
  // CleaveOligomerTreeViewModel::setData(const QModelIndex & index, 
  // 					  const QVariant & value, 
  // 					  int role)
  // {
  //   return true;
  // }


  QVariant 
  CleaveOligomerTreeViewModel::headerData(int section, 
					   Qt::Orientation orientation,
					   int role) const
  {
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole)
      return mpa_rootItem->data(section);
  
    return QVariant();
  }


  Qt::ItemFlags 
  CleaveOligomerTreeViewModel::flags(const QModelIndex &index) const
  {
    // We want to be able to drag items away from the treeView,
    // typically to the mzLab treeViews.
    Qt::ItemFlags defaultFlags = QAbstractItemModel::flags(index);
    
    if (index.isValid())
      return Qt::ItemIsDragEnabled | defaultFlags;
    else
      return defaultFlags;
  }
  
  
  QModelIndex 
  CleaveOligomerTreeViewModel::index(int row, int column,
				      const QModelIndex &parent) const
  {
    CleaveOligomerTreeViewItem *parentItem = 0;
  
    if (!parent.isValid())
      parentItem = mpa_rootItem;
    else
      parentItem = 
	static_cast<CleaveOligomerTreeViewItem *> 
	(parent.internalPointer());
  
    CleaveOligomerTreeViewItem *childItem = parentItem->child(row);
    if (childItem)
      return createIndex(row, column, childItem);
    else
      return QModelIndex();
  }


  QModelIndex 
  CleaveOligomerTreeViewModel::parent(const QModelIndex &index) const
  {
    if (!index.isValid())
      return QModelIndex();

    CleaveOligomerTreeViewItem *childItem = 
      static_cast<CleaveOligomerTreeViewItem *>(index.internalPointer());
  
    CleaveOligomerTreeViewItem *parentItem = childItem->parent();

    if (parentItem == mpa_rootItem)
      return QModelIndex();
  
    return createIndex(parentItem->row(), 0, parentItem);
  }


  int 
  CleaveOligomerTreeViewModel::rowCount(const QModelIndex &parent) const
  {
    CleaveOligomerTreeViewItem *parentItem;
  
    if (!parent.isValid())
      parentItem = mpa_rootItem;
    else
      parentItem = 
	static_cast<CleaveOligomerTreeViewItem *> 
	(parent.internalPointer());
  
    return parentItem->childCount();
  }


  int
  CleaveOligomerTreeViewModel::columnCount(const QModelIndex &parent) const
  {
    if (parent.isValid())
      return static_cast<CleaveOligomerTreeViewItem *> 
	(parent.internalPointer())->columnCount();
    else
      return mpa_rootItem->columnCount();
  }


  void 
  CleaveOligomerTreeViewModel::addOligomer(CleaveOligomer *oligomer)
  {
    Q_ASSERT(oligomer);
    
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    QList<QVariant> columnData;
    
    // We have to prepare the coordinates of the oligomer, which
    // might be a cross-linked oligomer...
	  
    QString text ;
	  
    int oligomerCount =
      static_cast<CoordinateList *>(oligomer)->size();
	  
    for (int kter = 0; kter < oligomerCount; ++kter)
      {
	Coordinates *coordinates = 
	  static_cast<CoordinateList *>(oligomer)->at(kter);
	      
	text += QString("[%1-%2]")
	  .arg(coordinates->start() + 1)
	  .arg(coordinates->end() + 1);
      }
    
    // And now convert the masses from double to string, so
    // that we can use our conversion system.

    QString mono = oligomer->mono(locale, MXP_OLIGOMER_DEC_PLACES);
    // locale.toString(oligomer->mono(), 'f', 5);

    QString avg = oligomer->avg(locale, MXP_OLIGOMER_DEC_PLACES);
      // locale.toString(oligomer->avg(), 'f', 5);
	    
    columnData  
      << oligomer->partialCleavage() << oligomer->name() << text
      << mono << avg
      << oligomer->charge() << oligomer->isModified();
	  
    CleaveOligomerTreeViewItem *newItem = 
      new CleaveOligomerTreeViewItem(columnData, mpa_rootItem);

    newItem->setOligomer(oligomer);
    
    int newRow = rowCount();

    mpa_rootItem->appendChild(newItem); 
  
    // There is no need to produce a model index for the parent, as we
    // know we are going to insert new row in the root of the tree(this
    // tree has no branches). Hence the QModelIndex() calls below.

    beginInsertRows(QModelIndex(), newRow, newRow);
    insertRow(newRow, QModelIndex());
    endInsertRows();

    emit layoutChanged();
  }
  
  
  int
  CleaveOligomerTreeViewModel::addOligomers(OligomerList *oligomerList)
  {
 //    qDebug() <<__FILE__ << __LINE__
// 	     << "Enter addOligomers";
    
    Q_ASSERT(oligomerList);
    
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();
    
    // This is going to be the index of the first row to insert.
    int initialRowCount = rowCount();
    int appendedOligomerCount = 0;
    
    while(oligomerList->size())
      {
	
	CleaveOligomer *oligomer = 
	  static_cast<CleaveOligomer *>(oligomerList->takeFirst());

	QList<QVariant> columnData;
	
	// We have to prepare the coordinates of the oligomer, which
	// might be a cross-linked oligomer...
	
	QString text ;
	
	int oligomerCount =
	  static_cast<CoordinateList *>(oligomer)->size();
	
	for(int kter = 0; kter < oligomerCount; ++kter)
	  {
	    Coordinates *coordinates = 
	      static_cast<CoordinateList *>(oligomer)->at(kter);
	    
	    text += QString("[%1-%2]")
	      .arg(coordinates->start() + 1)
	      .arg(coordinates->end() + 1);
	  }
	
	// And now convert the masses from double to string, so
	// that we can use our conversion system.
	
	QString mono = oligomer->mono(locale, MXP_OLIGOMER_DEC_PLACES);
	// locale.toString(oligomer->mono(), 'f', 5);
	
	QString avg = oligomer->avg(locale, MXP_OLIGOMER_DEC_PLACES);
	// locale.toString(oligomer->avg(), 'f', 5);
	
	columnData  
	  << oligomer->partialCleavage() << oligomer->name() << text
	  << mono << avg
	  << oligomer->charge() << oligomer->isModified();
	
	CleaveOligomerTreeViewItem *newItem = 
	  new CleaveOligomerTreeViewItem(columnData, mpa_rootItem);
	
	newItem->setOligomer(oligomer);
	
	mp_list->append(oligomer);	
	mpa_rootItem->appendChild(newItem); 
	++appendedOligomerCount;

// 	qDebug() <<__FILE__ << __LINE__
// 		 << "Appended oligomer:" << oligomer->name();
      }
    
//     qDebug() <<__FILE__ << __LINE__
// 	     << "appendedOligomerCount:" << appendedOligomerCount;
    
    // At this point we have transferred all the oligomers from the
    // list passed as parameter into the list handled by this
    // model. We have to insert the treeview rows now.

    // There is no need to produce a model index for the parent, as we
    // know we are going to insert the new rows in the root of the
    // tree(this tree has no branches). Hence the QModelIndex()
    // calls below.

    // Note that we cannot make any insertion if no oligomer was
    // generated in the first place(otherwise the -1 logic below will
    // induce a crash)!

    if (appendedOligomerCount)
      {
	beginInsertRows(QModelIndex(), 
			 initialRowCount, 
			 initialRowCount + appendedOligomerCount - 1);
	
	insertRows(initialRowCount, appendedOligomerCount - 1, QModelIndex());
	
	endInsertRows();
	
	emit layoutChanged();
      }
    
//     qDebug() <<__FILE__ << __LINE__
// 	     << "Exit addOligomers";

    return appendedOligomerCount;
  }
  
  
  CleaveOligomer *
  CleaveOligomerTreeViewModel::removeOligomer(const QModelIndex &index)
  {
    if (!index.isValid())
      return 0;
    
    int row = index.row();
    
    CleaveOligomer *oligomerFromList = 
      static_cast<CleaveOligomer *>(mp_list->at(row));

//     qDebug() <<__FILE__ << __LINE__
// 	     << "removeOligomer oligomerFromList:" << oligomerFromList->name();
        
//     CleaveOligomerTreeViewItem *item = 
//       static_cast<CleaveOligomerTreeViewItem *> 
//      (index.internalPointer());
    
    //   Oligomer *oligomerFromItem = static_cast<Oligomer *>(item->oligomer());
    //   qDebug() << "oligomerFromList:" << oligomerFromList->name();

    // Sanity check
    //if(oligomerFromList != oligomerFromItem)
    //  qFatal("Fata error at %s@%d. Program aborted.",
    //__FILE__, __LINE__);
    
    mp_list->removeAt(row);

//     qDebug() <<__FILE__ << __LINE__
// 	     << "removeOligomer  mp_list->removeAt(row):" << row;
    
    beginRemoveRows(QModelIndex(), index.row(),index. row());
    delete(mpa_rootItem->takeChild(index.row()));
    endRemoveRows(); 
    
    return oligomerFromList;
  }
  

  int
  CleaveOligomerTreeViewModel::removeOligomers()
  {
    // This is going to be the index of the first row to insert.
    int initialRowCount = rowCount();
    int removedOligomerCount = 0;
  
//     qDebug() << __FILE__ << __LINE__ 
// 	     << "rowCount:" << rowCount();
    
    while(rowCount())
      {
	CleaveOligomer *oligomer = 
	  static_cast<CleaveOligomer *>(mp_list->takeAt(0));

// 	qDebug() << __FILE__ << __LINE__ 
// 		 << "rowCount:" << rowCount();
	
	delete(mpa_rootItem->takeChild(0));
	delete oligomer;
	++removedOligomerCount;
      }
    
//     qDebug() <<__FILE__ << __LINE__
//  	     << "removedOligomerCount:" << removedOligomerCount;
    
    if (initialRowCount)
      {
	// Serious bug that appeared when switching from Qt 4.4.x
	// 	(x=3, if I recall correctly) to Qt 4.5.2.  Indeed, a
	// 	bug, as the prototype for beginRemoveRows() is
	// 	beginRemoveRows(const QModelIndex &parent, int first,
	// 	int last), with first and last the indices of the rows
	// 	to be removed.  Here, we want to remove all the rows,
	// 	that is index first=0 and index last=(initialRowCount
	// 	- 1) AND NOT what's commented here beneath.
	// 	beginRemoveRows(QModelIndex(), initialRowCount,
	// 	initialRowCount + removedOligomerCount);
	
	beginRemoveRows(QModelIndex(), 
			 0, 
			 initialRowCount - 1);
	
	//    delete(mpa_rootItem->takeChild(index.row()));
	
	endRemoveRows(); 
      }
    
//     qDebug() << __FILE__ << __LINE__ 
// 	     << "removedOligomerCount:" << removedOligomerCount;
    
    return removedOligomerCount;
  }
  


  bool 
  CleaveOligomerTreeViewModel::setupModelData 
 (CleaveOligomerTreeViewItem *parent)
  {
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    QList<CleaveOligomerTreeViewItem *> parents;
    CleaveOligomerTreeViewItem *currentItem = 0;
  
    CleaveOligomer *oligomer = 0;
  
    Q_ASSERT(parent);
  
    // Start with populating the very first item of the treeviewmodel.
    parents << parent;
  
    for (int iter = 0; iter < mp_list->size(); ++iter)
      {
	oligomer = 
	  static_cast<CleaveOligomer *>(mp_list->at(iter));
	    
	QList<QVariant> columnData;
	  
	// We have to prepare the coordinates of the oligomer, which
	// might be a cross-linked oligomer...
	  
	QString text ;
	  
	int oligomerCount =
	  static_cast<CoordinateList *>(oligomer)->size();
	  
	for(int jter = 0; jter < oligomerCount; ++jter)
	  {
	    Coordinates *coordinates = 
	      static_cast<CoordinateList *>(oligomer)->at(jter);
	      
	    text += QString("[%1-%2]")
	      .arg(coordinates->start() + 1)
	      .arg(coordinates->end() + 1);
	  }
		 
	// And now convert the masses from double to string, so
	// that we can use our conversion system.

	QString mono = oligomer->mono(locale, MXP_OLIGOMER_DEC_PLACES);
	// locale.toString(oligomer->mono(), 'f', 5);
	
	QString avg = oligomer->avg(locale, MXP_OLIGOMER_DEC_PLACES);
	// locale.toString(oligomer->avg(), 'f', 5);

	columnData  
	  << oligomer->partialCleavage() << oligomer->name() << text
	  << mono << avg
	  << oligomer->charge() << oligomer->isModified();
	  
	currentItem = new CleaveOligomerTreeViewItem(columnData,
						      parents [0]);

	currentItem->setOligomer(oligomer);
	  
	// Append the currently created item to the last partial.
	  
	parents [0]->appendChild(currentItem);
      }
	 
    // At this point we have finished setting up the Model data.

    return true;
  }

} // namespace massXpert
