/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Local includes
#include "mzLabInputOligomerTreeViewModel.hpp"
#include "mzLabInputOligomerTreeViewDlg.hpp"
#include "application.hpp"


namespace massXpert
{

  MzLabInputOligomerTreeViewModel::MzLabInputOligomerTreeViewModel(QObject *parent) 
    : QAbstractItemModel(parent)
  {
    QList<QVariant> rootData;

    Q_ASSERT(parent);
  
    rootData << tr("m/z") << tr("z");

    mpa_oligomerList = new OligomerList("NOT_SET");

    mpa_rootItem = new MzLabInputOligomerTreeViewItem(rootData);
      
    setupModelData(mpa_rootItem);
  }


  MzLabInputOligomerTreeViewModel::~MzLabInputOligomerTreeViewModel()
  {
    delete mpa_rootItem;

    if (mpa_oligomerList)
      {
	while (mpa_oligomerList->size())
	  {
	    delete mpa_oligomerList->takeFirst();
	    qDebug() << __FILE__ << __LINE__ 
		     << "Deleting oligomer.";
	  }
	
	delete (mpa_oligomerList);
      }
    	
    //    qDeleteAll(*mpa_oligomerList);
    //    delete(mpa_oligomerList);
  }


  void
  MzLabInputOligomerTreeViewModel::setParentDlg 
 (MzLabInputOligomerTreeViewDlg *dlg)
  {
    mp_parentDlg = dlg;
  }
  
  
  MzLabInputOligomerTreeViewDlg *
  MzLabInputOligomerTreeViewModel::getParentDlg()
  {
    return mp_parentDlg;
  }
  
  
  void
  MzLabInputOligomerTreeViewModel::setMzLabWnd(MzLabWnd *wnd)
  {
    mp_mzLabWnd = wnd;
  }
  
  
  MzLabWnd *
  MzLabInputOligomerTreeViewModel::getMzLabWnd()
  {
    return mp_mzLabWnd;
  }
  
  
  const OligomerList *
  MzLabInputOligomerTreeViewModel::oligomerList() const
  {
    return mpa_oligomerList;
  }
  

  void 
  MzLabInputOligomerTreeViewModel::setTreeView(QTreeView *treeView)
  {
    Q_ASSERT(treeView);
  
    mp_treeView = treeView;
  }


  QTreeView *
  MzLabInputOligomerTreeViewModel::treeView()
  {
    return mp_treeView;
  }


  QVariant 
  MzLabInputOligomerTreeViewModel::data(const QModelIndex &index, 
					 int role) const
  {
    if (!index.isValid())
      return QVariant();
  
    if (role != Qt::DisplayRole)
      return QVariant();
  
    MzLabInputOligomerTreeViewItem *item = 
      static_cast<MzLabInputOligomerTreeViewItem *>(index.internalPointer());
  
    return item->data(index.column());
  }


  void
  MzLabInputOligomerTreeViewModel::addOligomer(Oligomer *oligomer)
  {
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    Q_ASSERT(oligomer);
    
    MassType massType = getParentDlg()->massType();
    //    qDebug() << __FILE__ << __LINE__ << "massType:" << massType;
    
    
    QModelIndexList modelIndexList;
    QModelIndex index;
  
    MzLabInputOligomerTreeViewItem *item = 0;
    MzLabInputOligomerTreeViewItem *newItem = 0;


    QItemSelectionModel *selectionModel = 
      mp_treeView->selectionModel();
    modelIndexList = selectionModel->selectedRows();
  
    if (modelIndexList.size())
      index = modelIndexList.last();
    
    QString mass = 
      locale.toString(oligomer->mass(massType), 'f', MXP_OLIGOMER_DEC_PLACES);

    QList<QVariant> columnData; 
    columnData
      << mass << oligomer->charge();
  
    // We are willing to add an oligomer item row. Depending on where we
    // are currently(oligomer or nothing) we have to make some choices
    // as to where we add this new row(either insertion or appending).

    // The index param either is valid(an item was selected and index
    // reprensents it), or is invalid, no item was selected.

    if (!index.isValid())
      {
	// No item was selected so we just append the new oligomer item
	// to the root item. We also append the oligomer to the list of
	// <MxpOligomer *>.
	mpa_oligomerList->append(oligomer);
      
	// Create an item with those columnar strings. The parent of the
	// current item is the root item.
	newItem = 
	  new MzLabInputOligomerTreeViewItem(columnData, mpa_rootItem);
      
	newItem->setOligomer(oligomer);

//       	qDebug() << __FILE__ << __LINE__
// 		  << "New oligomer's charge:" << oligomer->charge();
	
	int newRow = rowCount();
	
	mpa_rootItem->appendChild(newItem);
      
	// There is no need to produce a model index for the parent, as we
	// know we are going to insert new row in the root of the tree(this
	// tree has no branches). Hence the QModelIndex() calls below.
      
	beginInsertRows(QModelIndex(), newRow, newRow);
#if DEBUG
	int res = insertRow(newRow, QModelIndex());
	qDebug() << __FILE__ << __LINE__ << "res:" << res;
#else
	insertRow(newRow, QModelIndex());
#endif
	endInsertRows();
      }
    else
      {
	// An item was effectively selected when the instruction to add
	// an oligomer was issued.

	item = static_cast<MzLabInputOligomerTreeViewItem *>
	 (index.internalPointer());
      
	int newRow = item->row() + 1;

	mpa_oligomerList->insert(newRow, oligomer);
      
	// Create an item with those columnar strings. The parent of
	// the current item is the root item.
	newItem = 
	  new MzLabInputOligomerTreeViewItem(columnData, mpa_rootItem);
      
	newItem->setOligomer(oligomer);
      
	mpa_rootItem->insertChild(newRow, newItem);
	      
	// There is no need to produce a model index for the parent, as
	// we know we are going to insert new row in the root of the
	// tree. Hence the QModelIndex() calls below.
      
	beginInsertRows(QModelIndex(), newRow, newRow);
#if DEBUG
	int res = insertRow(newRow, QModelIndex());
	qDebug() << __FILE__ << __LINE__ << "res:" << res;
#else
	insertRow(newRow, QModelIndex());
#endif
	endInsertRows();
      }
  }
  
  
  void 
  MzLabInputOligomerTreeViewModel::removeOligomer(QModelIndex &index)
  {
    if (!index.isValid())
      return;
    
    int row = index.row();
    
    Oligomer *oligomerFromList = mpa_oligomerList->at(row);
    
    MzLabInputOligomerTreeViewItem *item = 
      static_cast<MzLabInputOligomerTreeViewItem *> 
     (index.internalPointer());
    
    Oligomer *oligomerFromItem = item->oligomer();

    // Sanity check
    if (oligomerFromList != oligomerFromItem)
      qFatal("%s@%d", __FILE__, __LINE__);

    mpa_oligomerList->removeAt(row);

    beginRemoveRows(QModelIndex(), index.row(),index. row());
    delete(mpa_rootItem->takeChild(index.row()));
    endRemoveRows();   
  }
  
  void  
  MzLabInputOligomerTreeViewModel::dataChanged(QModelIndex index1,
						QModelIndex index2)
  {
    QAbstractItemModel::dataChanged(index1, index2);
  }
  

  QVariant 
  MzLabInputOligomerTreeViewModel::headerData(int section, 
					       Qt::Orientation orientation,
					       int role) const
  {
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole)
      return mpa_rootItem->data(section);
  
    return QVariant();
  }


  Qt::ItemFlags 
  MzLabInputOligomerTreeViewModel::flags(const QModelIndex &index) const
  {
    // We want to be able to drop stuff on items.
    Qt::ItemFlags defaultFlags = QAbstractItemModel::flags(index);
    
    if (index.isValid())
      return Qt::ItemIsDropEnabled | defaultFlags;
    else
      return Qt::ItemIsDropEnabled | defaultFlags;
  }
  
  
  QModelIndex 
  MzLabInputOligomerTreeViewModel::index(int row, int column,
					  const QModelIndex &parent) const
  {
    MzLabInputOligomerTreeViewItem *parentItem = 0;
    
    if (!parent.isValid())
      parentItem = mpa_rootItem;
    else
      parentItem = 
	static_cast<MzLabInputOligomerTreeViewItem *> 
	(parent.internalPointer());
  
    MzLabInputOligomerTreeViewItem *childItem = parentItem->child(row);
    if (childItem)
      return createIndex(row, column, childItem);
    else
      return QModelIndex();
  }


  QModelIndex 
  MzLabInputOligomerTreeViewModel::parent(const QModelIndex &index) const
  {
    if (!index.isValid())
      return QModelIndex();

    MzLabInputOligomerTreeViewItem *childItem = 
      static_cast<MzLabInputOligomerTreeViewItem *>(index.internalPointer());
  
    MzLabInputOligomerTreeViewItem *parentItem = childItem->parent();

    if (parentItem == mpa_rootItem)
      return QModelIndex();
  
    return createIndex(parentItem->row(), 0, parentItem);
  }


  int 
  MzLabInputOligomerTreeViewModel::rowCount(const QModelIndex &parent) const
  {
    MzLabInputOligomerTreeViewItem *parentItem;
  
    if (!parent.isValid())
      parentItem = mpa_rootItem;
    else
      parentItem = 
	static_cast<MzLabInputOligomerTreeViewItem *> 
	(parent.internalPointer());
  
    return parentItem->childCount();
  }


  int
  MzLabInputOligomerTreeViewModel::columnCount(const QModelIndex &parent) const
  {
    if (parent.isValid())
      return static_cast<MzLabInputOligomerTreeViewItem *> 
	(parent.internalPointer())->columnCount();
    else
      return mpa_rootItem->columnCount();
  }


  bool 
  MzLabInputOligomerTreeViewModel::setupModelData(MzLabInputOligomerTreeViewItem *parent)
  {
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    MzLabInputOligomerTreeViewItem *currentItem = 0;
  
    Oligomer *oligomer = 0;
  
    Q_ASSERT(parent);
  
    // Accessing the dialog parent is erroneous , as this function is
    // called upon construction of the model, which itself is
    // constructed while still not returned from construction of the
    // parent dialog. Because, the parent dialog constructor sets its
    // m_massType = MXT_MASS_MONO, we can just use MXT_MASS_MONO
    // below.

    //     MassType massType = getParentDlg()->massType();
    //     qDebug() << __FILE__ << __LINE__ 
    // 	      << "parentDlg:" << getParentDlg() 
    // 	      << "massType:" << massType;
    
    for (int iter = 0; iter < mpa_oligomerList->size(); ++iter)
      {
	oligomer = mpa_oligomerList->at(iter);
	
	QString mass = 
	  locale.toString(oligomer->mass(MXT_MASS_MONO), 'f', 
			   MXP_OLIGOMER_DEC_PLACES);

	QList<QVariant> columnData;
	
	columnData  
	  << mass << oligomer->charge();
	
	currentItem = 
	  new MzLabInputOligomerTreeViewItem(columnData, parent);
	
	currentItem->setOligomer(oligomer);
	
	// Append the currently created item to the last partial.
	
	parent->appendChild(currentItem);
      }
    
    // At this point we have finished setting up the Model data.
    return true;
  }
  

} // namespace massXpert
