#include "mbdesktop_view.h"
#ifdef PANGO
#include <pango/pango.h>
#include <pango/pangoxft.h>
#endif

static void
mbdesktop_view_paint_items(MBDesktop *mb, MBPixbufImage *img_dest);


void
mbdesktop_view_init_bg(MBDesktop *mb)
{
  MBPixbufImage *img_tmp;
  int dw, dh, dx, dy, r, g, b;
  char *p;

  if (mb->bg_img != NULL)
    mb_pixbuf_img_free(mb->pixbuf, mb->bg_img);

  switch (mb->bg->type)
    {
    case BG_SOLID:
      mb->bg_img = mb_pixbuf_img_rgba_new(mb->pixbuf, mb->desktop_width, 
					 mb->desktop_height);
      mb_pixbuf_img_fill(mb->pixbuf, mb->bg_img, 
			 mb->bg->data.cols[0], 
			 mb->bg->data.cols[1],
			 mb->bg->data.cols[2], 
			 0);
      /* 
	 TODO:
	  o The paint call, should check for the SOLID case and just
	    XFill a rectangle strait to the pixmap rather than the blit.
	    This could be faster / save memory.
      */
      break;
    case BG_TILED_PXM:
      mb->bg_img = mb_pixbuf_img_rgb_new(mb->pixbuf, mb->desktop_width, 
					 mb->desktop_height);

      if ((img_tmp = mb_pixbuf_img_new_from_file(mb->pixbuf, 
						 mb->bg->data.filename)) 
	  == NULL)
	{
	  fprintf(stderr,"Failed to load background : %s", 
		  mb->bg->data.filename);
	  mbdesktop_bg_parse_spec(mb, "col-solid:red");
	  mbdesktop_view_init_bg(mb);
	  return;
	}

      for (dy=0; dy < mb->desktop_height; dy += img_tmp->height)
	for (dx=0; dx < mb->desktop_width; dx += img_tmp->width)
	  {
	    if ( (dx + img_tmp->width) > mb->desktop_width )
	      dw = img_tmp->width - ((dx + img_tmp->width)-mb->desktop_width);
	    else
	      dw = img_tmp->width;

	    if ( (dy + img_tmp->height) > mb->desktop_height )
	      dh = img_tmp->height-((dy + img_tmp->height)-mb->desktop_height);
	    else
	      dh = img_tmp->height;
	    mb_pixbuf_img_copy(mb->pixbuf, mb->bg_img, img_tmp,
			       0, 0, dw, dh, dx, dy);
	  }
      mb_pixbuf_img_free(mb->pixbuf, img_tmp);
      break;
    case BG_STRETCHED_PXM:
      if ((img_tmp = mb_pixbuf_img_new_from_file(mb->pixbuf, 
						 mb->bg->data.filename)) 
	  == NULL)
	{
	  fprintf(stderr,"Failed to load background : %s", mb->bg->data.filename);
	  mbdesktop_bg_parse_spec(mb, "col-solid:red");
	  mbdesktop_view_init_bg(mb);
	  return;
	}
      mb->bg_img = mb_pixbuf_img_scale(mb->pixbuf, img_tmp, mb->desktop_width, 
				       mb->desktop_height);
      mb_pixbuf_img_free(mb->pixbuf, img_tmp);
      break;
    case BG_CENTERED_PXM:
      if ((img_tmp = mb_pixbuf_img_new_from_file(mb->pixbuf, 
						 mb->bg->data.filename)) 
	  == NULL)
	{
	  fprintf(stderr,"Failed to load background : %s", mb->bg->data.filename);
	  mbdesktop_bg_parse_spec(mb, "col-solid:red");
	  mbdesktop_view_init_bg(mb);
	  return;
	}
      dx = (mb->desktop_width - img_tmp->width) / 2;
      if (dx < 0) dx = 0;
      dy = (mb->desktop_height - img_tmp->height) / 2;
      if (dy < 0) dy = 0;
      mb->bg_img = mb_pixbuf_img_rgb_new(mb->pixbuf, mb->desktop_width, 
				     mb->desktop_height);
      mb_pixbuf_img_copy(mb->pixbuf, mb->bg_img, img_tmp,
			 0, 0,
			 (img_tmp->width > mb->desktop_width) ?
			 mb->desktop_width : img_tmp->width ,
			 (img_tmp->height > mb->desktop_height) ?
			 mb->desktop_height : img_tmp->height ,
			 dx, dy);
      mb_pixbuf_img_free(mb->pixbuf, img_tmp);
      break;
    case BG_GRADIENT_HORIZ:
      mb->bg_img = mb_pixbuf_img_rgb_new(mb->pixbuf, mb->desktop_width, 
					 mb->desktop_height);
      p = mb->bg_img->rgba;

      dw = mb->desktop_width;
      dh = mb->desktop_height;
      
      for(dx=0; dx<dw; dx++)
	{
	  r = mb->bg->data.gcols[0] + (( dx * (mb->bg->data.gcols[1] - mb->bg->data.gcols[0])) / dw); 
	  g = mb->bg->data.gcols[2] + (( dx * (mb->bg->data.gcols[3] - mb->bg->data.gcols[2])) / dw); 
	  b = mb->bg->data.gcols[4] + (( dx * (mb->bg->data.gcols[5] - mb->bg->data.gcols[4])) / dw); 

	  for(dy=0; dy<dh; dy++)
	    {
	      *p++ = r;
	      *p++ = g;
	      *p++ = b;
	    }
	}
      break;
    case BG_GRADIENT_VERT:
      mb->bg_img = mb_pixbuf_img_rgb_new(mb->pixbuf, mb->desktop_width, 
					 mb->desktop_height);
      p = mb->bg_img->rgba;

      dw = mb->desktop_width;
      dh = mb->desktop_height;
      
      for(dy=0; dy<dh; dy++)
	{
	  r = mb->bg->data.gcols[0] + (( dy * (mb->bg->data.gcols[1] - mb->bg->data.gcols[0])) / dh); 
	  g = mb->bg->data.gcols[2] + (( dy * (mb->bg->data.gcols[3] - mb->bg->data.gcols[2])) / dh); 
	  b = mb->bg->data.gcols[4] + (( dy * (mb->bg->data.gcols[5] - mb->bg->data.gcols[4])) / dh); 

	  for(dx=0; dx<dw; dx++)
	    {
	      *p++ = r;
	      *p++ = g;
	      *p++ = b;
	    }
	}
      break;
    }

  /* Now set the background */
  mbdesktop_view_set_root_pixmap(mb, mb->bg_img);

}

void
mbdesktop_view_set_root_pixmap(MBDesktop *mb, MBPixbufImage *img)
{
  Atom atom_root_pixmap_id = XInternAtom(mb->dpy, "_XROOTPMAP_ID", False); 
  
  if (mb->root_pxm != None) XFreePixmap(mb->dpy, mb->root_pxm);
  
  mb->root_pxm = XCreatePixmap(mb->dpy, mb->root, 
			       img->width, img->height,
			       mb->pixbuf->depth ); 

  mb_pixbuf_img_render_to_drawable(mb->pixbuf, img, (Drawable)mb->root_pxm,
				   0, 0);

  XChangeProperty(mb->dpy, mb->root, atom_root_pixmap_id, XA_PIXMAP, 
		  32, PropModeReplace, (unsigned char *) &mb->root_pxm, 1);
}


void  /* called only when desktop/workarea changes size or bgimage changes */
mbdesktop_view_configure(MBDesktop *mb)
{
  /* Assume mb->desktop_width, etc is up to date */

  int x,y,w,h;

  /* This is a bit crap - probably already called */
  if (mbdesktop_get_workarea(mb, &x, &y, &w, &h))
    {
      mb->workarea_x = x; 
      mb->workarea_y = y ;
      mb->workarea_width = w;
      mb->workarea_height = h;
    }

  mbdesktop_view_init_bg(mb); 	   /* reset the background */
  mbdesktop_view_paint(mb, False); /* repaint */

}


void  /* TODO: implement multiple views */
mbdesktop_set_view(MBDesktop *mb, int view)
{
  ;
}

void 
mbdesktop_view_paint(MBDesktop *mb, Bool use_cache)
{
  MBPixbufImage *img_dest;
  char *folder_title = "Home";

  if (use_cache && mb->backing_cache != None)
    {
      if (mb->had_kbd_input && mb->kbd_focus_item)
	{
	  if (mb->have_focus)
	    mbdesktop_view_item_highlight (mb, mb->kbd_focus_item, 
					   HIGHLIGHT_OUTLINE); 
	}
      return;
    }


  if (mb->backing_cache != None)
    XFreePixmap(mb->dpy, mb->backing_cache);

  mb->backing_cache = XCreatePixmap(mb->dpy, mb->root, 
				    mb->desktop_width, mb->desktop_height,
				    mb->pixbuf->depth ); 

#ifdef USE_XFT  
  if (mb->xftdraw != NULL)
    XftDrawDestroy (mb->xftdraw);

  mb->xftdraw = XftDrawCreate(mb->dpy, (Drawable) mb->backing_cache, 
			      mb->pixbuf->vis,
			      mb->pixbuf->root_cmap
			      );
#endif

  img_dest = mb_pixbuf_img_clone(mb->pixbuf, mb->bg_img);

  if ( mbdesktop_current_folder_view (mb) == VIEW_ICONS )
    mbdesktop_view_paint_items(mb, img_dest);
  else
    mbdesktop_view_paint_list(mb, img_dest);

  if (mb->current_head_item->item_parent)
    folder_title = mb->current_head_item->item_parent->name;
  else 
    folder_title = mb->top_head_item->name;

  XSetLineAttributes(mb->dpy, mb->gc, 1, LineSolid, CapRound, JoinRound);

#ifdef USE_XFT  

  if (mb->use_text_outline)
    XftDrawStringUtf8(mb->xftdraw, 
		      &mb->xftbgcol, 
		      mb->titlefont,
		      mb->workarea_x + ((48-32)/2) + 1,
		      mb->workarea_y + mb->titlefont->ascent + 1,
		      (unsigned char *)folder_title,
		      strlen(folder_title)
		      );

  XftDrawStringUtf8(mb->xftdraw, 
		    &mb->xftcol, 
		    mb->titlefont,
		    mb->workarea_x + ((48-32)/2),
 		    mb->workarea_y + mb->titlefont->ascent,
		    (unsigned char *)folder_title,
		    strlen(folder_title)
		    );

  XDrawLine(mb->dpy, mb->backing_cache, mb->gc, 
	    mb->workarea_x + ((48-32)/2),
	    mb->workarea_y + mb->titlefont->ascent 
	    + mb->titlefont->descent + 2,
	    mb->workarea_x + mb->workarea_width - ((48-32)/2),
	    mb->workarea_y + mb->titlefont->ascent 
	    + mb->titlefont->descent + 2);

#else

   XSetFont(mb->dpy, mb->gc, mb->titlefont->fid);
   XDrawString(mb->dpy, mb->backing_cache, mb->gc,
	       mb->workarea_x + ((48-32)/2),
	       mb->workarea_y + mb->titlefont->ascent,
	       (unsigned char *)folder_title,
	       strlen(folder_title));

   XDrawLine(mb->dpy, mb->backing_cache, mb->gc, 
	     mb->workarea_x + ((48-32)/2),
	     mb->workarea_y + mb->titlefont->ascent 
	     + mb->titlefont->descent + 2,
	     mb->workarea_x + mb->workarea_width - ((48-32)/2),
	     mb->workarea_y + mb->titlefont->ascent 
	     + mb->titlefont->descent + 2);

#endif


  XSetLineAttributes(mb->dpy, mb->gc, 1, LineOnOffDash, CapRound, JoinRound);

  XSetWindowBackgroundPixmap(mb->dpy, mb->win_top_level, mb->backing_cache);
  XClearWindow(mb->dpy, mb->win_top_level);

  if (mb->kbd_focus_item && mb->had_kbd_input)
    {
      if (mb->have_focus)
	mbdesktop_view_item_highlight (mb, mb->kbd_focus_item, 
				       HIGHLIGHT_OUTLINE_CLEAR); 
    }

  if (img_dest)
    mb_pixbuf_img_free(mb->pixbuf, img_dest);  
}

void
mbdesktop_view_paint_list(MBDesktop *mb, MBPixbufImage *dest_img)
{
  /* 
     XXX TODO
       - paint correct icon size

  */

  MBDesktopItem *item;
  MBPixbufImage *icon_img_small;
  int cur_y, cur_x, limit_y;

  if (mb->scroll_offset_item  == mb->current_head_item)
    mb->scroll_active = False;
  else
    mb->scroll_active = True;


  cur_x = mb->workarea_x + ((48-32)/2);
  cur_y = mb->workarea_y + mb->title_offset;
  limit_y = mb->workarea_y + mb->workarea_height;

  for(item = mb->scroll_offset_item; 
      item != NULL; 
      item = item->item_next_sibling)
    {
      if (item->icon)
	{
	  if ( (cur_y + mb->icon_size ) > limit_y) /* Off display ? */
	    {
	      mb->scroll_active = True;
	      break;
	    }

	  icon_img_small = mb_pixbuf_img_scale(mb->pixbuf, 
					       item->icon,
					       mb->icon_size, mb->icon_size);

	  mb_pixbuf_img_composite(mb->pixbuf, dest_img, 
				  icon_img_small, 
				  cur_x, 
				  cur_y );

	  mb_pixbuf_img_free(mb->pixbuf, icon_img_small);
	  item->x = cur_x;
	  item->y = cur_y;
	  item->width  = mb->workarea_width - ((48-32));
	  item->height = mb->icon_size;

	  cur_y += mb->icon_size;

	}
    }

  if (mb->scroll_active)
    {
      mb_pixbuf_img_composite(mb->pixbuf, dest_img, mb->img_scroll_up,
			      mb->workarea_x + mb->workarea_width-24, 2);
    
      mb_pixbuf_img_composite(mb->pixbuf, dest_img,
			      mb->img_scroll_down, 
			      mb->workarea_x + mb->workarea_width-40, 2);

    }


  mb_pixbuf_img_render_to_drawable(mb->pixbuf, dest_img, 
				   (Drawable)mb->backing_cache, 
				   0, 0);

  mb->last_visible_item = item;


  for(item = mb->scroll_offset_item; 
      item != NULL; 
      item = item->item_next_sibling)
    {
      int foffsety = 0;
      char *name = NULL;
#ifdef USE_XFT        
      XGlyphInfo extents;
#endif
      if (item == mb->last_visible_item)
	break;

      name = strdup(item->name);
#ifdef USE_XFT        
      XftTextExtentsUtf8(mb->dpy, mb->font, (unsigned char *)name,
		      strlen(name), &extents);

      foffsety = (mb->icon_size -(mb->font->ascent+mb->font->descent))/2;
      if (foffsety < 0) foffsety = 0;

      if (mb->use_text_outline)
	XftDrawStringUtf8(mb->xftdraw, 
			  &mb->xftcol, 
			  mb->font,
			  item->x + mb->icon_size + 1,
			  item->y + foffsety + mb->font->ascent + 1,
			  (unsigned char *)name, 
			  strlen(name)
			  );
		  

      XftDrawStringUtf8(mb->xftdraw, 
		     &mb->xftcol, 
		     mb->font,
		     item->x + mb->icon_size,
		     item->y + foffsety + mb->font->ascent,
		     (unsigned char *)name, 
		     strlen(name)
		     );
#endif
      free(name);
    }
}


static void
mbdesktop_view_paint_items(MBDesktop *mb, MBPixbufImage *img_dest)
{
  MBDesktopItem *item;
  MBPixbufImage *icon_img_small;

  int cur_x = 0, cur_y = 0, limit_x, limit_y;
  int item_horiz_border = (mb->item_width-(mb->icon_size))/2;

  if (mb->scroll_offset_item  == mb->current_head_item)
    mb->scroll_active = False;
  else
    mb->scroll_active = True;

  cur_x = mb->workarea_x;
  cur_y = mb->workarea_y + mb->title_offset;

  limit_x = mb->workarea_x + mb->workarea_width;
  limit_y = mb->workarea_y + mb->workarea_height + mb->title_offset;

  for(item = mb->scroll_offset_item; 
      item != NULL; 
      item = item->item_next_sibling)
    {
      if (item->type == ITEM_TYPE_MODULE_WINDOW)
	{
	  /* 
	     Its a window. 
	       - Move it ?
	       - Is it offscreen ?
	       - resize its width if changed ?
	       - send it an expose ?
	       - cur_y + window height. 
	  */
	}
      else
	{
	  if ((cur_x + mb->item_width) > limit_x) /* 'newline' */
	    {
	      cur_x = mb->workarea_x;
	      cur_y += mb->item_height;

	      if ( (cur_y+mb->item_height) > limit_y) /* Off display ? */
		{
		  /* 
		     set a flag to turn on scrolling
		  */
		  mb->scroll_active = True;
		  break;
		}
	    }

	  item->x      = cur_x;
	  item->y      = cur_y;
	  item->width  = mb->item_width;
	  item->height = mb->item_height;
	  
	  if (item->icon)
	    {
	      if (mb_pixbuf_img_get_width(item->icon) != mb->icon_size
		  || mb_pixbuf_img_get_height(item->icon) != mb->icon_size)
		{
		  icon_img_small = mb_pixbuf_img_scale(mb->pixbuf, item->icon,
						       mb->icon_size, 
						       mb->icon_size);
		  
		  mb_pixbuf_img_composite(mb->pixbuf, img_dest, 
					  icon_img_small, 
					  item->x + item_horiz_border, 
					  item->y );
		  
		  mb_pixbuf_img_free(mb->pixbuf, icon_img_small);
		  
		} else mb_pixbuf_img_composite(mb->pixbuf, img_dest, 
					       item->icon, 
					       item->x + item_horiz_border, 
					   item->y );
	    }
	  
	  cur_x += mb->item_width;
	}
    }

  if (mb->scroll_active)
    {
      mb_pixbuf_img_composite(mb->pixbuf, img_dest, mb->img_scroll_up,
			      mb->workarea_x + mb->workarea_width-24, 2);
    
      mb_pixbuf_img_composite(mb->pixbuf, img_dest, mb->img_scroll_down, 
			      mb->workarea_x + mb->workarea_width-40, 2);

    }


  mb_pixbuf_img_render_to_drawable(mb->pixbuf, img_dest, 
				   (Drawable)mb->backing_cache, 
				   0, 0);

  mb->last_visible_item = item;

  /* 2nd pass render text */

  for(item = mb->scroll_offset_item; 
      (item != NULL && item != mb->last_visible_item); 
      item = item->item_next_sibling)
    {
      int name_txt_width = 0;
      char *name = strdup(item->name);
#ifdef USE_XFT        
      XGlyphInfo extents;
      XftTextExtentsUtf8(mb->dpy, mb->font, (unsigned char *)name,
		      strlen(name), &extents);

      name_txt_width = extents.width;
#else
      XSetFont(mb->dpy, mb->gc, mb->font->fid);
      name_txt_width = XTextWidth(mb->font, name, strlen(name));
#endif

      if (item->type == ITEM_TYPE_MODULE_WINDOW)
	continue;

      if (name_txt_width > mb->item_width) /* too wide */
	{
	  char *p = name;
	  char *q = p;
	  int v_offset = 0;
	  int i = 0;

	  while (*(p++) != '\0' && i < 2)
	    if (isspace(*p) || *(p+1) == '\0')
	      {
		if (*(p+1) != '\0' ) *p = '\0';

#ifdef USE_XFT
		XftTextExtentsUtf8(mb->dpy, mb->font, (unsigned char *)q,
				   strlen(q), &extents);

		if (mb->use_text_outline)
		  XftDrawStringUtf8(mb->xftdraw, 
				    &mb->xftbgcol, 
				    mb->font,
				    item->x + (( mb->item_width - extents.width)/2) + 1, 
				    item->y + mb->icon_size + v_offset + 1 
				    + mb->font->ascent,
				    (unsigned char *)q, 
				    strlen(q)
				    );

		XftDrawStringUtf8(mb->xftdraw, 
			       &mb->xftcol, 
			       mb->font,
			       item->x + (( mb->item_width - extents.width)/2), 
			       item->y + mb->icon_size + v_offset 
			       + mb->font->ascent,
			       (unsigned char *)q, 
			       strlen(q)
			       );
		v_offset += mb->font->ascent + mb->font->descent;
#else
		name_txt_width = XTextWidth(mb->font, q, strlen(q));
		XDrawString(mb->dpy, mb->backing_cache, mb->gc,
			       item->x + (( mb->item_width - name_txt_width)/2), 
			       item->y + mb->icon_size + v_offset 
			       + mb->font->ascent,
			       (unsigned char *)q, 
			       strlen(q)
			    );
		v_offset += mb->font->ascent + mb->font->descent;
#endif
		q= p+1;
		p = q;
		i++;
	      }
	}
      else
	{
#ifdef USE_XFT

	  if (mb->use_text_outline)
	    XftDrawStringUtf8(mb->xftdraw, 
			      &mb->xftbgcol, 
			      mb->font,
			      item->x + (( mb->item_width - extents.width)/2) + 1, 
			      item->y + mb->icon_size + mb->font->ascent + 1,
			      (unsigned char *)name, 
			      strlen(name)
			      );

	  XftDrawStringUtf8(mb->xftdraw, 
			    &mb->xftcol, 
			    mb->font,
			    item->x + (( mb->item_width - extents.width)/2), 
			    item->y + mb->icon_size + mb->font->ascent,
			    (unsigned char *)name, 
			    strlen(name)
			    );
#else
	  XDrawString(mb->dpy, mb->backing_cache, mb->gc,
		      item->x + (( mb->item_width - name_txt_width)/2), 
		      item->y + mb->icon_size + mb->font->ascent,
		      (unsigned char *)name, strlen(name));


#endif
	}


      free(name);
    }
}

void
mbdesktop_view_item_highlight (MBDesktop     *mb, 
			       MBDesktopItem *item,
			       int            highlight_style)
{
  int cur_view = 0;
  int x = 0, y = 0, w = 0, h = 0;

  cur_view = mbdesktop_current_folder_view(mb);

  switch (cur_view)
    {
    case VIEW_ICONS:
      x = item->x;
      y = item->y      + mb->icon_size;
      w = item->width;
      h = item->height - mb->icon_size;
      break;
    case VIEW_LIST:
      x = item->x;
      y = item->y;
      w = item->width;
      h = item->height;
      break;
    }
  
  switch (highlight_style)
    {
    case HIGHLIGHT_OUTLINE:
    case HIGHLIGHT_OUTLINE_CLEAR:
      XDrawRectangle(mb->dpy, mb->win_top_level, mb->invert_gc, x, y, w, h);
      break;
    case HIGHLIGHT_FILL:
    case HIGHLIGHT_FILL_CLEAR:
      XFillRectangle(mb->dpy, mb->win_top_level, mb->invert_gc, x, y, w, h);
      break;
    }

  XFlush(mb->dpy);
}



/* XXX TOGO */

void 
mbdesktop_view_paint_icons(MBDesktop *mb, MBPixbufImage *img_dest)
{

  MBDesktopItem *item;
  MBPixbufImage *icon_img_small;
  int cur_x = 0, cur_y = 0, limit_x, limit_y;

  int item_horiz_border = (mb->item_width-(mb->icon_size))/2;

#define TITLE_OFFSET_Y 24

  cur_x = mb->workarea_x;
  cur_y = mb->workarea_y + mb->title_offset;

  limit_x = mb->workarea_x + mb->workarea_width;
  limit_y = mb->workarea_y + mb->workarea_height;

  for(item = mb->scroll_offset_item; 
      item != NULL; 
      item = item->item_next_sibling)
    {
      if ((cur_x + mb->item_width) > limit_x)
	{
	  cur_x = mb->workarea_x;
	  cur_y += mb->item_height;
	  if ( (cur_y+mb->item_height) > limit_y)
	    break;
	}
      item->x = cur_x;
      item->y = cur_y;

      if (item->icon)
	{
	  /* TODO: check icon is not already 32x32 */
	  icon_img_small = mb_pixbuf_img_scale(mb->pixbuf, 
						    item->icon,
						    mb->icon_size, mb->icon_size);

	  mb_pixbuf_img_composite(mb->pixbuf, img_dest, 
				  icon_img_small, 
				  item->x+item_horiz_border, 
				  item->y );

	  mb_pixbuf_img_free(mb->pixbuf, icon_img_small);
	}
 
      cur_x += mb->item_width;
    }

  mb_pixbuf_img_render_to_drawable(mb->pixbuf, img_dest, 
				   (Drawable)mb->backing_cache, 
				   0, 0);

  mb->last_visible_item = item;

  for(item = mb->scroll_offset_item; 
      (item != NULL && item != mb->last_visible_item); 
      item = item->item_next_sibling)
    {
      int name_txt_width = 0;
      char *name = strdup(item->name);
#ifdef USE_XFT        
      XGlyphInfo extents;
      XftTextExtentsUtf8(mb->dpy, mb->font, (unsigned char *)name,
		      strlen(name), &extents);

      name_txt_width = extents.width;
#else
      XSetFont(mb->dpy, mb->gc, mb->font->fid);
      name_txt_width = XTextWidth(mb->font, name, strlen(name));
#endif

#ifdef PANGO 			/* TOGO ??? */
      {
	PangoContext *pc = pango_xft_get_context (mb->dpy, 0);	/* XXX screen number */
	PangoLayout *pl = pango_layout_new (pc);
	PangoFontMap *fontmap = pango_xft_get_font_map (mb->dpy, 0);
	PangoFontDescription *fontdes = pango_font_description_new ();
	PangoFont *pangofont;
	PangoFontMetrics *pangometrics;

	GSList *lines;

	/* XXX pango_context_set_language 

	to make that work right, you will need to keep track of what language
	the string you are displaying is actually in (ie whether you found a
	localised one or had to fall back to the English version), and call
	pango_layout_set_language appropriately.

	*/
	pango_font_description_set_family (fontdes, "Verdana");
	pango_font_description_set_size (fontdes, 6 * PANGO_SCALE);

	pangofont = pango_font_map_load_font (fontmap, pc, fontdes);

	pangometrics = pango_font_get_metrics(pangofont, NULL);

	pango_layout_set_text (pl, name, strlen (name));
	pango_layout_set_font_description (pl, fontdes);
	pango_layout_set_width (pl, mb->item_width * PANGO_SCALE);
	pango_layout_set_justify (pl, TRUE);
	pango_layout_set_alignment (pl, PANGO_ALIGN_CENTER);

	for (lines = pango_layout_get_lines (pl); lines; lines = lines->next)
	  {
	    PangoLayoutLine *this = lines->data;
	    GSList *run;

	    for (run = this->runs; run; run = run->next)
	      {
		PangoLayoutRun *this = run->data;
		PangoRectangle rect;

		pango_layout_index_to_pos (pl, this->item->offset, &rect);
	    
		pango_xft_render (mb->xftdraw, &mb->xftcol, pangofont, this->glyphs,
				  item->x + rect.x / PANGO_SCALE,
				  item->y + mb->icon_size + rect.y + pango_font_metrics_get_ascent(pangometrics) / PANGO_SCALE);
	      }
	  }
	
	g_object_unref (pl);
	g_object_unref (pc);
      }
#endif /* XXX was else */
      if (name_txt_width > mb->item_width) /* too wide */
	{
	  char *p = name;
	  char *q = p;
	  int v_offset = 0;
	  int i = 0;

	  while (*(p++) != '\0' && i < 2)
	    if (isspace(*p) || *(p+1) == '\0')
	      {
		if (*(p+1) != '\0' ) *p = '\0';

#ifdef USE_XFT
		XftTextExtentsUtf8(mb->dpy, mb->font, (unsigned char *)q,
				   strlen(q), &extents);

		XftDrawStringUtf8(mb->xftdraw, 
			       &mb->xftcol, 
			       mb->font,
			       item->x + (( mb->item_width - extents.width)/2), 
			       item->y + mb->icon_size + v_offset 
			       + mb->font->ascent,
			       (unsigned char *)q, 
			       strlen(q)
			       );
		v_offset += mb->font->ascent + mb->font->descent;
#else
		name_txt_width = XTextWidth(mb->font, q, strlen(q));
		XDrawString(mb->dpy, mb->backing_cache, mb->gc,
			       item->x + (( mb->item_width - name_txt_width)/2), 
			       item->y + mb->icon_size + v_offset 
			       + mb->font->ascent,
			       (unsigned char *)q, 
			       strlen(q)
			    );
		v_offset += mb->font->ascent + mb->font->descent;
#endif
		q= p+1;
		p = q;
		i++;
	      }
	}
      else
	{
#ifdef USE_XFT
	  XftDrawStringUtf8(mb->xftdraw, 
			    &mb->xftcol, 
			    mb->font,
			    item->x + (( mb->item_width - extents.width)/2), 
			    item->y + mb->icon_size + mb->font->ascent,
			    (unsigned char *)name, 
			    strlen(name)
			    );
#else
	  XDrawString(mb->dpy, mb->backing_cache, mb->gc,
		      item->x + (( mb->item_width - name_txt_width)/2), 
		      item->y + mb->icon_size + mb->font->ascent,
		      (unsigned char *)name, strlen(name));


#endif
	}


      free(name);
    }
 
}
