(* Copyright (C) 2002, HELM Team.
 * 
 * This file is part of HELM, an Hypertextual, Electronic
 * Library of Mathematics, developed at the Computer Science
 * Department, University of Bologna, Italy.
 * 
 * HELM is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * HELM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with HELM; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA  02111-1307, USA.
 * 
 * For details, see the HELM World-Wide-Web page,
 * http://cs.unibo.it/helm/.
 *)

open AutoTypes;;
open AutoCache;;

let debug = false;;
let debug_print s = 
  if debug then prerr_endline (Lazy.force s);;

let is_propositional context sort = 
  match CicReduction.whd context sort with
  | Cic.Sort Cic.Prop 
  | Cic.Sort (Cic.CProp _) -> true
  | _-> false
;;


type auto_params = Cic.term list * (string * string) list 

let elems = ref [] ;;

(* closing a term w.r.t. its metavariables
   very naif version: it does not take dependencies properly into account *)

let naif_closure ?(prefix_name="xxx_") t metasenv context =
  let metasenv = ProofEngineHelpers.sort_metasenv metasenv in
  let n = List.length metasenv in
  let what = List.map (fun (i,cc,ty) -> Cic.Meta(i,[])) metasenv in
  let _,with_what =
    List.fold_left
      (fun (i,acc) (_,cc,ty) -> (i-1,Cic.Rel i::acc)) 
      (n,[]) metasenv 
  in
  let t = CicSubstitution.lift n t in
  let body =
    ProofEngineReduction.replace_lifting 
      ~equality:(fun c t1 t2 ->
         match t1,t2 with
         | Cic.Meta(i,_),Cic.Meta(j,_) -> i = j
         | _ -> false) 
      ~context ~what ~with_what ~where:t 
  in
  let _, t =
    List.fold_left
      (fun (n,t) (_,cc,ty) -> 
        n-1, Cic.Lambda(Cic.Name (prefix_name^string_of_int n),
               CicSubstitution.lift n ty,t))
      (n-1,body) metasenv 
  in
  t
;;

let lambda_close ?prefix_name t menv ctx =
  let t = naif_closure ?prefix_name t menv ctx in
    List.fold_left
      (fun (t,i) -> function 
        | None -> CicSubstitution.subst (Cic.Implicit None) t,i (* delift *)
        | Some (name, Cic.Decl ty) -> Cic.Lambda (name, ty, t),i+1
        | Some (name, Cic.Def (bo, ty)) -> Cic.LetIn (name, bo, ty, t),i+1)
      (t,List.length menv) ctx
;;
  
(* functions for retrieving theorems *)

exception FillingFailure of AutoCache.cache * int

let rec unfold context = function
  | Cic.Prod(name,s,t) -> 
      let t' = unfold ((Some (name,Cic.Decl s))::context) t in
        Cic.Prod(name,s,t')        
  | t -> ProofEngineReduction.unfold context t

let find_library_theorems dbd proof goal = 
  let univ = MetadataQuery.universe_of_goal ~dbd false proof goal in
  let terms = List.map CicUtil.term_of_uri univ in
  List.map 
    (fun t -> 
       (t,fst(CicTypeChecker.type_of_aux' [] [] t CicUniv.oblivion_ugraph))) 
    terms

let find_context_theorems context metasenv =
  let l,_ =
    List.fold_left
      (fun (res,i) ctxentry ->
         match ctxentry with
           | Some (_,Cic.Decl t) -> 
               (Cic.Rel i, CicSubstitution.lift i t)::res,i+1
           | Some (_,Cic.Def (_,t)) ->
               (Cic.Rel i, CicSubstitution.lift i t)::res,i+1
           | None -> res,i+1)
      ([],1) context
  in l

let rec is_an_equality = function
  | Cic.Appl [Cic.MutInd (uri, _, _); _; _; _] 
    when (LibraryObjects.is_eq_URI uri) -> true
  | Cic.Prod (_, _, t) -> is_an_equality t
  | _ -> false
;;

let partition_equalities =
  List.partition (fun (_,ty) -> is_an_equality ty)


let default_auto maxm _ _ cache _ _ _ _ = [],cache,maxm ;; 


let is_unit_equation context metasenv oldnewmeta term = 
  let head, metasenv, args, newmeta =
    TermUtil.saturate_term oldnewmeta metasenv context term 0
  in
  let propositional_args = 
    HExtlib.filter_map
      (function 
      | Cic.Meta(i,_) -> 
          let _,_,mt = CicUtil.lookup_meta i metasenv in
          let sort,u = 
            CicTypeChecker.type_of_aux' metasenv context mt 
              CicUniv.oblivion_ugraph
          in
          if is_propositional context sort then Some i else None 
      | _ -> assert false)
    args
  in
    if propositional_args = [] then 
      let newmetas = 
        List.filter (fun (i,_,_) -> i >= oldnewmeta) metasenv 
      in
        Some (args,metasenv,newmetas,head,newmeta)
    else None
;;

let get_candidates universe cache t =
  let candidates= 
    (Universe.get_candidates universe t)@(AutoCache.get_candidates cache t)
  in 
  let debug_msg =
    (lazy ("candidates for " ^ (CicPp.ppterm t) ^ " = " ^ 
             (String.concat "\n" (List.map CicPp.ppterm candidates)))) in
  debug_print debug_msg;
  candidates
;;

let only signature context metasenv t =
  try
    let ty,_ = 
      CicTypeChecker.type_of_aux' metasenv context t CicUniv.oblivion_ugraph 
    in
    let consts = MetadataConstraints.constants_of ty in
    let b = MetadataConstraints.UriManagerSet.subset consts signature in
    if b then b 
    else
      let ty' = unfold context ty in
      let consts' = MetadataConstraints.constants_of ty' in
      MetadataConstraints.UriManagerSet.subset consts' signature 
  with 
  | CicTypeChecker.TypeCheckerFailure _ -> assert false
  | ProofEngineTypes.Fail _ -> false (* unfold may fail *)
;;

let not_default_eq_term t =
  try
    let uri = CicUtil.uri_of_term t in
      not (LibraryObjects.in_eq_URIs uri)
  with Invalid_argument _ -> true

let retrieve_equations dont_filter signature universe cache context metasenv =
  match LibraryObjects.eq_URI() with
    | None -> [] 
    | Some eq_uri -> 
        let eq_uri = UriManager.strip_xpointer eq_uri in
        let fake= Cic.Meta(-1,[]) in
        let fake_eq = Cic.Appl [Cic.MutInd (eq_uri,0, []);fake;fake;fake] in
        let candidates = get_candidates universe cache fake_eq in
        if dont_filter then candidates
        else 
          let candidates = List.filter not_default_eq_term candidates in
          List.filter (only signature context metasenv) candidates 

let build_equality bag head args proof newmetas maxmeta = 
  match head with
  | Cic.Appl [Cic.MutInd (uri, _, _); ty; t1; t2] ->
      let p =
        if args = [] then proof else Cic.Appl (proof::args)
      in 
      let o = !Utils.compare_terms t1 t2 in
      let stat = (ty,t1,t2,o) in
      (* let w = compute_equality_weight stat in *)
      let w = 0 in 
      let proof = Equality.Exact p in
      let e = Equality.mk_equality bag (w, proof, stat, newmetas) in
      (* to clean the local context of metas *)
      Equality.fix_metas bag maxmeta e
  | _ -> assert false
;;

let partition_unit_equalities context metasenv newmeta bag equations =
  List.fold_left
    (fun (units,other,maxmeta)(t,ty) ->
       if not (CicUtil.is_meta_closed t && CicUtil.is_meta_closed ty) then
         let _ = 
           HLog.warn 
           ("Skipping " ^ CicMetaSubst.ppterm_in_context ~metasenv [] t context
             ^ " since it is not meta closed")
         in
         units,(t,ty)::other,maxmeta
       else
       match is_unit_equation context metasenv maxmeta ty with
         | Some (args,metasenv,newmetas,head,newmeta') ->
             let maxmeta,equality =
               build_equality bag head args t newmetas newmeta' in
             equality::units,other,maxmeta
         | None -> 
             units,(t,ty)::other,maxmeta)
    ([],[],newmeta) equations

let empty_tables = 
  (Saturation.make_active [], 
   Saturation.make_passive [],
   Equality.mk_equality_bag)

let init_cache_and_tables 
  ?dbd use_library paramod use_context dont_filter universe (proof, goal) 
=
  (* the local cache in initially empty  *)
  let cache = AutoCache.cache_empty in
  let _, metasenv, _subst,_, _, _ = proof in
  let signature = MetadataQuery.signature_of metasenv goal in
  let newmeta = CicMkImplicit.new_meta metasenv [] in
  let _,context,_ = CicUtil.lookup_meta goal metasenv in
  let ct = if use_context then find_context_theorems context metasenv else [] in
  debug_print 
    (lazy ("ho trovato nel contesto " ^ (string_of_int (List.length ct))));
  let lt = 
    match use_library, dbd with
    | true, Some dbd -> find_library_theorems dbd metasenv goal 
    | _ -> []
  in
  debug_print 
    (lazy ("ho trovato nella libreria " ^ (string_of_int (List.length lt))));
  let cache = cache_add_list cache context (ct@lt) in  
  let equations = 
    retrieve_equations dont_filter signature universe cache context metasenv 
  in
  debug_print 
    (lazy ("ho trovato equazioni n. "^(string_of_int (List.length equations))));
  let eqs_and_types =
    HExtlib.filter_map 
      (fun t -> 
         let ty,_ =
           CicTypeChecker.type_of_aux' 
             metasenv context t CicUniv.oblivion_ugraph
         in
         (* retrieve_equations could also return flexible terms *)
         if is_an_equality ty then Some(t,ty) 
         else
           try
             let ty' = unfold context ty in
             if is_an_equality ty' then Some(t,ty') else None
           with ProofEngineTypes.Fail _ -> None) 
      equations
  in
  let bag = Equality.mk_equality_bag () in
  let units, other_equalities, newmeta = 
    partition_unit_equalities context metasenv newmeta bag eqs_and_types 
  in
  (* SIMPLIFICATION STEP 
  let equalities = 
    let env = (metasenv, context, CicUniv.oblivion_ugraph) in 
    let eq_uri = HExtlib.unopt (LibraryObjects.eq_URI()) in
    Saturation.simplify_equalities bag eq_uri env units 
  in 
  *)
  let passive = Saturation.make_passive units in
  let no = List.length units in
  let active = Saturation.make_active [] in
  let active,passive,newmeta = 
    if paramod then active,passive,newmeta
    else
      Saturation.pump_actives 
        context bag newmeta active passive (no+1) infinity
  in 
    (active,passive,bag),cache,newmeta

let fill_hypothesis context metasenv oldnewmeta term tables (universe:Universe.universe) cache auto fast = 
  let head, metasenv, args, newmeta =
    TermUtil.saturate_term oldnewmeta metasenv context term 0
  in
  let propositional_args = 
    HExtlib.filter_map
      (function 
      | Cic.Meta(i,_) -> 
          let _,_,mt = CicUtil.lookup_meta i metasenv in
          let sort,u = 
            CicTypeChecker.type_of_aux' metasenv context mt 
              CicUniv.oblivion_ugraph
          in
          if is_propositional context sort then Some i else None 
      | _ -> assert false)
    args
  in
  let results,cache,newmeta = 
    if propositional_args = [] then 
      let newmetas = List.filter (fun (i,_,_) -> i >= oldnewmeta) metasenv in
      [args,metasenv,newmetas,head,newmeta],cache,newmeta
    else
      (*
      let proof = 
        None,metasenv,term,term (* term non e' significativo *)
      in *)
      let flags = 
        if fast then
          {AutoTypes.default_flags() with 
           AutoTypes.timeout = Unix.gettimeofday() +. 1.0;
           maxwidth = 2;maxdepth = 2;
           use_paramod=true;use_only_paramod=false}
        else
          {AutoTypes.default_flags() with
           AutoTypes.timeout = Unix.gettimeofday() +. 1.0;
           maxwidth = 2;maxdepth = 4;
           use_paramod=true;use_only_paramod=false} 
      in
      match auto newmeta tables universe cache context metasenv propositional_args flags with
      | [],cache,newmeta -> raise (FillingFailure (cache,newmeta))
      | substs,cache,newmeta ->
          List.map 
            (fun subst ->
              let metasenv = 
                CicMetaSubst.apply_subst_metasenv subst metasenv
              in
              let head = CicMetaSubst.apply_subst subst head in
              let newmetas = 
                List.filter (fun (i,_,_) ->i >= oldnewmeta) metasenv 
              in
              let args = List.map (CicMetaSubst.apply_subst subst) args in
              let newm = CicMkImplicit.new_meta metasenv subst in
                args,metasenv,newmetas,head,max newm newmeta)
            substs, cache, newmeta
  in
  results,cache,newmeta

let build_equalities auto context metasenv tables universe cache newmeta equations =
  List.fold_left 
    (fun (facts,cache,newmeta) (t,ty) ->
       (* in any case we add the equation to the cache *)
       let cache = AutoCache.cache_add_list cache context [(t,ty)] in
       try
         let saturated,cache,newmeta = 
           fill_hypothesis context metasenv newmeta ty tables universe cache auto true
         in
         let (active,passive,bag) = tables in
         let eqs,bag,newmeta = 
           List.fold_left 
             (fun (acc,bag,newmeta) (args,metasenv,newmetas,head,newmeta') ->
                let maxmeta,equality =
                  build_equality bag head args t newmetas newmeta'
                in
                  equality::acc,bag,maxmeta)
             ([],bag,newmeta) saturated
         in
           (eqs@facts, cache, newmeta)
       with FillingFailure (cache,newmeta) ->
         (* if filling hypothesis fails we add the equation to
            the cache *)
         (facts,cache,newmeta)
      )
    ([],cache,newmeta) equations

let close_more tables maxmeta context status auto universe cache =
  let (active,passive,bag) = tables in
  let proof, goalno = status in
  let _, metasenv,_subst,_,_, _ = proof in  
  let signature = MetadataQuery.signature_of metasenv goalno in
  let equations = 
    retrieve_equations false signature universe cache context metasenv 
  in
  let eqs_and_types =
    HExtlib.filter_map 
      (fun t -> 
         let ty,_ =
           CicTypeChecker.type_of_aux' metasenv context t
           CicUniv.oblivion_ugraph in
           (* retrieve_equations could also return flexible terms *)
           if is_an_equality ty then Some(t,ty) else None)
      equations in
  let units, cache, maxm = 
      build_equalities auto context metasenv tables universe cache maxmeta eqs_and_types in
  debug_print (lazy (">>>>>>> gained from a new context saturation >>>>>>>>>" ^
    string_of_int maxm));
  List.iter
    (fun e -> debug_print (lazy (Equality.string_of_equality e))) 
    units;
  debug_print (lazy ">>>>>>>>>>>>>>>>>>>>>>");
  let passive = Saturation.add_to_passive units passive in
  let no = List.length units in
  debug_print (lazy ("No = " ^ (string_of_int no)));
  let active,passive,newmeta = 
    Saturation.pump_actives context bag maxm active passive (no+1) infinity
  in 
    (active,passive,bag),cache,newmeta

let find_context_equalities 
  maxmeta bag context proof (universe:Universe.universe) cache 
=
  let module C = Cic in
  let module S = CicSubstitution in
  let module T = CicTypeChecker in
  let _,metasenv,_subst,_,_, _ = proof in
  let newmeta = max (ProofEngineHelpers.new_meta_of_proof ~proof) maxmeta in
  (* if use_auto is true, we try to close the hypothesis of equational
    statements using auto; a naif, and probably wrong approach *)
  let rec aux cache index newmeta = function
    | [] -> [], newmeta,cache
    | (Some (_, C.Decl (term)))::tl ->
        debug_print
          (lazy
             (Printf.sprintf "Examining: %d (%s)" index (CicPp.ppterm term)));
        let do_find context term =
          match term with
          | C.Prod (name, s, t) when is_an_equality t ->
              (try 
                
                let term = S.lift index term in
                let saturated,cache,newmeta = 
                  fill_hypothesis context metasenv newmeta term 
                    empty_tables universe cache default_auto false
                in
                let eqs,newmeta = 
                  List.fold_left 
                   (fun (acc,newmeta) (args,metasenv,newmetas,head,newmeta') ->
                     let newmeta, equality = 
                       build_equality
                         bag head args (Cic.Rel index) newmetas (max newmeta newmeta')
                     in
                     equality::acc, newmeta + 1)
                   ([],newmeta) saturated
                in
                 eqs, newmeta, cache
              with FillingFailure (cache,newmeta) ->
                [],newmeta,cache)
          | C.Appl [C.MutInd (uri, _, _); ty; t1; t2]
              when LibraryObjects.is_eq_URI uri ->
              let term = S.lift index term in
              let newmeta, e = 
                build_equality bag term [] (Cic.Rel index) [] newmeta
              in
              [e], (newmeta+1),cache
          | _ -> [], newmeta, cache
        in 
        let eqs, newmeta, cache = do_find context term in
        let rest, newmeta,cache = aux cache (index+1) newmeta tl in
        List.map (fun x -> index,x) eqs @ rest, newmeta, cache
    | _::tl ->
        aux cache (index+1) newmeta tl
  in
  let il, maxm, cache = 
    aux cache 1 newmeta context 
  in
  let indexes, equalities = List.split il in
  indexes, equalities, maxm, cache
;;

(********** PARAMETERS PASSING ***************)

let bool params name default =
    try 
      let s = List.assoc name params in 
      if s = "" || s = "1" || s = "true" || s = "yes" || s = "on" then true
      else if s = "0" || s = "false" || s = "no" || s= "off" then false
      else 
        let msg = "Unrecognized value for parameter "^name^"\n" in
        let msg = msg^"Accepted values are 1,true,yes,on and 0,false,no,off" in
        raise (ProofEngineTypes.Fail (lazy msg))
    with Not_found -> default
;; 

let string params name default =
    try List.assoc name params with
    | Not_found -> default
;; 

let int params name default =
    try int_of_string (List.assoc name params) with
    | Not_found -> default
    | Failure _ -> 
        raise (ProofEngineTypes.Fail (lazy (name ^ " must be an integer")))
;;  

let flags_of_params params ?(for_applyS=false) () =
 let int = int params in
 let bool = bool params in
 let close_more = bool "close_more" false in
 let use_paramod = bool "use_paramod" true in
 let use_only_paramod =
  if for_applyS then true else bool "paramodulation" false in
 let use_library = bool "library"  
   ((AutoTypes.default_flags()).AutoTypes.use_library) in
 let depth = int "depth" ((AutoTypes.default_flags()).AutoTypes.maxdepth) in
 let width = int "width" ((AutoTypes.default_flags()).AutoTypes.maxwidth) in
 let size = int "size" ((AutoTypes.default_flags()).AutoTypes.maxsize) in
 let gsize = int "gsize" ((AutoTypes.default_flags()).AutoTypes.maxgoalsizefactor) in
 let do_type = bool "type" false in
 let timeout = int "timeout" 0 in
  { AutoTypes.maxdepth = 
      if use_only_paramod then 2 else depth;
    AutoTypes.maxwidth = width;
    AutoTypes.maxsize = size;
    AutoTypes.timeout = 
      if timeout = 0 then
       if for_applyS then Unix.gettimeofday () +. 30.0
       else
         infinity
      else
       Unix.gettimeofday() +. (float_of_int timeout);
    AutoTypes.use_library = use_library; 
    AutoTypes.use_paramod = use_paramod;
    AutoTypes.use_only_paramod = use_only_paramod;
    AutoTypes.close_more = close_more;
    AutoTypes.dont_cache_failures = false;
    AutoTypes.maxgoalsizefactor = gsize;
    AutoTypes.do_types = do_type;
  }

let universe_of_params metasenv context universe tl =
  if tl = [] then universe else 
   let tys =
     List.map 
       (fun term ->
         fst (CicTypeChecker.type_of_aux' metasenv context term
                CicUniv.oblivion_ugraph))
       tl          
   in
     Universe.index_list Universe.empty context (List.combine tl tys)
;;


(***************** applyS *******************)

let new_metasenv_and_unify_and_t 
 dbd flags universe proof goal ?tables newmeta' metasenv' 
 context term' ty termty goal_arity 
=
 let (consthead,newmetasenv,arguments,_) =
   TermUtil.saturate_term newmeta' metasenv' context termty goal_arity in
 let term'' = 
   match arguments with [] -> term' | _ -> Cic.Appl (term'::arguments) 
 in
 let proof',oldmetasenv =
  let (puri,metasenv,_subst,pbo,pty, attrs) = proof in
   (puri,newmetasenv,_subst,pbo,pty, attrs),metasenv
 in
 let goal_for_paramod =
  match LibraryObjects.eq_URI () with
  | Some uri -> 
      Cic.Appl [Cic.MutInd (uri,0,[]); Cic.Sort Cic.Prop; consthead; ty]
  | None -> raise (ProofEngineTypes.Fail (lazy "No equality defined"))
 in
 let newmeta = CicMkImplicit.new_meta newmetasenv (*subst*) [] in
 let metasenv_for_paramod = (newmeta,context,goal_for_paramod)::newmetasenv in
 let proof'' = 
   let uri,_,_subst,p,ty, attrs = proof' in 
   uri,metasenv_for_paramod,_subst,p,ty, attrs 
 in
 let irl = CicMkImplicit.identity_relocation_list_for_metavariable context in
 let proof''',goals =
  ProofEngineTypes.apply_tactic
    (EqualityTactics.rewrite_tac ~direction:`RightToLeft
      ~pattern:(ProofEngineTypes.conclusion_pattern None) 
        (Cic.Meta(newmeta,irl)) [])
   (proof'',goal)
 in
 let goal = match goals with [g] -> g | _ -> assert false in
 let  proof'''', _  =
   ProofEngineTypes.apply_tactic 
     (PrimitiveTactics.apply_tac term'')
     (proof''',goal) 
 in
 match 
   let (active, passive,bag), cache, maxmeta =
     init_cache_and_tables ~dbd flags.use_library true true false universe
     (proof'''',newmeta)
   in
     Saturation.given_clause bag maxmeta (proof'''',newmeta) active passive 
       max_int max_int flags.timeout
 with
 | None, _,_,_ -> 
     raise (ProofEngineTypes.Fail (lazy ("FIXME: propaga le tabelle"))) 
 | Some (_,proof''''',_), active,passive,_  ->
     proof''''',
     ProofEngineHelpers.compare_metasenvs ~oldmetasenv
       ~newmetasenv:(let _,m,_subst,_,_, _ = proof''''' in m),  active, passive
;;

let rec count_prods context ty =
 match CicReduction.whd context ty with
    Cic.Prod (n,s,t) -> 1 + count_prods (Some (n,Cic.Decl s)::context) t
  | _ -> 0

let apply_smart 
  ~dbd ~term ~subst ~universe ?tables ~params:(univ,params) (proof, goal) 
=
 let module T = CicTypeChecker in
 let module R = CicReduction in
 let module C = Cic in
  let (_,metasenv,_subst,_,_, _) = proof in
  let metano,context,ty = CicUtil.lookup_meta goal metasenv in
  let flags = flags_of_params params ~for_applyS:true () in
  let universe = universe_of_params metasenv context universe univ in
  let newmeta = CicMkImplicit.new_meta metasenv subst in
   let exp_named_subst_diff,newmeta',newmetasenvfragment,term' =
    match term with
       C.Var (uri,exp_named_subst) ->
        let newmeta',newmetasenvfragment,exp_named_subst',exp_named_subst_diff =
         PrimitiveTactics.generalize_exp_named_subst_with_fresh_metas context newmeta uri
          exp_named_subst
        in
         exp_named_subst_diff,newmeta',newmetasenvfragment,
          C.Var (uri,exp_named_subst')
     | C.Const (uri,exp_named_subst) ->
        let newmeta',newmetasenvfragment,exp_named_subst',exp_named_subst_diff =
         PrimitiveTactics.generalize_exp_named_subst_with_fresh_metas context newmeta uri
          exp_named_subst
        in
         exp_named_subst_diff,newmeta',newmetasenvfragment,
          C.Const (uri,exp_named_subst')
     | C.MutInd (uri,tyno,exp_named_subst) ->
        let newmeta',newmetasenvfragment,exp_named_subst',exp_named_subst_diff =
         PrimitiveTactics.generalize_exp_named_subst_with_fresh_metas context newmeta uri
          exp_named_subst
        in
         exp_named_subst_diff,newmeta',newmetasenvfragment,
          C.MutInd (uri,tyno,exp_named_subst')
     | C.MutConstruct (uri,tyno,consno,exp_named_subst) ->
        let newmeta',newmetasenvfragment,exp_named_subst',exp_named_subst_diff =
         PrimitiveTactics.generalize_exp_named_subst_with_fresh_metas context newmeta uri
          exp_named_subst
        in
         exp_named_subst_diff,newmeta',newmetasenvfragment,
          C.MutConstruct (uri,tyno,consno,exp_named_subst')
     | _ -> [],newmeta,[],term
   in
   let metasenv' = metasenv@newmetasenvfragment in
   let termty,_ = 
     CicTypeChecker.type_of_aux' metasenv' context term' CicUniv.oblivion_ugraph
   in
   let termty = CicSubstitution.subst_vars exp_named_subst_diff termty in
   let goal_arity = count_prods context ty in
   let proof, gl, active, passive =
    new_metasenv_and_unify_and_t dbd flags universe proof goal ?tables
     newmeta' metasenv' context term' ty termty goal_arity
   in
    proof, gl, active, passive
;;

(****************** AUTO ********************)

let mk_irl ctx = CicMkImplicit.identity_relocation_list_for_metavariable ctx;;
let ugraph = CicUniv.oblivion_ugraph;;
let typeof = CicTypeChecker.type_of_aux';;
let ppterm ctx t = 
  let names = List.map (function None -> None | Some (x,_) -> Some x) ctx in
  CicPp.pp t names
;;
let is_in_prop context subst metasenv ty =
  let sort,u = typeof ~subst metasenv context ty CicUniv.oblivion_ugraph in
  is_propositional context sort
;;

let assert_proof_is_valid proof metasenv context goalty =
  if debug then
    begin
      let ty,u = typeof metasenv context proof CicUniv.oblivion_ugraph in
      let b,_ = CicReduction.are_convertible context ty goalty u in
        if not b then
          begin
            let names = 
              List.map (function None -> None | Some (x,_) -> Some x) context 
            in
              debug_print (lazy ("PROOF:" ^ CicPp.pp proof names));
              debug_print (lazy ("PROOFTY:" ^ CicPp.pp ty names));
              debug_print (lazy ("GOAL:" ^ CicPp.pp goalty names));
              debug_print (lazy ("MENV:" ^ CicMetaSubst.ppmetasenv [] metasenv));
          end;
        assert b
    end
  else ()
;;

let assert_subst_are_disjoint subst subst' =
  if debug then
    assert(List.for_all
             (fun (i,_) -> List.for_all (fun (j,_) -> i<>j) subst') 
             subst)
  else ()
;;

let split_goals_in_prop metasenv subst gl =
  List.partition 
    (fun g ->
      let _,context,ty = CicUtil.lookup_meta g metasenv in
      try
        let sort,u = typeof ~subst metasenv context ty ugraph in
        is_propositional context sort
      with 
      | CicTypeChecker.AssertFailure s 
      | CicTypeChecker.TypeCheckerFailure s -> 
          debug_print 
            (lazy ("NON TIPA" ^ ppterm context (CicMetaSubst.apply_subst subst ty)));
          debug_print s;
          false)
    (* FIXME... they should type! *)
    gl
;;

let split_goals_with_metas metasenv subst gl =
  List.partition 
    (fun g ->
      let _,context,ty = CicUtil.lookup_meta g metasenv in
      let ty = CicMetaSubst.apply_subst subst ty in
      CicUtil.is_meta_closed ty)
    gl
;;

let order_new_goals metasenv subst open_goals ppterm =
  let prop,rest = split_goals_in_prop metasenv subst open_goals in
  let closed_prop, open_prop = split_goals_with_metas metasenv subst prop in
  let closed_type, open_type = split_goals_with_metas metasenv subst rest in
  let open_goals =
    (List.map (fun x -> x,P) (open_prop @ closed_prop)) 
    @ 
    (List.map (fun x -> x,T) (open_type @ closed_type))
  in
  let tys = 
    List.map 
      (fun (i,sort) -> 
        let _,_,ty = CicUtil.lookup_meta i metasenv in i,ty,sort) open_goals 
  in
  debug_print (lazy ("   OPEN: "^
    String.concat "\n" 
      (List.map 
         (function
            | (i,t,P) -> string_of_int i   ^ ":"^ppterm t^ "Prop" 
            | (i,t,T) -> string_of_int i  ^ ":"^ppterm t^ "Type")
         tys)));
  open_goals
;;

let is_an_equational_goal = function
  | Cic.Appl [Cic.MutInd(u,_,_);_;_;_] when LibraryObjects.is_eq_URI u -> true
  | _ -> false
;;

(*
let prop = function (_,depth,P) -> depth < 9 | _ -> false;;
*)

let calculate_timeout flags = 
    if flags.timeout = 0. then 
      (debug_print (lazy "AUTO WITH NO TIMEOUT");
       {flags with timeout = infinity})
    else 
      flags 
;;
let is_equational_case goalty flags =
  let ensure_equational t = 
    if is_an_equational_goal t then true 
    else false
    (*
      let msg="Not an equational goal.\nYou cant use the paramodulation flag"in
      raise (ProofEngineTypes.Fail (lazy msg))
    *)
  in
  (flags.use_paramod && is_an_equational_goal goalty) || 
  (flags.use_only_paramod && ensure_equational goalty)
;;
(*
let cache_add_success sort cache k v =
  if sort = P then cache_add_success cache k v else cache_remove_underinspection
  cache k
;;
*)

type menv = Cic.metasenv
type subst = Cic.substitution
type goal = ProofEngineTypes.goal * int * AutoTypes.sort
let candidate_no = ref 0;;
type candidate = int * Cic.term
type cache = AutoCache.cache
type tables = 
  Saturation.active_table * Saturation.passive_table * Equality.equality_bag

type fail = 
  (* the goal (mainly for depth) and key of the goal *)
  goal * AutoCache.cache_key
type op = 
  (* goal has to be proved *)
  | D of goal 
  (* goal has to be cached as a success obtained using candidate as the first
   * step *)
  | S of goal * AutoCache.cache_key * candidate * int 
type elem = 
  (* menv, subst, size, operations done (only S), operations to do, failures to cache if any op fails *)
  menv * subst * int * op list * op list * fail list 
type status = 
  (* list of computations that may lead to the solution: all op list will
   * end with the same (S(g,_)) *)
  elem list
type auto_result = 
  (* menv, subst, alternatives, tables, cache, maxmeta *)
  | Proved of menv * subst * elem list * tables * cache * int
  | Gaveup of tables * cache * int 


(* the status exported to the external observer *)  
type auto_status = 
  (* context, (goal,candidate) list, and_list, history *)
  Cic.context * (int * Cic.term * bool * int * (int * Cic.term) list) list * 
  (int * Cic.term * int) list * Cic.term list

let d_prefix l =
  let rec aux acc = function
    | (D g)::tl -> aux (acc@[g]) tl
    | _ -> acc
  in
    aux [] l
;;
let prop_only l =
  List.filter (function (_,_,P) -> true | _ -> false) l
;;

let d_goals l =
  let rec aux acc = function
    | (D g)::tl -> aux (acc@[g]) tl
    | (S _)::tl -> aux acc tl
    | [] -> acc
  in
    aux [] l
;;
let calculate_goal_ty (goalno,_,_) s m = 
  try
    let _,cc,goalty = CicUtil.lookup_meta goalno m in
    (* XXX applicare la subst al contesto? *)
    Some (cc, CicMetaSubst.apply_subst s goalty)
  with CicUtil.Meta_not_found i when i = goalno -> None
;;
let calculate_closed_goal_ty (goalno,_,_) s = 
  try
    let cc,_,goalty = List.assoc goalno s in
    (* XXX applicare la subst al contesto? *)
    Some (cc, CicMetaSubst.apply_subst s goalty)
  with Not_found -> None
;;
let pp_status ctx status = 
  if debug then 
  let names = Utils.names_of_context ctx in
  let pp x = 
    let x = 
      ProofEngineReduction.replace 
        ~equality:(fun a b -> match b with Cic.Meta _ -> true | _ -> false) 
          ~what:[Cic.Rel 1] ~with_what:[Cic.Implicit None] ~where:x
    in
    CicPp.pp x names
  in
  let string_of_do m s (gi,_,_ as g) d =
    match calculate_goal_ty g s m with
    | Some (_,gty) -> Printf.sprintf "D(%d, %s, %d)" gi (pp gty) d
    | None -> Printf.sprintf "D(%d, _, %d)" gi d
  in
  let string_of_s m su k (ci,ct) gi =
    Printf.sprintf "S(%d, %s, %s, %d)" gi (pp k) (pp ct) ci
  in
  let string_of_ol m su l =
    String.concat " | " 
      (List.map 
        (function 
          | D (g,d,s) -> string_of_do m su (g,d,s) d 
          | S ((gi,_,_),k,c,_) -> string_of_s m su k c gi) 
        l)
  in
  let string_of_fl m s fl = 
    String.concat " | " 
      (List.map (fun ((i,_,_),ty) -> 
         Printf.sprintf "(%d, %s)" i (pp ty)) fl)
  in
  let rec aux = function
    | [] -> ()
    | (m,s,_,_,ol,fl)::tl ->
        Printf.eprintf "< [%s] ;;; [%s]>\n" 
          (string_of_ol m s ol) (string_of_fl m s fl);
        aux tl
  in
    Printf.eprintf "-------------------------- status -------------------\n";
    aux status;
    Printf.eprintf "-----------------------------------------------------\n";
;;
  
let auto_status = ref [] ;;
let auto_context = ref [];;
let in_pause = ref false;;
let pause b = in_pause := b;;
let cond = Condition.create ();;
let mutex = Mutex.create ();;
let hint = ref None;;
let prune_hint = ref [];;

let step _ = Condition.signal cond;;
let give_hint n = hint := Some n;;
let give_prune_hint hint =
  prune_hint := hint :: !prune_hint
;;

let check_pause _ =
  if !in_pause then
    begin
      Mutex.lock mutex;
      Condition.wait cond mutex;
      Mutex.unlock mutex
    end
;;

let get_auto_status _ = 
  let status = !auto_status in
  let and_list,elems,last = 
    match status with
    | [] -> [],[],[]
    | (m,s,_,don,gl,fail)::tl ->
        let and_list = 
          HExtlib.filter_map 
            (fun (id,d,_ as g) -> 
              match calculate_goal_ty g s m with
              | Some (_,x) -> Some (id,x,d) | None -> None)
            (d_goals gl)
        in
        let rows = 
          (* these are the S goalsin the or list *)
          let orlist = 
            List.map
              (fun (m,s,_,don,gl,fail) -> 
                HExtlib.filter_map
                  (function S (g,k,c,_) -> Some (g,k,c) | _ -> None) 
                  (List.rev don @ gl))
              status
          in
          (* this function eats id from a list l::[id,x] returning x, l *)
          let eat_tail_if_eq id l = 
            let rec aux (s, l) = function
              | [] -> s, l
              | ((id1,_,_),k1,c)::tl when id = id1 ->
                  (match s with
                  | None -> aux (Some c,l) tl
                  | Some _ -> assert false)
              | ((id1,_,_),k1,c as e)::tl -> aux (s, e::l) tl
            in
            let c, l = aux (None, []) l in
            c, List.rev l
          in
          let eat_in_parallel id l =
            let rec aux (b,eaten, new_l as acc) l =
              match l with
              | [] -> acc
              | l::tl ->
                  match eat_tail_if_eq id l with
                  | None, l -> aux (b@[false], eaten, new_l@[l]) tl
                  | Some t,l -> aux (b@[true],eaten@[t], new_l@[l]) tl
            in
            aux ([],[],[]) l
          in
          let rec eat_all rows l =
            match l with
            | [] -> rows
            | elem::or_list ->
                match List.rev elem with
                | ((to_eat,depth,_),k,_)::next_lunch ->
                    let b, eaten, l = eat_in_parallel to_eat l in
                    let eaten = HExtlib.list_uniq eaten in
                    let eaten = List.rev eaten in
                    let b = true (* List.hd (List.rev b) *) in
                    let rows = rows @ [to_eat,k,b,depth,eaten] in
                    eat_all rows l
                | [] -> eat_all rows or_list
          in
          eat_all [] (List.rev orlist)
        in
        let history = 
          HExtlib.filter_map
            (function (S (_,_,(_,c),_)) -> Some c | _ -> None) 
            gl 
        in
(*         let rows = List.filter (fun (_,l) -> l <> []) rows in *)
        and_list, rows, history
  in
  !auto_context, elems, and_list, last
;;

(* Works if there is no dependency over proofs *)
let is_a_green_cut goalty =
  CicUtil.is_meta_closed goalty
;;
let rec first_s = function
  | (D _)::tl -> first_s tl
  | (S (g,k,c,s))::tl -> Some ((g,k,c,s),tl)
  | [] -> None
;;
let list_union l1 l2 =
  (* TODO ottimizzare compare *)
  HExtlib.list_uniq (List.sort compare (l1 @ l1))
;;
let eat_head todo id fl orlist = 
  let rec aux acc = function
  | [] -> [], acc
  | (m, s, _, _, todo1, fl1)::tl as orlist -> 
      let rec aux1 todo1 =
        match first_s todo1 with
        | None -> orlist, acc
        | Some (((gno,_,_),_,_,_), todo11) ->
            (* TODO confronto tra todo da ottimizzare *)
            if gno = id && todo11 = todo then 
              aux (list_union fl1 acc) tl
            else 
              aux1 todo11
      in
       aux1 todo1
  in 
    aux fl orlist
;;
let close_proof p ty menv context = 
  let metas =
    List.map fst (CicUtil.metas_of_term p @ CicUtil.metas_of_term ty)
  in
  let menv = List.filter (fun (i,_,_) -> List.exists ((=)i) metas) menv in
  naif_closure p menv context
;;
(* XXX capire bene quando aggiungere alla cache *)
let add_to_cache_and_del_from_orlist_if_green_cut
  g s m cache key todo orlist fl ctx size minsize
= 
  let cache = cache_remove_underinspection cache key in
  (* prima per fare la irl usavamo il contesto vero e proprio e non quello 
   * canonico! XXX *)
  match calculate_closed_goal_ty g s with
  | None -> assert false
  | Some (canonical_ctx , gty) ->
      let goalno,depth,sort = g in
      let irl = mk_irl canonical_ctx in
      let goal = Cic.Meta(goalno, irl) in
      let proof = CicMetaSubst.apply_subst s goal in
      let green_proof, closed_proof = 
        let b = is_a_green_cut proof in
        if not b then
          b, (* close_proof proof gty m ctx *) proof 
        else
          b, proof
      in
      debug_print (lazy ("TENTATIVE CACHE: " ^ CicPp.ppterm key));
      if is_a_green_cut key then
        (* if the initia goal was closed, we cut alternatives *)
        let _ = debug_print (lazy ("MANGIO: " ^ string_of_int goalno)) in
        let orlist, fl = eat_head todo goalno fl orlist in
        let cache = 
          if size < minsize then 
            (debug_print (lazy ("NO CACHE: 2 (size <= minsize)"));cache)
          else 
          (* if the proof is closed we cache it *)
          if green_proof then cache_add_success cache key proof
          else (* cache_add_success cache key closed_proof *) 
            (debug_print (lazy ("NO CACHE: (no gree proof)"));cache)
        in
        cache, orlist, fl, true
      else
        let cache = 
          debug_print (lazy ("TENTATIVE CACHE: " ^ CicPp.ppterm gty));
          if size < minsize then 
            (debug_print (lazy ("NO CACHE: (size <= minsize)")); cache) else
          (* if the substituted goal and the proof are closed we cache it *)
          if is_a_green_cut gty then
            if green_proof then cache_add_success cache gty proof
            else (* cache_add_success cache gty closed_proof *) 
              (debug_print (lazy ("NO CACHE: (no green proof (gty))"));cache)
          else (*
            try
              let ty, _ =
                CicTypeChecker.type_of_aux' ~subst:s 
                  m ctx closed_proof CicUniv.oblivion_ugraph
              in
              if is_a_green_cut ty then 
                cache_add_success cache ty closed_proof
              else cache
            with
            | CicTypeChecker.TypeCheckerFailure _ ->*) 
          (debug_print (lazy ("NO CACHE: (no green gty )"));cache)
        in
        cache, orlist, fl, false
;;
let close_failures (fl : fail list) (cache : cache) = 
  List.fold_left 
    (fun cache ((gno,depth,_),gty) -> 
      debug_print (lazy ("FAIL: INDUCED: " ^ string_of_int gno));
      cache_add_failure cache gty depth) 
    cache fl
;;
let put_in_subst subst metasenv  (goalno,_,_) canonical_ctx t ty =
  let entry = goalno, (canonical_ctx, t,ty) in
  assert_subst_are_disjoint subst [entry];
  let subst = entry :: subst in
  let metasenv = CicMetaSubst.apply_subst_metasenv subst metasenv in
  subst, metasenv
;;
let mk_fake_proof metasenv subst (goalno,_,_) goalty context = 
  None,metasenv,subst ,Cic.Meta(goalno,mk_irl context),goalty, [] 
;;
let equational_case 
  tables maxm cache depth fake_proof goalno goalty subst context 
    flags
=
  let active,passive,bag = tables in
  let ppterm = ppterm context in
  let status = (fake_proof,goalno) in
    if flags.use_only_paramod then
      begin
        debug_print (lazy ("PARAMODULATION SU: " ^ 
                         string_of_int goalno ^ " " ^ ppterm goalty ));
        let goal_steps, saturation_steps, timeout =
          max_int,max_int,flags.timeout 
        in
        match
          Saturation.given_clause bag maxm status active passive 
            goal_steps saturation_steps timeout
        with 
          | None, active, passive, maxmeta -> 
              [], (active,passive,bag), cache, maxmeta, flags
          | Some(subst',(_,metasenv,_subst,proof,_, _),open_goals),active,
            passive,maxmeta ->
              assert_subst_are_disjoint subst subst';
              let subst = subst@subst' in
              let open_goals = 
                order_new_goals metasenv subst open_goals ppterm 
              in
              let open_goals = 
                List.map (fun (x,sort) -> x,depth-1,sort) open_goals 
              in
              incr candidate_no;
                      [(!candidate_no,proof),metasenv,subst,open_goals], 
                (active,passive,bag), 
                cache, maxmeta, flags
      end
    else
      begin
        debug_print 
          (lazy 
           ("SUBSUMPTION SU: " ^ string_of_int goalno ^ " " ^ ppterm goalty));
        let res, maxmeta = 
          Saturation.all_subsumed bag maxm status active passive 
        in
        assert (maxmeta >= maxm);
        let res' =
          List.map 
            (fun (subst',(_,metasenv,_subst,proof,_, _),open_goals) ->
               assert_subst_are_disjoint subst subst';
               let subst = subst@subst' in
               let open_goals = 
                 order_new_goals metasenv subst open_goals ppterm 
               in
               let open_goals = 
                 List.map (fun (x,sort) -> x,depth-1,sort) open_goals 
               in
               incr candidate_no;
                 (!candidate_no,proof),metasenv,subst,open_goals)
            res 
          in
          res', (active,passive,bag), cache, maxmeta, flags 
      end
;;

let try_candidate 
  goalty tables maxm subst fake_proof goalno depth context cand 
=
  let ppterm = ppterm context in
  try 
    let subst,((_,metasenv,_,_,_,_), open_goals),maxmeta =
        (PrimitiveTactics.apply_with_subst ~subst ~maxmeta:maxm ~term:cand)
        (fake_proof,goalno) 
    in
    debug_print (lazy ("   OK: " ^ ppterm cand));
    let metasenv = CicRefine.pack_coercion_metasenv metasenv in
    let open_goals = order_new_goals metasenv subst open_goals ppterm in
    let open_goals = List.map (fun (x,sort) -> x,depth-1,sort) open_goals in
    incr candidate_no;
    Some ((!candidate_no,cand),metasenv,subst,open_goals), tables , maxmeta
  with 
    | ProofEngineTypes.Fail s -> None,tables, maxm
    | CicUnification.Uncertain s ->  None,tables, maxm
;;

let sort_new_elems = 
 List.sort (fun (_,_,_,l1) (_,_,_,l2) -> 
  List.length (prop_only l1) - List.length (prop_only l2))
;;

let applicative_case 
  tables maxm depth subst fake_proof goalno goalty metasenv context universe
  cache
= 
  let candidates = get_candidates universe cache goalty in
  let tables, elems, maxm = 
    List.fold_left 
      (fun (tables,elems,maxm) cand ->
        match 
          try_candidate goalty
            tables maxm subst fake_proof goalno depth context cand
        with
        | None, tables,maxm  -> tables,elems, maxm 
        | Some x, tables, maxm -> tables,x::elems, maxm)
      (tables,[],maxm) candidates
  in
  let elems = sort_new_elems elems in
  elems, tables, cache, maxm 
;;

let equational_and_applicative_case 
  universe flags m s g gty tables cache maxm context 
=
  let goalno, depth, sort = g in
  let fake_proof = mk_fake_proof m s g gty context in
  if is_equational_case gty flags then
    let elems,tables,cache,maxm1, flags =
      equational_case tables maxm cache
        depth fake_proof goalno gty s context flags 
    in
    let maxm = maxm1 in
    let more_elems, tables, cache, maxm1 =
      if flags.use_only_paramod then
        [],tables, cache, maxm
      else
        applicative_case 
          tables maxm depth s fake_proof goalno 
            gty m context universe cache 
    in
    let maxm = maxm1 in
      elems@more_elems, tables, cache, maxm, flags            
  else
    let elems, tables, cache, maxm =
      applicative_case tables maxm depth s fake_proof goalno 
        gty m context universe cache 
    in
      elems, tables, cache, maxm, flags  
;;
let rec condition_for_hint i = function
  | [] -> false
  | S (_,_,(j,_),_):: tl -> j <> i (* && condition_for_hint i tl *)
  | _::tl -> condition_for_hint i tl
;;
let remove_s_from_fl (id,_,_) (fl : fail list) =
  let rec aux = function
    | [] -> []
    | ((id1,_,_),_)::tl when id = id1 -> tl
    | hd::tl ->  hd :: aux tl
  in 
    aux fl
;;

let prunable_for_size flags s m todo =
  let rec aux b = function
    | (S _)::tl -> aux b tl
    | (D (_,_,T))::tl -> aux b tl
    | (D g)::tl -> 
	(match calculate_goal_ty g s m with
          | None -> aux b tl
          | Some (canonical_ctx, gty) -> 
            let gsize, _ = 
              Utils.weight_of_term 
		~consider_metas:false ~count_metas_occurrences:true gty in
	    let newb = b || gsize > flags.maxgoalsizefactor in
	    aux newb tl)
    | [] -> b
  in
    aux false todo

(*
let prunable ty todo =
  let rec aux b = function
    | (S(_,k,_,_))::tl -> aux (b || Equality.meta_convertibility k ty) tl
    | (D (_,_,T))::tl -> aux b tl
    | D _::_ -> false
    | [] -> b
  in
    aux false todo
;;
*)

let prunable menv subst ty todo =
  let rec aux = function
    | (S(_,k,_,_))::tl ->
	 (match Equality.meta_convertibility_subst k ty menv with
	  | None -> aux tl
	  | Some variant -> 
	       no_progress variant tl (* || aux tl*))
    | (D (_,_,T))::tl -> aux tl
    | _ -> false
  and no_progress variant = function
    | [] -> (*prerr_endline "++++++++++++++++++++++++ no_progress";*) true
    | D ((n,_,P) as g)::tl -> 
	(match calculate_goal_ty g subst menv with
           | None -> no_progress variant tl
           | Some (_, gty) -> 
	       (match calculate_goal_ty g variant menv with
		  | None -> assert false
		  | Some (_, gty') ->
		      if gty = gty' then
			 no_progress variant tl
		      else false))
    | _::tl -> no_progress variant tl
  in
    aux todo

;;
let condition_for_prune_hint prune (m, s, size, don, todo, fl) =
  let s = 
    HExtlib.filter_map (function S (_,_,(c,_),_) -> Some c | _ -> None) todo 
  in
  List.for_all (fun i -> List.for_all (fun j -> i<>j) prune) s
;;
let filter_prune_hint l =
  let prune = !prune_hint in
  prune_hint := []; (* possible race... *)
  if prune = [] then l
  else List.filter (condition_for_prune_hint prune) l
;;
let auto_main tables maxm context flags universe cache elems =
  auto_context := context;
  let rec aux tables maxm flags cache (elems : status) =
(*     pp_status context elems; *)
(* DEBUGGING CODE: uncomment these two lines to stop execution at each iteration
    auto_status := elems;
    check_pause ();
*)
    let elems = filter_prune_hint elems in
    match elems with
    | (m, s, size, don, todo, fl)::orlist when !hint <> None ->
        (match !hint with
        | Some i when condition_for_hint i todo ->
            aux tables maxm flags cache orlist
        | _ ->
          hint := None;
          aux tables maxm flags cache elems)
    | [] ->
        (* complete failure *)
        Gaveup (tables, cache, maxm)
    | (m, s, _, _, [],_)::orlist ->
        (* complete success *)
        Proved (m, s, orlist, tables, cache, maxm)
    | (m, s, size, don, (D (_,_,T))::todo, fl)::orlist 
      when not flags.AutoTypes.do_types ->
        (* skip since not Prop, don't even check if closed by side-effect *)
        aux tables maxm flags cache ((m, s, size, don, todo, fl)::orlist)
    | (m, s, size, don, (S(g, key, c,minsize) as op)::todo, fl)::orlist ->
        (* partial success, cache g and go on *)
        let cache, orlist, fl, sibling_pruned = 
          add_to_cache_and_del_from_orlist_if_green_cut 
            g s m cache key todo orlist fl context size minsize
        in
        debug_print (lazy (AutoCache.cache_print context cache));
        let fl = remove_s_from_fl g fl in
        let don = if sibling_pruned then don else op::don in
        aux tables maxm flags cache ((m, s, size, don, todo, fl)::orlist)
    | (m, s, size, don, todo, fl)::orlist 
      when List.length(prop_only (d_goals todo)) > flags.maxwidth ->
        debug_print (lazy ("FAIL: WIDTH"));
        (* too many goals in and generated by last th *)
        let cache = close_failures fl cache in
        aux tables maxm flags cache orlist
    | (m, s, size, don, todo, fl)::orlist when size > flags.maxsize ->
        debug_print 
          (lazy ("FAIL: SIZE: "^string_of_int size ^ 
            " > " ^ string_of_int flags.maxsize ));
        (* we already have a too large proof term *)
        let cache = close_failures fl cache in
        aux tables maxm flags cache orlist
    | _ when Unix.gettimeofday () > flags.timeout ->
        (* timeout *)
        debug_print (lazy ("FAIL: TIMEOUT"));
        Gaveup (tables, cache, maxm)
    | (m, s, size, don, (D (gno,depth,_ as g))::todo, fl)::orlist as status ->
        (* attack g *)
        match calculate_goal_ty g s m with
        | None -> 
            (* closed by side effect *)
            debug_print (lazy ("SUCCESS: SIDE EFFECT: " ^ string_of_int gno));
            aux tables maxm flags cache ((m,s,size,don,todo, fl)::orlist)
        | Some (canonical_ctx, gty) -> 
            let gsize, _ = 
              Utils.weight_of_term ~consider_metas:false ~count_metas_occurrences:true gty 
            in
            if gsize > flags.maxgoalsizefactor then
	      (debug_print (lazy ("FAIL: SIZE: goal: "^string_of_int gsize));
               aux tables maxm flags cache orlist)
            else if prunable_for_size flags s m todo then
		(debug_print (lazy ("POTO at depth: "^(string_of_int depth)));
	         aux tables maxm flags cache orlist)
	    else
            (* still to be proved *)
            (debug_print (lazy ("EXAMINE: "^CicPp.ppterm gty));
            match cache_examine cache gty with
            | Failed_in d when d >= depth -> 
                (* fail depth *)
                debug_print (lazy ("FAIL: DEPTH (cache): "^string_of_int gno));
                let cache = close_failures fl cache in
                aux tables maxm flags cache orlist
            | UnderInspection -> 
                (* fail loop *)
                debug_print (lazy ("FAIL: LOOP: " ^ string_of_int gno));
                let cache = close_failures fl cache in
                aux tables maxm flags cache orlist
            | Succeded t -> 
                debug_print (lazy ("SUCCESS: CACHE HIT: " ^ string_of_int gno));
                let s, m = put_in_subst s m g canonical_ctx t gty in
                aux tables maxm flags cache ((m, s, size, don,todo, fl)::orlist)
            | Notfound 
            | Failed_in _ when depth > 0 -> 
                ( (* more depth or is the first time we see the goal *)
                    if prunable m s gty todo then
                      (debug_print (lazy(
                         "FAIL: LOOP: one father is equal"));
                       aux tables maxm flags cache orlist)
                    else
                    let cache = cache_add_underinspection cache gty depth in
                    auto_status := status;
                    check_pause ();
                    debug_print 
                      (lazy ("INSPECTING: " ^ 
                        string_of_int gno ^ "("^ string_of_int size ^ "): "^
                        CicPp.ppterm gty));
                    (* elems are possible computations for proving gty *)
                    let elems, tables, cache, maxm, flags =
                      equational_and_applicative_case 
                        universe flags m s g gty tables cache maxm context
                    in
                    if elems = [] then
                      (* this goal has failed *)
                      let cache = close_failures ((g,gty)::fl) cache in
                      aux tables maxm flags cache orlist
                    else
                      (* elems = (cand,m,s,gl) *)
                      let size_gl l = List.length 
                        (List.filter (function (_,_,P) -> true | _ -> false) l) 
                      in
                      let elems = 
                        let inj_gl gl = List.map (fun g -> D g) gl in
                        let rec map = function
                          | [] -> assert false
                          | (cand,m,s,gl)::[] ->
                              (* in the last one we add the failure *)
                              let todo = 
                                inj_gl gl @ (S(g,gty,cand,size+1))::todo 
                              in
                              (* we are the last in OR, we fail on g and 
                               * also on all failures implied by g *)
                              (m,s, size + size_gl gl, don, todo, (g,gty)::fl)
                              :: orlist
                          | (cand,m,s,gl)::tl -> 
                              (* we add the S step after gl and before todo *)
                              let todo = 
                                inj_gl gl @ (S(g,gty,cand,size+1))::todo 
                              in
                              (* since we are not the last in OR, we do not
                               * imply failures *)
                              (m,s, size + size_gl gl, don, todo, []) :: map tl
                        in
                          map elems
                      in
                        aux tables maxm flags cache elems)
            | _ -> 
                (* no more depth *)
                debug_print (lazy ("FAIL: DEPTH: " ^ string_of_int gno));
                let cache = close_failures fl cache in
                aux tables maxm flags cache orlist)
  in
    (aux tables maxm flags cache elems : auto_result)
;;
    

let
  auto_all_solutions maxm tables universe cache context metasenv gl flags 
=
  let goals = order_new_goals metasenv [] gl CicPp.ppterm in
  let goals = 
    List.map 
      (fun (x,s) -> D (x,flags.maxdepth,s)) goals 
  in
  let elems = [metasenv,[],1,[],goals,[]] in
  let rec aux tables maxm solutions cache elems flags =
    match auto_main tables maxm context flags universe cache elems with
    | Gaveup (tables,cache,maxm) ->
        solutions,cache,maxm
    | Proved (metasenv,subst,others,tables,cache,maxm) -> 
        if Unix.gettimeofday () > flags.timeout then
          ((subst,metasenv)::solutions), cache, maxm
        else
          aux tables maxm ((subst,metasenv)::solutions) cache others flags
  in
  let rc = aux tables maxm [] cache elems flags in
    match rc with
    | [],cache,maxm -> [],cache,maxm
    | solutions,cache,maxm -> 
        let solutions = 
          HExtlib.filter_map
            (fun (subst,newmetasenv) ->
              let opened = 
                ProofEngineHelpers.compare_metasenvs ~oldmetasenv:metasenv ~newmetasenv
              in
              if opened = [] then Some subst else None)
            solutions
        in
         solutions,cache,maxm
;;

(* }}} ****************** AUTO ***************)

let auto flags metasenv tables universe cache context metasenv gl =
  let initial_time = Unix.gettimeofday() in
  let goals = order_new_goals metasenv [] gl CicPp.ppterm in
  let goals = List.map (fun (x,s) -> D(x,flags.maxdepth,s)) goals in
  let elems = [metasenv,[],1,[],goals,[]] in
  match auto_main tables 0 context flags universe cache elems with
  | Proved (metasenv,subst,_, tables,cache,_) -> 
      debug_print(lazy
        ("TIME:"^string_of_float(Unix.gettimeofday()-.initial_time)));
      Some (subst,metasenv), cache
  | Gaveup (tables,cache,maxm) -> 
      debug_print(lazy
        ("TIME:"^string_of_float(Unix.gettimeofday()-.initial_time)));
      None,cache
;;

let applyS_tac ~dbd ~term ~params ~universe =
 ProofEngineTypes.mk_tactic
  (fun status ->
    try 
      let proof, gl,_,_ =
       apply_smart ~dbd ~term ~subst:[] ~params ~universe status
      in 
       proof, gl
    with 
    | CicUnification.UnificationFailure msg
    | CicTypeChecker.TypeCheckerFailure msg ->
        raise (ProofEngineTypes.Fail msg))

let auto_tac ~(dbd:HSql.dbd) ~params:(univ,params) ~universe (proof, goal) =
  let _,metasenv,_subst,_,_, _ = proof in
  let _,context,goalty = CicUtil.lookup_meta goal metasenv in
  let universe = universe_of_params metasenv context universe univ in
  let flags = flags_of_params params () in
  let use_library = flags.use_library in
  let tables,cache,newmeta =
    init_cache_and_tables ~dbd use_library flags.use_only_paramod true 
      false universe (proof, goal) in
  let tables,cache,newmeta =
    if flags.close_more then
      close_more 
        tables newmeta context (proof, goal) 
          auto_all_solutions universe cache 
    else tables,cache,newmeta in
  let initial_time = Unix.gettimeofday() in
  let (_,oldmetasenv,_subst,_,_, _) = proof in
  hint := None;
  let elem = 
    metasenv,[],1,[],[D (goal,flags.maxdepth,P)],[]
  in
  match auto_main tables newmeta context flags universe cache [elem] with
    | Proved (metasenv,subst,_, tables,cache,_) -> 
        (*prerr_endline 
          ("TIME:"^string_of_float(Unix.gettimeofday()-.initial_time));*)
        let proof,metasenv =
        ProofEngineHelpers.subst_meta_and_metasenv_in_proof
          proof goal subst metasenv
        in
        let opened = 
          ProofEngineHelpers.compare_metasenvs ~oldmetasenv
            ~newmetasenv:metasenv
        in
          proof,opened
    | Gaveup (tables,cache,maxm) -> 
        debug_print
          (lazy ("TIME:"^
            string_of_float(Unix.gettimeofday()-.initial_time)));
        raise (ProofEngineTypes.Fail (lazy "Auto gave up"))
;;

let auto_tac ~dbd ~params ~universe = 
  ProofEngineTypes.mk_tactic (auto_tac ~params ~dbd ~universe);;

let eq_of_goal = function
  | Cic.Appl [Cic.MutInd(uri,0,_);_;_;_] when LibraryObjects.is_eq_URI uri ->
      uri
  | _ -> raise (ProofEngineTypes.Fail (lazy ("The goal is not an equality ")))
;;

(* performs steps of rewrite with the universe, obtaining if possible 
 * a trivial goal *)
let solve_rewrite_tac ~universe ~params:(univ,params) (proof,goal as status)= 
  let _,metasenv,_subst,_,_,_ = proof in
  let _,context,ty = CicUtil.lookup_meta goal metasenv in
  let steps = int_of_string (string params "steps" "1") in 
  let universe = universe_of_params metasenv context universe univ in
  let eq_uri = eq_of_goal ty in
  let (active,passive,bag), cache, maxm =
     (* we take the whole universe (no signature filtering) *)
     init_cache_and_tables false true false true universe (proof,goal) 
  in
  let initgoal = [], metasenv, ty in
  let table = 
    let equalities = (Saturation.list_of_passive passive) in
    (* we demodulate using both actives passives *)
    List.fold_left (fun tbl eq -> Indexing.index tbl eq) (snd active) equalities
  in
  let env = metasenv,context,CicUniv.oblivion_ugraph in
  match Indexing.solve_demodulating bag env table initgoal steps with 
  | Some (proof, metasenv, newty) ->
      let refl = 
        match newty with
        | Cic.Appl[Cic.MutInd _;eq_ty;left;_] ->
            Equality.Exact (Equality.refl_proof eq_uri eq_ty left)
        | _ -> assert false
      in
      let proofterm,_ = 
        Equality.build_goal_proof 
          bag eq_uri proof refl newty [] context metasenv
      in
      ProofEngineTypes.apply_tactic
        (PrimitiveTactics.apply_tac ~term:proofterm) status
  | None -> 
      raise 
        (ProofEngineTypes.Fail (lazy 
          ("Unable to solve with " ^ string_of_int steps ^ " demodulations")))
;;
let solve_rewrite_tac ~params ~universe () =
  ProofEngineTypes.mk_tactic (solve_rewrite_tac ~universe ~params)
;;

(* DEMODULATE *)
let demodulate_tac ~dbd ~universe ~params:(univ, params) (proof,goal)= 
  let curi,metasenv,_subst,pbo,pty, attrs = proof in
  let metano,context,ty = CicUtil.lookup_meta goal metasenv in
  let universe = universe_of_params metasenv context universe univ in
  let irl = CicMkImplicit.identity_relocation_list_for_metavariable context in
  let initgoal = [], metasenv, ty in
  let eq_uri = eq_of_goal ty in
  let (active,passive,bag), cache, maxm =
     init_cache_and_tables 
       ~dbd false true true false universe (proof,goal) 
  in
  let equalities = (Saturation.list_of_passive passive) in
  (* we demodulate using both actives passives *)
  let table = 
    List.fold_left 
      (fun tbl eq -> Indexing.index tbl eq) 
      (snd active) equalities
  in
  let changed,(newproof,newmetasenv, newty) = 
    Indexing.demodulation_goal bag
      (metasenv,context,CicUniv.oblivion_ugraph) table initgoal 
  in
  if changed then
    begin
      let opengoal = Equality.Exact (Cic.Meta(maxm,irl)) in
      let proofterm,_ = 
        Equality.build_goal_proof bag
          eq_uri newproof opengoal ty [] context metasenv
      in
        let extended_metasenv = (maxm,context,newty)::metasenv in
        let extended_status = 
          (curi,extended_metasenv,_subst,pbo,pty, attrs),goal in
        let (status,newgoals) = 
          ProofEngineTypes.apply_tactic 
            (PrimitiveTactics.apply_tac ~term:proofterm)
            extended_status in
        (status,maxm::newgoals)
    end
  else (* if newty = ty then *)
    raise (ProofEngineTypes.Fail (lazy "no progress"))
  (*else ProofEngineTypes.apply_tactic 
    (ReductionTactics.simpl_tac
      ~pattern:(ProofEngineTypes.conclusion_pattern None)) initialstatus*)
;;

let demodulate_tac ~dbd ~params ~universe = 
  ProofEngineTypes.mk_tactic (demodulate_tac ~dbd ~params ~universe);;

let pp_proofterm = Equality.pp_proofterm;;

let revision = "$Revision: 8654 $";;
let size_and_depth context metasenv t = 100, 100
