(*
    ||M||  This file is part of HELM, an Hypertextual, Electronic        
    ||A||  Library of Mathematics, developed at the Computer Science     
    ||T||  Department, University of Bologna, Italy.                     
    ||I||                                                                
    ||T||  HELM is free software; you can redistribute it and/or         
    ||A||  modify it under the terms of the GNU General Public License   
    \   /  version 2 or (at your option) any later version.      
     \ /   This software is distributed as is, NO WARRANTY.     
      V_______________________________________________________________ *)

(* $Id: nCicPp.ml 9390 2008-12-19 10:11:39Z tassi $ *)

module C = NCic
module R = NReference

let r2s pp_fix_name r = 
  try
    match r with
    | R.Ref (u,R.Ind (_,i,_)) -> 
        (match NCicLibrary.get_obj u with
        | _,_,_,_, C.Inductive (_,_,itl,_) ->
            let _,name,_,_ = List.nth itl i in name
        | _ -> assert false)
    | R.Ref (u,R.Con (i,j,_)) -> 
        (match NCicLibrary.get_obj u with
        | _,_,_,_, C.Inductive (_,_,itl,_) ->
            let _,_,_,cl = List.nth itl i in
            let _,name,_ = List.nth cl (j-1) in name
        | _ -> assert false)
    | R.Ref (u,(R.Decl | R.Def _)) -> 
        (match NCicLibrary.get_obj u with
        | _,_,_,_, C.Constant (_,name,_,_,_) -> name
        | _ -> assert false)
    | R.Ref (u,(R.Fix (i,_,_)|R.CoFix i)) ->
        (match NCicLibrary.get_obj u with
        | _,_,_,_, C.Fixpoint (_,fl,_) -> 
            if pp_fix_name then
              let _,name,_,_,_ = List.nth fl i in name
            else 
              NUri.name_of_uri u ^"("^ string_of_int i ^ ")"
        | _ -> assert false)
  with NCicLibrary.ObjectNotFound _ -> R.string_of_reference r
;;

let string_of_implicit_annotation = function
  | `Closed -> "▪"
  | `Type -> ""
  | `Hole -> "□"
  | `Term -> "Term"
  | `Typeof x -> "Ty("^string_of_int x^")"
;;

let ppterm ~context ~subst ~metasenv:_ ?(inside_fix=false) t = 
  let buff = Buffer.create 100 in
  let f = Format.formatter_of_buffer buff in
  let module F = Format in
  let rec aux ?(toplevel=false) ctx = function
   | C.Rel m ->
         (try 
            let name = List.nth ctx (m-1) in
            F.fprintf f "%s" (if name = "_" then "__"^string_of_int m else name)
         with Failure "nth" | Invalid_argument "List.nth" -> 
             F.fprintf f " -%d" (m - List.length ctx))
   | C.Const r -> F.fprintf f "%s" (r2s inside_fix r)
   | C.Prod ("_",s,t) -> 
       if not toplevel then F.fprintf f "(";
       F.fprintf f "@[<hov 1>";
       (match s with 
       | C.Prod ("_",_,_) -> aux ~toplevel:false ctx s 
       | _ -> aux ~toplevel:true ctx s);
       F.fprintf f "@;→ ";
       aux ~toplevel:true ("_"::ctx) t;
       F.fprintf f "@]";
       if not toplevel then F.fprintf f ")";
   | C.Prod (name,s,t) -> 
       if not toplevel then F.fprintf f "(";
       F.fprintf f "@[<hov 1>";
       F.fprintf f "@[<hov 2>∀%s:@;" name;
       aux ~toplevel:true ctx s; 
       F.fprintf f "@].@;";
       aux ~toplevel:true (name::ctx) t;
       F.fprintf f "@]";
       if not toplevel then F.fprintf f ")";
   | C.Lambda (name,s,t) -> 
       if not toplevel then F.fprintf f "(";
       F.fprintf f "@[<hov 1>";
       F.fprintf f "λ%s:" name;
       aux ~toplevel:true ctx s; 
       F.fprintf f ".@;";
       aux ~toplevel:true (name::ctx) t;
       F.fprintf f "@]";
       if not toplevel then F.fprintf f ")";
   | C.LetIn (name,ty,t,b) -> 
       if not toplevel then F.fprintf f "(";
       F.fprintf f "@[<hov 1>";
       F.fprintf f "let %s:@;" name;
       aux ~toplevel:true ctx ty;
       F.fprintf f "@;≝ ";
       aux ~toplevel:true ctx t;
       F.fprintf f "@;in@;";
       (aux ~toplevel:true (name::ctx) b);
       F.fprintf f "@]";
       if not toplevel then F.fprintf f ")";
   | C.Match (r,oty,t,pl) ->
       F.fprintf f "@[<hov>match ";
       aux ~toplevel:true ctx t;
       F.fprintf f "@;return ";
       aux ~toplevel:true ctx oty;
       F.fprintf f "@; @[<v>[ ";
       if pl <> [] then
         begin
           F.fprintf f "@[<hov 2>%s ⇒@;" 
             (try r2s inside_fix (R.mk_constructor 1 r)
              with R.IllFormedReference _ -> "#ERROR#");
           aux ~toplevel:true ctx (List.hd pl);
           F.fprintf f "@]";
           ignore(List.fold_left 
             (fun i t -> 
              F.fprintf f "@;| @[<hov 2>%s ⇒@;" 
                (try r2s inside_fix (R.mk_constructor i r)
                 with R.IllFormedReference _ -> "#ERROR#");
              aux ~toplevel:true ctx t; 
              F.fprintf f "@]";
              i+1)
             2 (List.tl pl));
         end;
      F.fprintf f "]@] @]";
   | C.Appl [] | C.Appl [_] | C.Appl (C.Appl _::_) -> 
       F.fprintf f "BAD APPLICATION"
   | C.Appl l -> 
       F.fprintf f "@[<hov 2>";
       if not toplevel then F.fprintf f "(";
       aux ctx (List.hd l);
       List.iter (fun x -> F.fprintf f "@;";aux ctx x) (List.tl l);
       if not toplevel then F.fprintf f ")";
       F.fprintf f "@]"
   | C.Implicit annot -> 
       F.fprintf f "?%s" (string_of_implicit_annotation annot)
   | C.Meta (n,lc) when List.mem_assoc n subst -> 
        let _,_,t,_ = List.assoc n subst in
        aux ctx (NCicSubstitution.subst_meta lc t)
   | C.Meta (n,(shift,C.Irl len)) -> 
        F.fprintf f "?%d(%d,%d)" n shift len
   | C.Meta (n,(shift,C.Ctx l)) -> 
        F.fprintf f "?%d(%d,[" n shift;
        if List.length l > 0 then
          begin 
            aux ctx (NCicSubstitution.lift shift (List.hd l));
            List.iter (fun x -> F.fprintf f ",";aux ctx x) 
             (List.map (NCicSubstitution.lift shift) (List.tl l));
          end;
        F.fprintf f "])"
   | C.Sort C.Prop -> F.fprintf f "Prop"
   | C.Sort (C.Type []) -> F.fprintf f "Type0"
   | C.Sort (C.Type [false, u]) -> F.fprintf f "%s" (NUri.name_of_uri u)
   | C.Sort (C.Type [true, u]) -> F.fprintf f "S(%s)" (NUri.name_of_uri u)
   | C.Sort (C.Type l) -> 
       F.fprintf f "Max(";
       aux ctx (C.Sort (C.Type [List.hd l]));
       List.iter (fun x -> F.fprintf f ",";aux ctx (C.Sort (C.Type [x])))
        (List.tl l);
       F.fprintf f ")"
  in 
  aux ~toplevel:true (List.map fst context) t;
  F.fprintf f "@?";
  Buffer.contents buff
;;

let ppobj = function
  | (u,_,metasenv,subst,NCic.Fixpoint (b, fl, _)) -> 
      "{"^NUri.string_of_uri u^"}\n"^
      (if b then "let rec " else "let corec ") ^
       String.concat "\nand " 
        (List.map (fun (_,name,n,ty,bo) ->
          name^ " on " ^ string_of_int n ^ " : " ^ 
          ppterm ~metasenv ~subst ~context:[] ty ^ " :=\n"^
          ppterm ~metasenv ~subst ~context:[] ~inside_fix:true bo) fl)
  | (u,_,metasenv,subst,NCic.Inductive (b, leftno,tyl, _)) -> 
      "{"^NUri.string_of_uri u^"} with "^string_of_int leftno^" fixed params\n"^
      (if b then "inductive " else "coinductive ")^
      String.concat "\nand "
        (List.map (fun (_,name,ty,cl) ->
          name^": "^ppterm ~metasenv ~subst ~context:[] ty^ " :=\n"^
          String.concat "\n"
          (List.map (fun (_,name,ty) ->
           "  | "^name^": "^ppterm ~metasenv ~subst ~context:[] ty)
           cl)) tyl) ^ "."
  | (u,_,metasenv,subst,NCic.Constant (_,name,None,ty, _)) -> 
      "{"^NUri.string_of_uri u^"}\n"^
        "axiom " ^ name ^ " : " ^ 
          ppterm ~metasenv ~subst ~context:[] ty ^ "\n"
  | (u,_,metasenv,subst,NCic.Constant (_,name,Some bo,ty, _)) ->
      "{"^NUri.string_of_uri u^"}\n"^
        "definition " ^ name ^ " : " ^ 
          ppterm ~metasenv ~subst ~context:[] ty ^ " := \n"^
          ppterm ~metasenv ~subst ~context:[] bo ^ "\n"
;;

let rec ppcontext ?(sep="\n") ~subst ~metasenv = function
  | [] -> ""
  | (name, NCic.Decl t) :: tl -> 
      ppcontext ~sep ~subst ~metasenv tl ^
      name ^ ": " ^ ppterm ~subst ~metasenv ~context:tl t ^ sep
  | (name, NCic.Def (bo,ty)) :: tl->
      ppcontext ~sep ~subst ~metasenv tl ^
      name ^ ": " ^ ppterm ~subst ~metasenv ~context:tl ty ^ 
      " := " ^ ppterm ~subst ~metasenv ~context:tl bo ^ sep
;;

let rec ppmetasenv ~subst metasenv = function
  | [] -> ""
  | (i,(name, ctx, ty)) :: tl ->
      let name = match name with Some n -> "(\""^n^"\")" | _ -> "" in
      ppcontext ~sep:"; " ~subst ~metasenv ctx ^
      " ⊢ ?"^string_of_int i^name^" : " ^ 
      ppterm ~metasenv ~subst ~context:ctx ty ^ "\n" ^
      ppmetasenv ~subst metasenv tl
;;

let ppmetasenv ~subst metasenv = ppmetasenv ~subst metasenv metasenv;;

let rec ppsubst ~subst ~metasenv = function
  | [] -> ""
  | (i,(name, ctx, t, ty)) :: tl ->
      
      let name = match name with Some n -> "("^n^")" | _ -> "" in
      ppcontext ~sep:"; " ~subst ~metasenv ctx ^
      " ⊢ ?"^string_of_int i^name^" := " ^ 
      ppterm ~metasenv ~subst ~context:ctx t ^ " : " ^
      ppterm ~metasenv ~subst ~context:ctx ty ^ "\n" ^
      ppsubst ~subst ~metasenv tl
;;

let ppsubst ~metasenv subst = ppsubst ~metasenv ~subst subst;;

let _ = NCicSubstitution.set_ppterm ppterm;;

