#ifndef H_single_axon
#define H_single_axon 1
//
// Simulate a single axon with an extracellular resistance.  This is a test for
// our Hodgkin-Huxley simulator.
//
#include <math.h>
#include "hodgkin_huxley.h"
#include "array.h"

//
// The following class contains the information for a particular segment
// of the axon.
//
struct AxonCompartment : public Hodgkin_Huxley
{
  friend class SingleAxon;

  Float compartment_length;	// The length of this compartment.
  Float capacitance;		// The capacitance of this compartment (nF).
  Float Vm, last_Vm;		// Transmembrane voltage and last transmembrane
				// voltage.

  AxonCompartment(Float ini_v = -65.) : // Constructor.
    Hodgkin_Huxley(ini_v),
    compartment_length(0),
    Vm(ini_v),  last_Vm(ini_v),
    last_hh(ini_v)
  {}

protected:
  Hodgkin_Huxley last_hh;	// The previous hh values (so we can advance
				// multiple times after correcting the
				// voltage).
};

//
// The following class simulates an axon with von Neuman (sealed end) boundary
// conditions.
//
class SingleAxon
{
public:
//
// Constructor for the ExAxon class.  Arguments:
//
// 1) The z grid (the coordinates of the centers of each compartment).
// 2) The number of compartments.
// 3) The diameter of the axon (um).
// 4) The capacitance (uF/cm^2).
// 5) The intracellular resistivity (Ohm-cm).
//
  SingleAxon(const Float *z_grid, unsigned n_compartments, Float axon_diameter,
	     Float capacitance, Float int_resistivity);
				//%input z_grid(n_compartments)

//
// Initialize the whole axon to a given transmembrane voltage.
//
  void initialize(Float ini_v);

//
// Advance to the next timestep.  Arguments:
// 1) dt.
// 2) The extracellular voltages.
//    We assume a Neumann condition on the extracellular voltages::
//    dVe/dx = 0 at the edges.
//
// All voltages and HH state variables are advanced.
//
  void advance(Float dt, const Float *Ve);
				//%input Ve(THIS->get_n_compartments())

//
// Save the current voltages so we can compute the time derivatives.
//
  void save_voltages();

//
// Make a copy of the current Hodgkin-Huxley state variables:
//
  void save_hh_state();

//
// Restore the saved HH state variables.  This allows multiple iterations
// for a single timestep.
//
  void restore_hh_state();

//
// Get the number of compartments:
//
  unsigned get_n_compartments() const { return compartments.length(); }

//
// Return the voltage at a given position along the axon.  Arguments:
// 1) The position (um).
// The voltage is interpolated between the two adjacent points.
//
// vm is the transmembrane voltage, not the intracellular voltage.
//
  Float vm(Float x_pos) const;

//
// Get the whole vector of voltages:
//
  void get_vm(Float *out_vec) const;//%output out_vec(THIS->get_n_compartments())

//
// Get the transmembrane current per unit area at a given point.  Arguments:
// 1) The x value.
//
  Float Jm(Float xpos) const;

//
// Get all the transmembrane currents from each compartment.  This is the
// total current through the membrane of the compartment, not the current
// per unit area.
//
  void Im_compartment(Float *Im_vec) const;//%output Im_vec(THIS->get_n_compartments())

//
// Return pointers to axon compartments:
//
  void get_compartments(AxonCompartment **cvec) const;//%output cvec(THIS->get_n_compartments())

//
// Global data you can change without messing things up:
//
  Float g_syn;			// The synaptic conductance at the tip.
  static Float E_syn;		// The reversal potential.

private:
  unsigned last_k;		// Last k value for inquiry by Jm(xpos) or
				// Vm(xpos).
  Float last_dt;		// The last timestep used (stored so we can 
				// compute capacitative currents).
  Float axon_radius;		// Radius of the axon (um).
  Float membrane_capacitance;	// Capacitance per unit area, in units of
				// nF/um^2.
  Float resistivity_int;	// Internal resistivity in MOhm-um.  (Note
				// these units!)

  array<AxonCompartment> compartments;	// The various compartments.
  array<Float> subdiag;		// The subdiagonal elements (used for
				// solving the tridiaongal system).
  array<Float> z_grid;		// The locations of each of the grid points.
				// These are the centers of the compartments.

  void setup_subdiag();		// Compute the subdiagonals.
};

#endif // H_single_axon
