/*!
  @file           DBMSrv_HSSNodes.cpp
  @author         Bernd Vorsprach - bernd.vorsprach@sap.com
  @brief          DBMServer to DBMServer Communication - Implementation

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/

/*
  -----------------------------------------------------------------------------
  includes
  -----------------------------------------------------------------------------
 */
#define XPARAM_NAMES_ONLY
#include "SAPDB/KernelCommon/ParameterNames/KernelParam_RunTime.h"
#include "SAPDB/RunTime/Configuration/RTEConf_ParameterAccess.hpp"
#include "SAPDB/RunTime/Configuration/RTEConf_ParameterAccess.hpp"
#include "SAPDB/DBM/Srv/DBMSrv_HSSNodes.hpp"

#include "hsp02.h"

#define CMD_RESTART_STANDBY    "db_execute restart standby"
#define CMD_INIT_STANDBY       "db_execute init standby"
#define CMD_PARAM_REMOVE       "param_rmfile"
#define CMD_PARAM_DIRECTPUT    "param_directput"
#define CMD_PARAM_START        "param_startsession"
#define CMD_PARAM_ABORT        "param_abortsession"
#define CMD_PARAM_RESTORE      "param_restore 1"
#define CMD_PARAM_CHECKALL     "param_checkall"
#define CMD_DB_ADDVOLUME       "db_addvolume"

#define CMD_DB_RESTARTSTANDBY  "db_execute restart standby"
#define CMD_DB_INITSTANDBY     "db_execute init standby"

/*! @brief   constructor */
DBMSrv_HSSNode :: DBMSrv_HSSNode
    (       SAPDB_Int                 nNumber,
      const Tools_DynamicUTF8String & sServer,
      const Tools_DynamicUTF8String & sDatabase,
      const Tools_DynamicUTF8String & sUser,
              SAPDBErr_MessageList  & oMsgList )
    : DBMCli_Database(sServer, sDatabase, sUser, oMsgList, false),
      m_nNumber(nNumber),
      m_nDelayTime(0)
{
} // end DBMSrv_HSSNode :: DBMSrv_HSSNode

/*! @brief   destructor */
DBMSrv_HSSNode :: ~DBMSrv_HSSNode
    ( )
{
  if (DBMCli_Session::IsConnected()) {
    DBMCli_Session::Disconnect();
  } // end if
} // end DBMSrv_HSSNode :: ~DBMSrv_HSSNode

/*! @brief   Sets delay time */
DBMSrv_HSSNode & DBMSrv_HSSNode :: SetDelayTime
  ( SAPDB_Int8 nDelayTime )
{
  m_nDelayTime = nDelayTime;
  return *this;
} // end DBMSrv_HSSNode :: SetDelayTime


/*! @brief execute a DBMServer command */
bool DBMSrv_HSSNode::Execute
  ( const Tools_DynamicUTF8String & sCommand,
          Tools_DynamicUTF8String & sAnswer )
{
  bool bRc = true;
  m_oMessageList.ClearMessageList();

  bRc = DBMCli_Session::Execute(sCommand, m_oMessageList);

  if (bRc || (((int) m_oMessageList.ID()) < -100)) {
    sAnswer = (const char *) DBMCli_Session::GetResult();
  } // end if

  return bRc;
} // end DBMSrv_HSSNode::Execute

/*! @brief start node in standby mode */
bool DBMSrv_HSSNode::DBStandby
  (  )
{
  m_oMessageList.ClearMessageList();

  // check state of database
  DBMCli_State & oState = this->GetState();
  if (!oState.Refresh(m_oMessageList)) {
    return false;
  } // end if
  if (oState.Value() != DBMCLI_DBSTATE_OFFLINE && oState.Value() != DBMCLI_DBSTATE_COLD) {
    SAPDBErr_MessageList oMsg(DBMSrv_DBMError(DBRUN), 0);
    m_oMessageList = oMsg;
    return false;
  } // end if

  // start database to cold
  if (oState.Value() == DBMCLI_DBSTATE_OFFLINE) {
    this->Start(m_oMessageList);
  } // end if

  // send RESTART STANDBY to standby
  if (!DBMCli_Session::Execute(CMD_RESTART_STANDBY, m_oMessageList)) {
    return false;
  } // end if

  return true;
} // end DBMSrv_HSSNode::DBStandby

/*! @brief add volume to node */
bool DBMSrv_HSSNode::AddVolume 
  ( const Tools_DynamicUTF8String & sAddVolume )
{
  Tools_DynamicUTF8String sCommand;

  sCommand.Assign(CMD_DB_ADDVOLUME)
          .Append(" ")
          .Append(sAddVolume);

  // send add_volume Command to node
  if (!DBMCli_Session::Execute(sCommand, m_oMessageList)) {
    // send db_stop in errors
    SAPDBErr_MessageList oDummy;
    DBMCli_Database::Stop(oDummy);
    return false;
  } // end if

  return true;
} // end DBMSrv_HSSNode::AddVolume 



/*!
  @brief   Constructor
 */
DBMSrv_HSSNodes :: DBMSrv_HSSNodes
  ( )
{

} // end DBMSrv_HSSNodes :: DBMSrv_HSSNodes

/*!
  @brief   Destructor
 */
DBMSrv_HSSNodes :: ~DBMSrv_HSSNodes
  ( )
{
  this->UnloadNodes(this->m_oNodes);
} // end DBMSrv_HSSNodes :: ~DBMSrv_HSSNodes

/*!
  @brief   public member function
 */
void DBMSrv_HSSNodes :: SetConnectionData
  ( const Tools_DynamicUTF8String & sDatabase,
    const Tools_DynamicUTF8String & sUser,
    const Tools_DynamicUTF8String & sPassword )
{
  m_sDatabase = sDatabase;
  m_sUser     = sUser;
  m_sPassword = sPassword;
} // end DBMSrv_HSSNodes :: SetConnectionData

/*!
  @brief   public member function
 */
bool DBMSrv_HSSNodes :: Create
  (       Tools_DynamicUTF8String & sNode,
          Tools_DynamicUTF8String & sOsUser,
          Tools_DynamicUTF8String & sOsPwd,
          Tools_DynamicUTF8String & sInstallation,
          SAPDB_Int8                nDelayTime,
          SAPDB_Bool                bUser )
{
  DBMCli_Node oNode(sNode, m_oMessageList, false);

  sNode.ToUpper();

  // create database
  if (!oNode.CreateDatabase(sNode, 
                            this->m_sDatabase,
                            this->m_sUser,
                            this->m_sPassword,
                            sOsUser, 
                            sOsPwd, 
                            sInstallation, 
                            m_oMessageList)) {
    SAPDBErr_MessageList oDBM(DBMSrv_DBMError(HSS), 0);
    oDBM.AppendNewMessage(m_oMessageList);
    this->m_oMessageList = oDBM;
    return false;
  } // end if

  int nNode = this->m_oNodes.size();
  ++nNode;
  
  RTEConf_Parameter oConfigFile(m_sDatabase.CharPtr(), 
                                0, 
                                RTECONF_MAXNAMELENGTH,
                                RTECONF_MAXSTRINGLENGTH);
 
  SAPDB_Bool bFileExist = false;
  
  RTEConf_Parameter::StringElement sName[RTECONF_MAXNAMELENGTH + 1];

  oConfigFile.Read(bFileExist, m_oMessageList);
  if (!bFileExist) {
    m_oMessageList.ClearMessageList();
  } // end if
  if (m_oMessageList.IsEmpty()) {
    sprintf((char *) sName, (char *) KERNELPARAM_HS_NODE_"%03d", nNode);
    oConfigFile.SetValue(sName, (RTEConf_Parameter::String) sNode.CharPtr(), m_oMessageList);
    if (m_oMessageList.IsEmpty()) {
      sprintf((char *) sName, (char *) KERNELPARAM_HS_DELAY_TIME_"%03d", nNode);
      oConfigFile.SetValue(sName, nDelayTime, m_oMessageList);
      if (m_oMessageList.IsEmpty()) {
        oConfigFile.Write(m_oMessageList);
      } // end if
    } // end if
  } // end if

  Tools_DynamicUTF8String  sUserPwd;
  sUserPwd.Assign(this->m_sUser).Append(",").Append(this->m_sPassword);
  DBMSrv_HSSNode *         pNode = new DBMSrv_HSSNode(nNode, sNode, m_sDatabase, sUserPwd, this->m_oMessageList );
  if (pNode != NULL) {

    // add os user to service
    if (m_oMessageList.IsEmpty()) {
      if (bUser) {
        Tools_DynamicUTF8String  sRegister;
        sRegister.Assign("db_reg -u ").Append(sOsUser).Append(",").Append(sOsPwd);
        if (!pNode->DBMCli_Session::Execute(sRegister, m_oMessageList)) {
          this->HSSError(m_oMessageList);
        } // end if
      } // end if
    } // end if


    // copy parameters
    if (m_oMessageList.IsEmpty()) {
      this->CopyParams(pNode);
    } // end if

    // start database
    if (m_oMessageList.IsEmpty()) {
      if (!pNode->Start(m_oMessageList)) {
        this->HSSError(m_oMessageList);
      } // endif
    } // endif

    // stop database
    if (m_oMessageList.IsEmpty()) {
      if (!pNode->Stop(m_oMessageList)) {
        this->HSSError(m_oMessageList);
      } // endif
    } // endif

  } // endif

  if (!m_oMessageList.IsEmpty()) {

    SAPDBErr_MessageList oDummy;
    if (pNode != NULL) {
      pNode->Stop(oDummy);
      pNode->DBMCli_Session::Execute("db_drop", oDummy);
      delete pNode;
    } // end if
    oConfigFile.Read(bFileExist, oDummy);
    if (oDummy.IsEmpty()) {
      sprintf((char *) sName, (char *) KERNELPARAM_HS_NODE_"%03d", nNode);
      oConfigFile.Delete(sName, oDummy);
      sprintf((char *) sName, (char *) KERNELPARAM_HS_DELAY_TIME_"%03d", nNode);
      oConfigFile.Delete(sName, oDummy);
      oConfigFile.Write(oDummy);
    } // end if

    return false;
  } // end if

  if (pNode != NULL) {
    this->m_oNodes.push_back(pNode);
    pNode->SetDelayTime(nDelayTime);
  } // end if

  return true;
} // end DBMSrv_HSSNodes :: Create

/*!
  @brief   public member function
 */
bool DBMSrv_HSSNodes :: Remove
  ( const Tools_DynamicUTF8String & sNode )
{
  // get the node
  DBMSrv_HSSNode * pNode = GetNode(sNode);
  if (pNode == NULL) {
    SAPDBErr_MessageList oDBM(DBMSRV_ERR_HSSNODEUNKNOWN);
    this->m_oMessageList = oDBM;
    return false;
  } // end if

  // check state of database
  DBMCli_State & oState = pNode->GetState();
  if (!oState.Refresh(m_oMessageList)) {

    // database does not exist
    m_oMessageList.ClearMessageList();

  } else {

    if (oState.Value() == DBMCLI_DBSTATE_WARM) {
      SAPDBErr_MessageList oMsg(DBMSrv_DBMError(DBRUN), 0);
      return this->HSSError(oMsg);
    } // end if

    // stop database
    if (!pNode->Stop(m_oMessageList)) {
      return this->HSSError(m_oMessageList);
    } // endif

    // drop database 
    if (!pNode->DropDatabase(this->m_sDatabase, this->m_sUser, this->m_sPassword, m_oMessageList)) {
      return this->HSSError(m_oMessageList);
    } // end if

  } // end if

  // remove database from parameters
  RTEConf_Parameter oConfigFile(m_sDatabase.CharPtr(), 
                                0, 
                                RTECONF_MAXNAMELENGTH,
                                RTECONF_MAXSTRINGLENGTH);
 
  SAPDB_Bool bFileExist = false;
  
  RTEConf_Parameter::StringElement sName[RTECONF_MAXNAMELENGTH + 1];

  oConfigFile.Read(bFileExist, m_oMessageList);
  if (!bFileExist) {
    m_oMessageList.ClearMessageList();
  } // end if
  if (m_oMessageList.IsEmpty()) {
    sprintf((char *) sName, (char *) KERNELPARAM_HS_NODE_"%03d", pNode->GetNumber());
    oConfigFile.Delete(sName, m_oMessageList);
    if (m_oMessageList.IsEmpty()) {
      SAPDBErr_MessageList oDummy;
      sprintf((char *) sName, (char *) KERNELPARAM_HS_DELAY_TIME_"%03d", pNode->GetNumber());
      oConfigFile.Delete(sName, oDummy);

      oConfigFile.Write(m_oMessageList);
    } // end if
  } // end if
  if (!m_oMessageList.IsEmpty()) {
    return false;
  } // end if

  // remove node from list
  DelNode(sNode, true, this->m_oNodes);

  return true;
} // end DBMSrv_HSSNodes :: Remove

/*!
  @brief   public member function
 */
bool DBMSrv_HSSNodes :: Standby
  ( const Tools_DynamicUTF8String & sNode )
{
  m_oMessageList.ClearMessageList();

  // get the node
  DBMSrv_HSSNode * pNode = GetNode(sNode);
  if (pNode == NULL) {
    SAPDBErr_MessageList oDBM(DBMSRV_ERR_HSSNODEUNKNOWN);
    this->m_oMessageList = oDBM;
    return false;
  } // end if

  // check state of database
  DBMCli_State & oState = pNode->GetState();
  if (!oState.Refresh(m_oMessageList)) {
    return false;
  } // end if
  if (oState.Value() != DBMCLI_DBSTATE_OFFLINE && oState.Value() != DBMCLI_DBSTATE_COLD) {
    SAPDBErr_MessageList oMsg(DBMSrv_DBMError(DBRUN), 0);
    return this->HSSError(oMsg);
  } // end if

  // stop database
  if (!pNode->Stop(m_oMessageList)) {
    return this->HSSError(m_oMessageList);
  } // endif

  // copy parameters
  if (!this->CopyParams(pNode)) {
    return false;
  } // end if

  // start database
  if (!pNode->Start(m_oMessageList)) {
    return this->HSSError(m_oMessageList);
  } // endif

  Tools_DynamicUTF8String sAnswer;

  // restart standby
  if (pNode->Execute(CMD_DB_RESTARTSTANDBY, sAnswer)) {
    return true;
  } // end if

  // stop database
  if (!pNode->Stop(m_oMessageList)) {
    return this->HSSError(m_oMessageList);
  } // endif

  // start database
  if (!pNode->Start(m_oMessageList)) {
    return this->HSSError(m_oMessageList);
  } // endif

  // init standby
  if (!pNode->Execute(CMD_DB_INITSTANDBY, sAnswer)) {
    return this->HSSError(pNode->LastMessage());
  } // end if

  // restart standby
  if (!pNode->Execute(CMD_DB_RESTARTSTANDBY, sAnswer)) {
    return this->HSSError(pNode->LastMessage());
  } // end if

  return true;
} // end DBMSrv_HSSNodes :: Standby

/*!
  @brief   public member function
 */
bool DBMSrv_HSSNodes :: Execute
  ( const Tools_DynamicUTF8String & sNode,
    const Tools_DynamicUTF8String & sCommand )
{
  Tools_DynamicUTF8String sAnswer;

  return this->Execute(sNode, sCommand, sAnswer);
} // end DBMSrv_HSSNodes :: Execute

/*!
  @brief   public member function
 */
bool DBMSrv_HSSNodes :: Execute
  (       SAPDB_Int                 nNode,
    const Tools_DynamicUTF8String & sCommand )
{
  Tools_DynamicUTF8String sAnswer;

  return this->Execute(nNode, sCommand, sAnswer);
} // end DBMSrv_HSSNodes :: Execute

/*!
  @brief   public member function
 */
bool DBMSrv_HSSNodes :: Execute
  ( const Tools_DynamicUTF8String & sNode,
    const Tools_DynamicUTF8String & sCommand,
          Tools_DynamicUTF8String & sAnswer )
{
  if (this->m_oNodes.empty()) {
    if (!this->LoadNodes()) {
      SAPDBErr_MessageList oDBM(DBMSrv_DBMError(HSS), 0);
      oDBM.AppendNewMessage(m_oMessageList);
      this->m_oMessageList = oDBM;
      return false;
    } // end if
  } // end if

  DBMSrv_HSSNode * pNode = this->FindNode(sNode, this->m_oNodes);

  return this->Execute(pNode, sCommand, sAnswer);
} // end DBMSrv_HSSNodes :: Execute

/*!
  @brief   public member function
 */
bool DBMSrv_HSSNodes :: Execute
  ( SAPDB_Int                       nNode,
    const Tools_DynamicUTF8String & sCommand,
          Tools_DynamicUTF8String & sAnswer )
{
  if (this->m_oNodes.empty()) {
    if (!this->LoadNodes()) {
      SAPDBErr_MessageList oDBM(DBMSrv_DBMError(HSS), 0);
      oDBM.AppendNewMessage(m_oMessageList);
      this->m_oMessageList = oDBM;
      return false;
    } // end if
  } // end if

  DBMSrv_HSSNode * pNode = this->FindNode(nNode, this->m_oNodes);

  return this->Execute(pNode, sCommand, sAnswer);
} // end DBMSrv_HSSNodes :: Execute

/*!
  @brief   public member function
 */
bool DBMSrv_HSSNodes :: Execute
  (       DBMSrv_HSSNode          * pNode,
    const Tools_DynamicUTF8String & sCommand,
          Tools_DynamicUTF8String & sAnswer )
{
  if (pNode == NULL) {
    SAPDBErr_MessageList oDBM(DBMSrv_DBMError(HSS), 0);
    SAPDBErr_MessageList oMsg(DBMSRV_ERR_HSSNODEUNKNOWN);
    oDBM.AppendNewMessage(oMsg);
    this->m_oMessageList = oDBM;
    return false;
  } // end if

  if (!pNode->Execute(sCommand, sAnswer)) {
    SAPDBErr_MessageList oDBM(DBMSrv_DBMError(HSS), 0);
    oDBM.AppendNewMessage(pNode->LastMessage());
    this->m_oMessageList = oDBM;
    return false;
  } // end if

  return true;
} // end DBMSrv_HSSNodes :: Execute

/*!
  @brief   private member function
 */
bool DBMSrv_HSSNodes :: LoadNodes
  ( )
{
  m_oMessageList.ClearMessageList();

  RTEConf_Parameter::StringElement sValue[RTECONF_MAXSTRINGLENGTH + 1];
  RTEConf_Parameter::Integer       nValue;
  RTEConf_Parameter::Name          sName;
  RTEConf_Parameter::StringElement sDelayName[RTECONF_MAXNAMELENGTH + 1];
  RTEConf_Parameter::Type          oType;

  RTEConf_Parameter oConfigFile(m_sDatabase.CharPtr(), 
                                0, 
                                RTECONF_MAXNAMELENGTH,
                                RTECONF_MAXSTRINGLENGTH);
 
  SAPDB_Bool                  bFileExist = false;
  SAPDB_Bool                  bEnd       = false;
  SAPDB_Int                   nNumber    = 0;
  DBMSrv_HSSNode            * pNode      = NULL;
  Tools_DynamicUTF8String     sUserPwd   = "";
  Tools_DynamicUTF8String     sNameToInt;

  NodeList                    oOldNodes;

  NodeList::iterator oIterator = this->m_oNodes.begin();
  while (oIterator != this->m_oNodes.end()) {
    oOldNodes.push_back(*oIterator);
    ++oIterator;
  } // end while
  this->m_oNodes.clear();

  oConfigFile.Read(bFileExist, m_oMessageList);
  if (!bFileExist) {
    m_oMessageList.ClearMessageList();
  } // end if
  if (bFileExist && m_oMessageList.IsEmpty()) {
    if (oConfigFile.GetValue ((RTEConf_Parameter::Name) KERNELPARAM_HS_STORAGE_DLL, sValue, m_oMessageList)) {
      this->m_sStorageDLL = sValue;
      if (oConfigFile.GetValue ((RTEConf_Parameter::Name) KERNELPARAM_OFFICIAL_NODE, sValue, m_oMessageList)) {
        this->m_sOfficialNode = sValue;
        RTEConf_ParameterIterator   oIterator  = oConfigFile.GetIterator();
        do {
          if (oIterator.Next(sName, oType, bEnd, m_oMessageList)) {
            if (STRNCMP_UTF8(sName, KERNELPARAM_HS_NODE_, STRLEN_UTF8(KERNELPARAM_HS_NODE_)) == 0) {
              if (oConfigFile.GetValue (sName, sValue, m_oMessageList)) {
                sNameToInt = sName;
                nNumber = atoi(sNameToInt.SubStr((Tools_DynamicUTF8String::BasisElementIndex) STRLEN_UTF8(KERNELPARAM_HS_NODE_)).CharPtr());
                // search node
                pNode = FindNode(nNumber, oOldNodes);
                
                // create node
                if (pNode == NULL) {
                  sUserPwd.Assign(this->m_sUser).Append(",").Append(this->m_sPassword);
                  pNode = new DBMSrv_HSSNode(nNumber, sValue, m_sDatabase, sUserPwd, this->m_oMessageList );
                } else {
                  DelNode(sValue, false, oOldNodes);
                } // end if
                if (pNode != NULL) {
                  this->m_oNodes.push_back(pNode);
                  // read and set delay time
                  sprintf((char *) sDelayName, (char *) KERNELPARAM_HS_DELAY_TIME_"%03d", nNumber);
                  if (oConfigFile.GetValue (sDelayName, nValue, m_oMessageList)) {
                    pNode->SetDelayTime(nValue);
                  } else {
                    pNode->SetDelayTime(0);
                    m_oMessageList.ClearMessageList();
                  } // end if
                } // end if
              } // end if
            } // end if
          } // end if
        }  while ( (m_oMessageList.IsEmpty()) && !bEnd);
      } // end if
    } // end if
  } // end if

  if (!m_oMessageList.IsEmpty()) {
    this->UnloadNodes(this->m_oNodes);
  } // end if

  UnloadNodes(oOldNodes);

  return m_oMessageList.IsEmpty();
} // end DBMSrv_HSSNodes :: LoadNodes

/*!
  @brief   private member function
 */
void DBMSrv_HSSNodes :: UnloadNodes
  ( NodeList                & oNodes)
{
  NodeList::iterator  oIterator = oNodes.begin();
  DBMSrv_HSSNode    * pNode     = NULL;

  while (oIterator != oNodes.end()) {
    pNode = (*oIterator);
    ++oIterator;
    if (pNode != NULL) {
      delete pNode;
    } // end if
  } // end while

  oNodes.clear();
} // end DBMSrv_HSSNodes :: UnloadNodes

/*! @brief   public member function */
DBMSrv_HSSNode * DBMSrv_HSSNodes :: GetNode
  ( const Tools_DynamicUTF8String & sNode )
{
  if (this->m_oNodes.empty()) {
    if (!this->LoadNodes()) {
      SAPDBErr_MessageList oDBM(DBMSrv_DBMError(HSS), 0);
      oDBM.AppendNewMessage(m_oMessageList);
      this->m_oMessageList = oDBM;
      return NULL;
    } // end if
  } // end if

  return this->FindNode(sNode, this->m_oNodes);
} // end DBMSrv_HSSNodes :: GetNode

/*! @brief   public member function */
DBMSrv_HSSNode * DBMSrv_HSSNodes :: GetNode
  ( SAPDB_Int                       nNode )
{
  if (this->m_oNodes.empty()) {
    if (!this->LoadNodes()) {
      SAPDBErr_MessageList oDBM(DBMSrv_DBMError(HSS), 0);
      oDBM.AppendNewMessage(m_oMessageList);
      this->m_oMessageList = oDBM;
      return NULL;
    } // end if
  } // end if

  return this->FindNode(nNode, this->m_oNodes);
} // end DBMSrv_HSSNodes :: GetNode

/*!
  @brief   private member function
 */
DBMSrv_HSSNode * DBMSrv_HSSNodes :: FindNode
  ( SAPDB_Int                 nNumber,
    const NodeList          & oNodes )
{
  NodeList::const_iterator oIterator = oNodes.begin();

  while (oIterator != oNodes.end()) {
    if ((*oIterator)->GetNumber() == nNumber) {
      return *oIterator;
    } // end if
    ++oIterator;
  } // end while

  return NULL;
} // end DBMSrv_HSSNodes :: FindNode

/*!
  @brief   private member function
 */
DBMSrv_HSSNode * DBMSrv_HSSNodes :: FindNode
  ( Tools_DynamicUTF8String   sName,
    const NodeList          & oNodes )
{
  NodeList::const_iterator oIterator = oNodes.begin();

  while (oIterator != oNodes.end()) {
    if (sName.ToUpper().Compare((*oIterator)->GetName()) == 0) {
      return *oIterator;
    } // end if
    ++oIterator;
  } // end while


  // maybe a number
  unsigned int nIndex = 0;
  for (nIndex = 0; nIndex < sName.Length(); ++nIndex) {
    if (!isdigit(sName[nIndex])) {
      return NULL;
    } // end if
  } // end for

  return FindNode(atoi(sName.CharPtr()), oNodes);
} // end DBMSrv_HSSNodes :: FindNode

/*!
  @brief   private member function
 */
void DBMSrv_HSSNodes :: DelNode
  ( Tools_DynamicUTF8String   sName,
    bool                      bDelete,
    NodeList                & oNodes )
{
  NodeList::iterator oIterator = oNodes.begin();

  while (oIterator != oNodes.end()) {
    if (sName.ToUpper().Compare((*oIterator)->GetName()) == 0) {
      (*oIterator)->Disconnect();
      if (bDelete) {
        delete (*oIterator);
      } // end if
      oNodes.erase(oIterator);
      return;
    } // end if
    ++oIterator;
  } // end while

    // maybe a number
  unsigned int nIndex = 0;
  for (nIndex = 0; nIndex < sName.Length(); ++nIndex) {
    if (!isdigit(sName[nIndex])) {
      return;
    } // end if
  } // end for

  SAPDB_Int nNumber = atoi(sName.CharPtr());
  oIterator = oNodes.begin();

  while (oIterator != oNodes.end()) {
    if ((*oIterator)->GetNumber() == nNumber) {
      (*oIterator)->Disconnect();
      if (bDelete) {
        delete (*oIterator);
      } // end if
      oNodes.erase(oIterator);
      return;
    } // end if
    ++oIterator;
  } // end while

} // end DBMSrv_HSSNodes :: DelNode

/*! @brief   private member function */
bool DBMSrv_HSSNodes :: HSSError
    (const SAPDBErr_MessageList & oMsg)
{
  SAPDBErr_MessageList oDBM(DBMSrv_DBMError(HSS), 0);
  oDBM.AppendNewMessage(oMsg);
  this->m_oMessageList = oDBM;
  return false;
} // end DBMSrv_HSSNodes :: HSSError

/*! @brief public member function */
Tools_DynamicUTF8String DBMSrv_HSSNodes :: ListHSSInfo
    ( )
{
  Tools_DynamicUTF8String          sInfo;
  Tools_DynamicUTF8String          sDelay;
  RTEConf_Parameter::StringElement sNodeName[RTECONF_MAXNAMELENGTH + 1];
  SAPDB_Int                        nNumber = 0;

  LoadNodes();
  
  sInfo.Assign(KERNELPARAM_HS_STORAGE_DLL)
       .Append("\t")
       .Append(this->m_sStorageDLL)
       .Append("\n");

  sInfo.Append(KERNELPARAM_OFFICIAL_NODE)
       .Append("\t")
       .Append(this->m_sOfficialNode)
       .Append("\n");

  sInfo.Append("CURRENT_NODE")
       .Append("\t")
       .Append(RTE_ISystem::Instance().GetLocalNodeName())
       .Append("\n");

  NodeList::iterator oIterator = this->m_oNodes.begin();

  while (oIterator != this->m_oNodes.end()) {
    ++nNumber;
    sprintf((char *) sNodeName, (char *) KERNELPARAM_HS_NODE_"%03d", nNumber);
    sDelay.ConvertFromInt((*oIterator)->GetDelayTime());
    sInfo.Append(sNodeName)
         .Append("\t")
         .Append((*oIterator)->GetName())
         .Append((((40 - (int) strlen((*oIterator)->GetName()))) > 0) ? 41 - (int)strlen((*oIterator)->GetName()) : 1, ' ')
         .Append("\t")
         .Append(sDelay)
         .Append("\n");
    ++oIterator;
  } // end while

  return sInfo;
} // end DBMSrv_HSSNodes :: ListHSSInfo

/*!
  @brief public member function
 */
bool DBMSrv_HSSNodes :: CopyParams
  ( DBMSrv_HSSNode * pNode )
{
  SAPDBErr_MessageList  oTmpMsg;

  // load and copy parameters
  m_oMessageList.ClearMessageList();

  RTEConf_Parameter::Integer       iValue;
  RTEConf_Parameter::Real          rValue;
  RTEConf_Parameter::StringElement sValue[RTECONF_MAXSTRINGLENGTH + 1];
  RTEConf_Parameter::CryptInfo     cValue;
  RTEConf_Parameter::Name          sParamName;
  RTEConf_Parameter::Type          oType;

  RTEConf_Parameter oConfigFile(m_sDatabase.CharPtr(), 
                                0, 
                                RTECONF_MAXNAMELENGTH,
                                RTECONF_MAXSTRINGLENGTH);
 
  SAPDB_Bool                  bFileExist = false;
  SAPDB_Bool                  bEnd       = false;

  Tools_DynamicUTF8String     sCommand;
  Tools_DynamicUTF8String     sAnswer;

  oConfigFile.Read(bFileExist, m_oMessageList);
  if (!bFileExist) {
    m_oMessageList.ClearMessageList();
  } // end if
  if (bFileExist && m_oMessageList.IsEmpty()) {

    if (!pNode->Execute(CMD_PARAM_REMOVE, sAnswer)) {
      return this->HSSError(pNode->LastMessage());
    } // end if

    if (!pNode->Execute(CMD_PARAM_START, sAnswer)) {
      m_oMessageList = pNode->LastMessage();
      pNode->Execute(CMD_PARAM_RESTORE, sAnswer);
      return this->HSSError(m_oMessageList);
    } // end if

    RTEConf_ParameterIterator   oIterator  = oConfigFile.GetIterator();
    do {
      if (oIterator.Next(sParamName, oType, bEnd, m_oMessageList)) {

        // read and convert parameters
        switch (oType) {
          case RTEConf_Parameter::t_Integer:
            if (oConfigFile.GetValue (sParamName, iValue, m_oMessageList)) {
              sprintf ( (char *) &sValue[0], "%ld" , (long) iValue );
            } // end if
            break;
          case RTEConf_Parameter::t_Real:
            if(oConfigFile.GetValue (sParamName, rValue, m_oMessageList)) {
              sprintf ( (char *) &sValue[0], "%lf" , (double) rValue );
            } // end if
            break;
          case RTEConf_Parameter::t_String:
            oConfigFile.GetValue (sParamName, sValue, m_oMessageList);
            break;
          case RTEConf_Parameter::t_CryptInfo:
            if (oConfigFile.GetValue (sParamName, cValue, m_oMessageList)) {
              tsp00_Name  pwClear;
              tsp00_Namec pwClearc;
              tsp00_CryptPw    pwTemp;
              memcpy(&(pwTemp[0]), &(cValue[0]), sizeof(pwTemp));
              s02decrypt(pwClear, pwTemp);
              pwClearc.p2c(pwClear);
              strcpy((char *) sValue, pwClearc);
            } // end if
            break;
        } // end switch

        // write parameter
        if (m_oMessageList.IsEmpty()) {
          sCommand.Assign(CMD_PARAM_DIRECTPUT)
                  .Append(" ")
                  .Append(sParamName)
                  .Append(" ")
                  .Append(sValue);
          pNode->Execute(sCommand, sAnswer);
//          if (!pNode->Execute(sCommand, sAnswer)) {
//            m_oMessageList = pNode->LastMessage();
//          } // end if
        } // end if

      } // end if
    }  while ( (m_oMessageList.IsEmpty()) && !bEnd);

    // abort session
    pNode->Execute(CMD_PARAM_ABORT, sAnswer);

    if (!pNode->Execute(CMD_PARAM_CHECKALL, sAnswer)) {
      m_oMessageList = pNode->LastMessage();
    } // end if

    // check for error
    if (!m_oMessageList.IsEmpty()) {
      pNode->Execute(CMD_PARAM_RESTORE, sAnswer);
      return this->HSSError(m_oMessageList);
    } // end if

  } // end if

  return true;
} // end DBMSrv_HSSNodes :: CopyParams

/*!
  @brief public member function
 */
bool DBMSrv_HSSNodes :: AddVolumeToCluster
    ( const Tools_DynamicUTF8String & sAddVolume,
            Tools_DynamicUTF8String & sAnswer )
{
  bool                         bReturn      = true;    
  NodeList::iterator           oIterator    = m_oNodes.begin();
  Tools_DynamicUTF8String      sCurrentNode = RTE_ISystem::Instance().GetLocalNodeName();
  const SAPDBErr_MessageList * pMsg         = NULL;
  Tools_DynamicUTF8String      sNumber;

  sAnswer.Assign("");

  while (oIterator != m_oNodes.end()) {
    if (sCurrentNode.ToUpper().Compare((*oIterator)->GetName()) != 0) {
      sAnswer.Append("Standby: ")
             .Append((*oIterator)->GetName())
             .Append("\n");
      if (!(*oIterator)->AddVolume(sAddVolume)) {
        pMsg = &(*oIterator)->LastMessage ();
        while (pMsg != NULL) {
          sNumber.ConvertFromInt((int) pMsg->ID());
          sAnswer.Append("    ")
                 .Append(sNumber)
                 .Append(",")
                 .Append(pMsg->Message())
                 .Append("\n");
          pMsg = pMsg->NextMessage();
          bReturn = false;
        } // end while
      } else {
        sAnswer.Append("    OK\n");
      } // end if
    } // end if
    ++oIterator;
  } // end while

  return bReturn;
} // end DBMSrv_HSSNodes :: AddVolumeToCluster
